<?php
/**
 * Abstract module class.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Abstract module class.
 *
 * All detection modules must extend this class.
 *
 * @since 1.0.0
 */
abstract class TrustLens_Module {

	/**
	 * Module ID.
	 *
	 * @var string
	 */
	protected string $id = '';

	/**
	 * Module name.
	 *
	 * @var string
	 */
	protected string $name = '';

	/**
	 * Module description.
	 *
	 * @var string
	 */
	protected string $description = '';

	/**
	 * Is this a Pro feature?
	 *
	 * @var bool
	 */
	protected bool $is_pro = false;

	/**
	 * Get module ID.
	 *
	 * @since 1.0.0
	 * @return string Module ID.
	 */
	public function get_id(): string {
		return $this->id;
	}

	/**
	 * Get module name.
	 *
	 * @since 1.0.0
	 * @return string Module name.
	 */
	public function get_name(): string {
		return $this->name;
	}

	/**
	 * Get module description.
	 *
	 * @since 1.0.0
	 * @return string Module description.
	 */
	public function get_description(): string {
		return $this->description;
	}

	/**
	 * Check if this is a Pro feature.
	 *
	 * @since 1.0.0
	 * @return bool True if Pro feature.
	 */
	public function is_pro(): bool {
		return $this->is_pro;
	}

	/**
	 * Check if module is enabled.
	 *
	 * @since 1.0.0
	 * @return bool True if enabled.
	 */
	public function is_enabled(): bool {
		// Pro modules require license.
		if ( $this->is_pro && ! $this->is_pro_active() ) {
			return false;
		}

		return (bool) get_option( "trustlens_module_{$this->id}_enabled", true );
	}

	/**
	 * Check if Pro is active.
	 *
	 * @since 1.0.0
	 * @return bool True if Pro is active.
	 */
	protected function is_pro_active(): bool {
		// Check Freemius license - uses __premium_only to strip from free version.
		if ( function_exists( 'wstl_fs' ) && wstl_fs()->can_use_premium_code__premium_only() ) {
			return true;
		}

		return (bool) apply_filters( 'trustlens/is_pro_active', false );
	}

	/**
	 * Register WooCommerce hooks.
	 *
	 * @since 1.0.0
	 */
	abstract public function register_hooks(): void;

	/**
	 * Calculate signal score for customer.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @return array {
	 *     Signal data.
	 *
	 *     @type int    $score  Signal score contribution (-50 to +50).
	 *     @type string $reason Human-readable reason.
	 * }
	 */
	abstract public function get_signal( string $email_hash ): array;

	/**
	 * Get module settings fields.
	 *
	 * @since 1.0.0
	 * @return array Settings fields for admin.
	 */
	public function get_settings_fields(): array {
		return array();
	}

	/**
	 * Log a module-specific event.
	 *
	 * @since 1.0.0
	 * @param string   $email_hash Customer email hash.
	 * @param string   $event_type Event type.
	 * @param array    $data       Event data.
	 * @param int|null $order_id   Optional order ID.
	 */
	protected function log_event( string $email_hash, string $event_type, array $data = array(), ?int $order_id = null ): void {
		$data['module'] = $this->id;
		wstl_log_event( $email_hash, $event_type, $data, $order_id );
	}

	/**
	 * Queue score update for customer.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @param int    $delay      Delay in seconds.
	 */
	protected function queue_score_update( string $email_hash, int $delay = 60 ): void {
		wstl_queue_score_update( $email_hash, $delay );
	}
}
