<?php
/**
 * Installation and database setup.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * TrustLens installation class.
 *
 * @since 1.0.0
 */
class TrustLens_Install {

	/**
	 * Check whether this site can execute Pro functionality.
	 *
	 * Kept local to install flow so activation can enforce safe defaults
	 * without depending on includes/functions.php load order.
	 *
	 * @since 1.2.0
	 * @return bool True when a paid Freemius plan is active.
	 */
	private static function can_use_pro_features(): bool {
		if ( ! function_exists( 'wstl_fs' ) ) {
			return false;
		}

		$fs = wstl_fs();
		if ( ! is_object( $fs ) || ! method_exists( $fs, 'is_not_paying' ) ) {
			return false;
		}

		return ! $fs->is_not_paying();
	}

	/**
	 * Run on plugin activation.
	 *
	 * @since 1.0.0
	 */
	public static function activate(): void {
		self::create_tables();
		self::create_options();
		self::schedule_events();

		// Schedule welcome summary email (24 hours after activation).
		if ( ! get_option( 'trustlens_welcome_summary_sent' ) ) {
			wp_schedule_single_event( time() + DAY_IN_SECONDS, 'trustlens/welcome_summary' );
		}

		// Set activation flag for redirect.
		set_transient( 'trustlens_activated', true, 30 );

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Run on plugin deactivation.
	 *
	 * @since 1.0.0
	 */
	public static function deactivate(): void {
		self::unschedule_events();

		// Unschedule scheduled reports.
		if ( class_exists( 'TrustLens_Scheduled_Reports' ) ) {
			TrustLens_Scheduled_Reports::unschedule_reports();
		}

		flush_rewrite_rules();
	}

	/**
	 * Create database tables.
	 *
	 * @since 1.0.0
	 */
	public static function create_tables(): void {
		global $wpdb;

		$installed_version = get_option( 'trustlens_db_version' );

		if ( $installed_version === TRUSTLENS_DB_VERSION ) {
			return;
		}

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$charset_collate = $wpdb->get_charset_collate();

		// Customers table.
		$table_customers = $wpdb->prefix . 'trustlens_customers';
		$sql_customers = "CREATE TABLE {$table_customers} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			email_hash varchar(32) NOT NULL,
			customer_id bigint(20) unsigned DEFAULT NULL,
			customer_email varchar(255) DEFAULT NULL,
			customer_type enum('user','guest') DEFAULT 'guest',
			trust_score tinyint(3) unsigned DEFAULT 50,
			segment enum('vip','trusted','normal','caution','risk','critical') DEFAULT 'normal',
			total_orders int(10) unsigned DEFAULT 0,
			total_order_value decimal(12,2) DEFAULT 0.00,
			total_refunds int(10) unsigned DEFAULT 0,
			total_refund_value decimal(12,2) DEFAULT 0.00,
			full_refunds int(10) unsigned DEFAULT 0,
			partial_refunds int(10) unsigned DEFAULT 0,
			return_rate decimal(5,2) DEFAULT 0.00,
			total_coupons_used int(10) unsigned DEFAULT 0,
			first_order_coupons int(10) unsigned DEFAULT 0,
			coupon_then_refund int(10) unsigned DEFAULT 0,
			total_disputes int(10) unsigned DEFAULT 0,
			disputes_won int(10) unsigned DEFAULT 0,
			disputes_lost int(10) unsigned DEFAULT 0,
			linked_accounts int(10) unsigned DEFAULT 0,
			cancelled_orders int(10) unsigned DEFAULT 0,
			order_edits int(10) unsigned DEFAULT 0,
			reviews_before_refund int(10) unsigned DEFAULT 0,
			shipping_address_hash varchar(32) DEFAULT NULL,
			last_ip varchar(45) DEFAULT NULL,
			is_blocked tinyint(1) DEFAULT 0,
			is_allowlisted tinyint(1) DEFAULT 0,
			admin_notes text DEFAULT NULL,
			first_order_date datetime DEFAULT NULL,
			last_order_date datetime DEFAULT NULL,
			last_refund_date datetime DEFAULT NULL,
			score_updated_at datetime DEFAULT NULL,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			UNIQUE KEY idx_email_hash (email_hash),
			KEY idx_customer_id (customer_id),
			KEY idx_segment (segment),
			KEY idx_trust_score (trust_score),
			KEY idx_return_rate (return_rate),
			KEY idx_shipping_hash (shipping_address_hash),
			KEY idx_is_blocked (is_blocked)
		) $charset_collate;";

		dbDelta( $sql_customers );

		// Events table.
		$table_events = $wpdb->prefix . 'trustlens_events';
		$sql_events = "CREATE TABLE {$table_events} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			email_hash varchar(32) NOT NULL,
			event_type varchar(50) NOT NULL,
			event_data longtext DEFAULT NULL,
			order_id bigint(20) unsigned DEFAULT NULL,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY idx_email_hash (email_hash),
			KEY idx_event_type (event_type),
			KEY idx_order_id (order_id),
			KEY idx_created_at (created_at)
		) $charset_collate;";

		dbDelta( $sql_events );

		// Signals table.
		$table_signals = $wpdb->prefix . 'trustlens_signals';
		$sql_signals = "CREATE TABLE {$table_signals} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			email_hash varchar(32) NOT NULL,
			module_id varchar(50) NOT NULL,
			signal_score smallint(6) DEFAULT 0,
			signal_reason varchar(255) DEFAULT NULL,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY idx_email_hash (email_hash),
			KEY idx_module_id (module_id)
		) $charset_collate;";

		dbDelta( $sql_signals );

		// Category stats table.
		$table_category_stats = $wpdb->prefix . 'trustlens_category_stats';
		$sql_category_stats = "CREATE TABLE {$table_category_stats} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			email_hash varchar(32) NOT NULL,
			category_slug varchar(200) NOT NULL,
			order_count int(10) unsigned DEFAULT 0,
			order_value decimal(12,2) DEFAULT 0.00,
			refund_count int(10) unsigned DEFAULT 0,
			refund_value decimal(12,2) DEFAULT 0.00,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			UNIQUE KEY idx_email_category (email_hash, category_slug),
			KEY idx_email_hash (email_hash),
			KEY idx_category_slug (category_slug)
		) $charset_collate;";

		dbDelta( $sql_category_stats );

		// Fingerprints table for linked accounts detection.
		$table_fingerprints = $wpdb->prefix . 'trustlens_fingerprints';
		$sql_fingerprints = "CREATE TABLE {$table_fingerprints} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			email_hash varchar(32) NOT NULL,
			fingerprint_type varchar(50) NOT NULL,
			fingerprint_hash varchar(32) NOT NULL,
			first_seen datetime DEFAULT CURRENT_TIMESTAMP,
			last_seen datetime DEFAULT CURRENT_TIMESTAMP,
			times_seen int(10) unsigned DEFAULT 1,
			PRIMARY KEY (id),
			UNIQUE KEY idx_unique_fingerprint (email_hash, fingerprint_type, fingerprint_hash),
			KEY idx_email_hash (email_hash),
			KEY idx_fingerprint_type (fingerprint_type),
			KEY idx_fingerprint_hash (fingerprint_hash)
		) $charset_collate;";

		dbDelta( $sql_fingerprints );

		// Webhook logs table.
		$table_webhook_logs = $wpdb->prefix . 'trustlens_webhook_logs';
		$sql_webhook_logs = "CREATE TABLE {$table_webhook_logs} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			endpoint_id varchar(36) DEFAULT NULL,
			endpoint_url varchar(500) NOT NULL,
			event varchar(50) NOT NULL,
			payload longtext DEFAULT NULL,
			response_code smallint(3) unsigned DEFAULT NULL,
			response_body text DEFAULT NULL,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY idx_endpoint_id (endpoint_id),
			KEY idx_event (event),
			KEY idx_created_at (created_at)
		) $charset_collate;";

		dbDelta( $sql_webhook_logs );

		// Automation logs table.
		$table_automation_logs = $wpdb->prefix . 'trustlens_automation_logs';
		$sql_automation_logs = "CREATE TABLE {$table_automation_logs} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			rule_id varchar(36) NOT NULL,
			rule_name varchar(255) DEFAULT NULL,
			email_hash varchar(32) NOT NULL,
			trigger_type varchar(50) NOT NULL,
			action_type varchar(50) NOT NULL,
			order_id bigint(20) unsigned DEFAULT NULL,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY idx_rule_id (rule_id),
			KEY idx_email_hash (email_hash),
			KEY idx_created_at (created_at)
		) $charset_collate;";

		dbDelta( $sql_automation_logs );

		// Add new columns to customers table if they don't exist.
		self::add_column_if_missing( $table_customers, 'linked_count', 'int(10) unsigned DEFAULT 0 AFTER linked_accounts' );
		self::add_column_if_missing( $table_customers, 'tags', 'text DEFAULT NULL AFTER admin_notes' );

		// Update version.
		update_option( 'trustlens_db_version', TRUSTLENS_DB_VERSION );
	}

	/**
	 * Add column to table if it doesn't exist.
	 *
	 * @since 1.1.0
	 * @param string $table  Table name.
	 * @param string $column Column name.
	 * @param string $definition Column definition.
	 */
	private static function add_column_if_missing( string $table, string $column, string $definition ): void {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- SHOW COLUMNS for schema check, table name from trusted internal array.
		$column_exists = $wpdb->get_results( $wpdb->prepare(
			"SHOW COLUMNS FROM {$table} LIKE %s",
			$column
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		if ( empty( $column_exists ) ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- ALTER TABLE for schema migration, variables are from trusted internal array.
			$wpdb->query( "ALTER TABLE {$table} ADD COLUMN {$column} {$definition}" );
		}
	}

	/**
	 * Create default options.
	 *
	 * @since 1.0.0
	 */
	public static function create_options(): void {
		$can_use_pro = self::can_use_pro_features();

		// Only set defaults if not already set.
		$defaults = array(
			'trustlens_installed_timestamp' => time(),
			'trustlens_min_orders'        => 3,
			'trustlens_enable_blocking'   => false,
			'trustlens_enable_dashboard'  => true,
			'trustlens_enable_order_warning' => true,
			'trustlens_hold_segments'     => array( 'risk', 'critical' ),
			'trustlens_block_message'     => __( 'We are unable to process your order at this time. Please contact support for assistance.', 'trustlens' ),
			'trustlens_segment_thresholds' => array(
				'vip'      => 90,
				'trusted'  => 70,
				'normal'   => 50,
				'caution'  => 30,
				'risk'     => 10,
				'critical' => 0,
			),
			// Module settings.
			'trustlens_module_returns_enabled'    => true,
			'trustlens_module_orders_enabled'     => true,
			'trustlens_module_coupons_enabled'    => true,
			'trustlens_module_categories_enabled' => true,
			'trustlens_module_chargebacks_enabled' => true,
			'trustlens_returns_high_threshold'    => 40,
			'trustlens_returns_critical_threshold' => 60,
			// Chargebacks settings (Pro).
			'trustlens_chargebacks_auto_block'    => 2,
			// Advanced coupon settings (Pro).
			'trustlens_coupons_block_linked_abuse' => false,
			'trustlens_coupons_max_first_order'   => 2,
			// Notification settings.
			'trustlens_enable_notifications'      => false,
			'trustlens_notification_email'        => get_option( 'admin_email' ),
			// Free notifications.
			'trustlens_notify_blocked_checkout'   => true,
			'trustlens_notify_welcome_summary'    => true,
			'trustlens_notify_weekly_summary'     => true,
			// Pro notifications.
			'trustlens_notify_high_risk_order'    => $can_use_pro,
			'trustlens_notify_segment_critical'   => $can_use_pro,
			'trustlens_enable_daily_digest'       => false,
			'trustlens_notify_high_value_order'   => $can_use_pro,
			'trustlens_notify_high_value_threshold' => 500,
			'trustlens_notify_high_value_score'   => 30,
			'trustlens_notify_repeat_refunder'    => $can_use_pro,
			'trustlens_notify_repeat_refunder_count' => 3,
			'trustlens_notify_repeat_refunder_days'  => 30,
			'trustlens_notify_velocity'           => $can_use_pro,
			'trustlens_notify_velocity_count'     => 3,
			'trustlens_notify_velocity_hours'     => 24,
			'trustlens_notify_score_recovery'     => $can_use_pro,
			'trustlens_notify_new_customer_risk'  => $can_use_pro,
			'trustlens_notify_new_customer_risk_score' => 25,
			'trustlens_notify_monthly_report'     => false,
			'trustlens_notify_chargeback_filed'   => $can_use_pro,
			// Automation settings.
			'trustlens_enable_automation'      => false,
			'trustlens_automation_rules'       => array(),
			// Webhook settings.
			'trustlens_enable_webhooks'        => false,
			'trustlens_webhook_endpoints'      => array(),
			'trustlens_webhook_events'         => array( 'score_updated', 'customer_blocked', 'checkout_blocked', 'high_risk_order' ),
			// Payment method risk controls (Pro).
			'trustlens_enable_payment_method_controls' => false,
			'trustlens_payment_method_control_segments' => array( 'risk', 'critical' ),
			'trustlens_payment_method_control_gateways' => array(),
			'trustlens_payment_method_control_min_total' => 0,
			'trustlens_payment_method_control_message' => __( 'Some payment methods are unavailable for this order. Please choose another payment option.', 'trustlens' ),
			'trustlens_payment_method_control_velocity_enabled' => false,
			'trustlens_payment_method_control_velocity_count' => 3,
			'trustlens_payment_method_control_velocity_hours' => 24,
			'trustlens_payment_method_control_linked_risk_enabled' => false,
			// Linked accounts settings.
			'trustlens_module_linked_accounts_enabled' => true,
			'trustlens_linked_accounts_penalty' => 5,
			// Scheduled reports settings.
			'trustlens_enable_weekly_reports'  => false,
			'trustlens_enable_monthly_reports' => false,
			'trustlens_weekly_report_recipients'  => get_option( 'admin_email' ),
			'trustlens_monthly_report_recipients' => get_option( 'admin_email' ),
		);

		foreach ( $defaults as $key => $value ) {
			if ( get_option( $key ) === false ) {
				update_option( $key, $value );
			}
		}
	}

	/**
	 * Schedule recurring events.
	 *
	 * @since 1.0.0
	 */
	public static function schedule_events(): void {
		// Schedule daily cleanup.
		if ( ! wp_next_scheduled( 'trustlens/daily_cleanup' ) ) {
			wp_schedule_event( time(), 'daily', 'trustlens/daily_cleanup' );
		}

		// Pro-only: daily digest and monthly report (handlers in class-notifications-pro.php).
		$has_pro_notifications = file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-notifications-pro.php' ) && self::can_use_pro_features();
		if ( $has_pro_notifications ) {
			if ( ! wp_next_scheduled( 'trustlens/daily_digest' ) ) {
				$tomorrow_7am = strtotime( 'tomorrow 7:00am' );
				wp_schedule_event( $tomorrow_7am, 'daily', 'trustlens/daily_digest' );
			}
			if ( ! wp_next_scheduled( 'trustlens/monthly_report' ) ) {
				$first_of_month = strtotime( 'first day of next month 8:00am' );
				wp_schedule_event( $first_of_month, 'monthly', 'trustlens/monthly_report' );
			}
		}

		// Schedule weekly summary (Monday 7 AM) — free notification.
		if ( ! wp_next_scheduled( 'trustlens/weekly_summary' ) ) {
			$next_monday = strtotime( 'next monday 7:00am' );
			wp_schedule_event( $next_monday, 'weekly', 'trustlens/weekly_summary' );
		}
	}

	/**
	 * Unschedule events on deactivation.
	 *
	 * @since 1.0.0
	 */
	public static function unschedule_events(): void {
		$events = array(
			'trustlens/daily_cleanup',
			'trustlens/daily_digest',
			'trustlens/weekly_summary',
			'trustlens/monthly_report',
			'trustlens/welcome_summary',
		);

		foreach ( $events as $event ) {
			$timestamp = wp_next_scheduled( $event );
			if ( $timestamp ) {
				wp_unschedule_event( $timestamp, $event );
			}
		}
	}

	/**
	 * Check if database needs upgrade.
	 *
	 * @since 1.0.0
	 */
	public static function maybe_upgrade(): void {
		$installed_version = get_option( 'trustlens_db_version' );

		if ( $installed_version !== TRUSTLENS_DB_VERSION ) {
			self::create_tables();
		}
	}
}

// Check for database upgrades on plugins_loaded.
add_action( 'plugins_loaded', array( 'TrustLens_Install', 'maybe_upgrade' ), 5 );
