<?php
/**
 * Data export functionality for GDPR compliance.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Data export class.
 *
 * Handles data export and erasure for GDPR compliance.
 *
 * @since 1.0.0
 */
class TrustLens_Data_Export {

	/**
	 * Single instance.
	 *
	 * @var TrustLens_Data_Export|null
	 */
	private static ?TrustLens_Data_Export $instance = null;

	/**
	 * Get instance.
	 *
	 * @since 1.0.0
	 * @return TrustLens_Data_Export
	 */
	public static function instance(): TrustLens_Data_Export {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks(): void {
		// WordPress privacy hooks.
		add_filter( 'wp_privacy_personal_data_exporters', array( $this, 'register_exporter' ) );
		add_filter( 'wp_privacy_personal_data_erasers', array( $this, 'register_eraser' ) );

		// Admin export action.
		add_action( 'wp_ajax_wstl_export_customers', array( $this, 'ajax_export_customers' ) );
		add_action( 'wp_ajax_wstl_export_customer_data', array( $this, 'ajax_export_customer_data' ) );
	}

	/**
	 * Register data exporter.
	 *
	 * @since 1.0.0
	 * @param array $exporters Registered exporters.
	 * @return array Modified exporters.
	 */
	public function register_exporter( array $exporters ): array {
		$exporters['trustlens'] = array(
			'exporter_friendly_name' => __( 'TrustLens Customer Data', 'trustlens' ),
			'callback'               => array( $this, 'export_personal_data' ),
		);
		return $exporters;
	}

	/**
	 * Register data eraser.
	 *
	 * @since 1.0.0
	 * @param array $erasers Registered erasers.
	 * @return array Modified erasers.
	 */
	public function register_eraser( array $erasers ): array {
		$erasers['trustlens'] = array(
			'eraser_friendly_name' => __( 'TrustLens Customer Data', 'trustlens' ),
			'callback'             => array( $this, 'erase_personal_data' ),
		);
		return $erasers;
	}

	/**
	 * Export personal data for GDPR request.
	 *
	 * @since 1.0.0
	 * @param string $email_address Email address.
	 * @param int    $page          Page number.
	 * @return array Export data.
	 */
	public function export_personal_data( string $email_address, int $page = 1 ): array {
		$export_items = array();
		$email_hash = wstl_get_email_hash( $email_address );
		$customer = wstl_get_customer( $email_hash );

		if ( $customer ) {
			// Customer profile data.
			$data = array(
				array(
					'name'  => __( 'Email Hash', 'trustlens' ),
					'value' => $customer->email_hash,
				),
				array(
					'name'  => __( 'Trust Score', 'trustlens' ),
					'value' => $customer->trust_score . '/100',
				),
				array(
					'name'  => __( 'Segment', 'trustlens' ),
					'value' => wstl_get_segment_label( $customer->segment ),
				),
				array(
					'name'  => __( 'Total Orders', 'trustlens' ),
					'value' => $customer->total_orders,
				),
				array(
					'name'  => __( 'Total Refunds', 'trustlens' ),
					'value' => $customer->total_refunds,
				),
				array(
					'name'  => __( 'Return Rate', 'trustlens' ),
					'value' => number_format( $customer->return_rate, 1 ) . '%',
				),
				array(
					'name'  => __( 'Is Blocked', 'trustlens' ),
					'value' => $customer->is_blocked ? __( 'Yes', 'trustlens' ) : __( 'No', 'trustlens' ),
				),
				array(
					'name'  => __( 'Is Allowlisted', 'trustlens' ),
					'value' => $customer->is_allowlisted ? __( 'Yes', 'trustlens' ) : __( 'No', 'trustlens' ),
				),
				array(
					'name'  => __( 'First Order Date', 'trustlens' ),
					'value' => $customer->first_order_date ?: __( 'N/A', 'trustlens' ),
				),
				array(
					'name'  => __( 'Last Order Date', 'trustlens' ),
					'value' => $customer->last_order_date ?: __( 'N/A', 'trustlens' ),
				),
			);

			$export_items[] = array(
				'group_id'          => 'trustlens_customer',
				'group_label'       => __( 'TrustLens Customer Profile', 'trustlens' ),
				'group_description' => __( 'Customer trust and behavior data collected by TrustLens.', 'trustlens' ),
				'item_id'           => 'customer-' . $customer->id,
				'data'              => $data,
			);

			// Events data.
			global $wpdb;
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
			$events = $wpdb->get_results( $wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}trustlens_events WHERE email_hash = %s ORDER BY created_at DESC LIMIT 100",
				$email_hash
			) );
			// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

			foreach ( $events as $event ) {
				$export_items[] = array(
					'group_id'          => 'trustlens_events',
					'group_label'       => __( 'TrustLens Events', 'trustlens' ),
					'group_description' => __( 'Activity events tracked by TrustLens.', 'trustlens' ),
					'item_id'           => 'event-' . $event->id,
					'data'              => array(
						array(
							'name'  => __( 'Event Type', 'trustlens' ),
							'value' => $event->event_type,
						),
						array(
							'name'  => __( 'Date', 'trustlens' ),
							'value' => $event->created_at,
						),
						array(
							'name'  => __( 'Order ID', 'trustlens' ),
							'value' => $event->order_id ?: __( 'N/A', 'trustlens' ),
						),
					),
				);
			}
		}

		return array(
			'data' => $export_items,
			'done' => true,
		);
	}

	/**
	 * Erase personal data for GDPR request.
	 *
	 * @since 1.0.0
	 * @param string $email_address Email address.
	 * @param int    $page          Page number.
	 * @return array Erasure result.
	 */
	public function erase_personal_data( string $email_address, int $page = 1 ): array {
		global $wpdb;

		$email_hash = wstl_get_email_hash( $email_address );
		$items_removed = 0;

		// Delete customer record.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$customer_deleted = $wpdb->delete(
			$wpdb->prefix . 'trustlens_customers',
			array( 'email_hash' => $email_hash )
		);

		if ( $customer_deleted ) {
			$items_removed++;
		}

		// Delete events.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$events_deleted = $wpdb->delete(
			$wpdb->prefix . 'trustlens_events',
			array( 'email_hash' => $email_hash )
		);

		$items_removed += (int) $events_deleted;

		// Delete signals.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$signals_deleted = $wpdb->delete(
			$wpdb->prefix . 'trustlens_signals',
			array( 'email_hash' => $email_hash )
		);

		$items_removed += (int) $signals_deleted;

		return array(
			'items_removed'  => $items_removed,
			'items_retained' => false,
			'messages'       => array(),
			'done'           => true,
		);
	}

	/**
	 * AJAX: Export all customers to CSV.
	 *
	 * @since 1.0.0
	 */
	public function ajax_export_customers(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Export query, table name from $wpdb->prefix is safe.
		$customers = $wpdb->get_results(
			"SELECT * FROM {$wpdb->prefix}trustlens_customers ORDER BY trust_score ASC"
		);

		$csv_data = array();
		$csv_data[] = array(
			'Email',
			'Trust Score',
			'Segment',
			'Total Orders',
			'Total Refunds',
			'Return Rate',
			'Total Order Value',
			'Total Refund Value',
			'Is Blocked',
			'Is Allowlisted',
			'First Order',
			'Last Order',
		);

		foreach ( $customers as $customer ) {
			$csv_data[] = array(
				$customer->customer_email,
				$customer->trust_score,
				$customer->segment,
				$customer->total_orders,
				$customer->total_refunds,
				number_format( $customer->return_rate, 1 ) . '%',
				$customer->total_order_value,
				$customer->total_refund_value,
				$customer->is_blocked ? 'Yes' : 'No',
				$customer->is_allowlisted ? 'Yes' : 'No',
				$customer->first_order_date,
				$customer->last_order_date,
			);
		}

		$filename = 'trustlens-customers-' . gmdate( 'Y-m-d' ) . '.csv';

		wp_send_json_success( array(
			'filename' => $filename,
			'data'     => $csv_data,
		) );
	}

	/**
	 * AJAX: Export single customer data.
	 *
	 * @since 1.0.0
	 */
	public function ajax_export_customer_data(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$email_hash = sanitize_text_field( wp_unslash( $_POST['email_hash'] ?? '' ) );

		if ( empty( $email_hash ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid customer.', 'trustlens' ) ) );
		}

		$customer = wstl_get_customer( $email_hash );

		if ( ! $customer ) {
			wp_send_json_error( array( 'message' => __( 'Customer not found.', 'trustlens' ) ) );
		}

		// Get events.
		global $wpdb;
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$events = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}trustlens_events WHERE email_hash = %s ORDER BY created_at DESC",
			$email_hash
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Get signals.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$signals = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}trustlens_signals WHERE email_hash = %s",
			$email_hash
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		$export_data = array(
			'customer' => $customer,
			'events'   => $events,
			'signals'  => $signals,
		);

		$filename = 'trustlens-customer-' . substr( $email_hash, 0, 8 ) . '-' . gmdate( 'Y-m-d' ) . '.json';

		wp_send_json_success( array(
			'filename' => $filename,
			'data'     => $export_data,
		) );
	}
}
