<?php
/**
 * Customer repository for database operations.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Customer repository class.
 *
 * @since 1.0.0
 */
class TrustLens_Customer_Repository {

	/**
	 * Table name.
	 *
	 * @var string
	 */
	private string $table;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		global $wpdb;
		$this->table = $wpdb->prefix . 'trustlens_customers';
	}

	/**
	 * Get customer by email hash.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @return object|null Customer object or null.
	 */
	public function get_by_hash( string $email_hash ): ?object {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$customer = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table} WHERE email_hash = %s",
			$email_hash
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		return $customer ?: null;
	}

	/**
	 * Get customer by ID.
	 *
	 * @since 1.0.0
	 * @param int $id Customer ID.
	 * @return object|null Customer object or null.
	 */
	public function get_by_id( int $id ): ?object {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$customer = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table} WHERE id = %d",
			$id
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		return $customer ?: null;
	}

	/**
	 * Get customer by WP user ID.
	 *
	 * @since 1.0.0
	 * @param int $customer_id WooCommerce customer ID.
	 * @return object|null Customer object or null.
	 */
	public function get_by_customer_id( int $customer_id ): ?object {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$customer = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table} WHERE customer_id = %d",
			$customer_id
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		return $customer ?: null;
	}

	/**
	 * Update customer data.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @param array  $data       Data to update.
	 * @return bool True on success.
	 */
	public function update( string $email_hash, array $data ): bool {
		global $wpdb;

		$data['updated_at'] = current_time( 'mysql' );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$result = $wpdb->update(
			$this->table,
			$data,
			array( 'email_hash' => $email_hash )
		);

		return $result !== false;
	}

	/**
	 * Insert or update customer.
	 *
	 * @since 1.0.0
	 * @param array $data Customer data.
	 * @return bool True on success.
	 */
	public function upsert( array $data ): bool {
		global $wpdb;

		if ( empty( $data['email_hash'] ) ) {
			return false;
		}

		$existing = $this->get_by_hash( $data['email_hash'] );

		if ( $existing ) {
			return $this->update( $data['email_hash'], $data );
		}

		$data['created_at'] = current_time( 'mysql' );
		$data['updated_at'] = current_time( 'mysql' );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$result = $wpdb->insert( $this->table, $data );

		return $result !== false;
	}

	/**
	 * Delete customer.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @return bool True on success.
	 */
	public function delete( string $email_hash ): bool {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$result = $wpdb->delete(
			$this->table,
			array( 'email_hash' => $email_hash )
		);

		return $result !== false;
	}

	/**
	 * Get customers with pagination.
	 *
	 * @since 1.0.0
	 * @param array $args Query arguments.
	 * @return array Customers array.
	 */
	public function get_customers( array $args = array() ): array {
		global $wpdb;

		$defaults = array(
			'per_page' => 20,
			'page'     => 1,
			'orderby'  => 'trust_score',
			'order'    => 'ASC',
			'segment'  => '',
			'search'   => '',
		);

		$args = wp_parse_args( $args, $defaults );

		$where = array( '1=1' );
		$values = array();

		// Filter by segment.
		if ( ! empty( $args['segment'] ) ) {
			$where[] = 'segment = %s';
			$values[] = $args['segment'];
		}

		// Search by email.
		if ( ! empty( $args['search'] ) ) {
			$where[] = 'customer_email LIKE %s';
			$values[] = '%' . $wpdb->esc_like( $args['search'] ) . '%';
		}

		$where_sql = implode( ' AND ', $where );

		// Sanitize orderby.
		$allowed_orderby = array(
			'trust_score', 'total_orders', 'total_refunds', 'return_rate',
			'total_refund_value', 'last_order_date', 'created_at',
		);
		$orderby = in_array( $args['orderby'], $allowed_orderby, true ) ? $args['orderby'] : 'trust_score';
		$order = strtoupper( $args['order'] ) === 'DESC' ? 'DESC' : 'ASC';

		$offset = ( $args['page'] - 1 ) * $args['per_page'];

		$sql = "SELECT * FROM {$this->table} WHERE {$where_sql} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
		$values[] = $args['per_page'];
		$values[] = $offset;

		if ( ! empty( $values ) ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $sql is built above with whitelisted orderby/order and table from $wpdb->prefix.
			$sql = $wpdb->prepare( $sql, $values );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- SQL is prepared above when values exist, orderby/order are whitelisted, table from $wpdb->prefix.
		return $wpdb->get_results( $sql );
	}

	/**
	 * Count customers.
	 *
	 * @since 1.0.0
	 * @param array $args Query arguments.
	 * @return int Customer count.
	 */
	public function count( array $args = array() ): int {
		global $wpdb;

		$where = array( '1=1' );
		$values = array();

		if ( ! empty( $args['segment'] ) ) {
			$where[] = 'segment = %s';
			$values[] = $args['segment'];
		}

		if ( ! empty( $args['search'] ) ) {
			$where[] = 'customer_email LIKE %s';
			$values[] = '%' . $wpdb->esc_like( $args['search'] ) . '%';
		}

		$where_sql = implode( ' AND ', $where );
		$sql = "SELECT COUNT(*) FROM {$this->table} WHERE {$where_sql}";

		if ( ! empty( $values ) ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $sql is built above with table from $wpdb->prefix.
			$sql = $wpdb->prepare( $sql, $values );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- SQL is prepared above when values exist, table from $wpdb->prefix.
		return (int) $wpdb->get_var( $sql );
	}

	/**
	 * Get customers by segment.
	 *
	 * @since 1.0.0
	 * @param string $segment Segment name.
	 * @param int    $limit   Number of customers.
	 * @return array Customers array.
	 */
	public function get_by_segment( string $segment, int $limit = 10 ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		return $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$this->table}
			 WHERE segment = %s AND is_allowlisted = 0
			 ORDER BY trust_score ASC, total_refund_value DESC
			 LIMIT %d",
			$segment,
			$limit
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
	}

	/**
	 * Get high-risk customers.
	 *
	 * @since 1.0.0
	 * @param int $limit Number of customers.
	 * @return array Customers array.
	 */
	public function get_high_risk( int $limit = 10 ): array {
		global $wpdb;

		$cache_key = 'wstl_high_risk_' . $limit;
		$cached = get_transient( $cache_key );

		if ( $cached !== false ) {
			return $cached;
		}

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$this->table}
			 WHERE segment IN ('risk', 'critical') AND is_allowlisted = 0
			 ORDER BY FIELD(segment, 'critical', 'risk'), trust_score ASC
			 LIMIT %d",
			$limit
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		set_transient( $cache_key, $results, 5 * MINUTE_IN_SECONDS );

		return $results;
	}

	/**
	 * Get segment counts.
	 *
	 * @since 1.0.0
	 * @return array Segment counts.
	 */
	public function get_segment_counts(): array {
		global $wpdb;

		$cache_key = 'wstl_segment_counts';
		$cached = get_transient( $cache_key );

		if ( $cached !== false ) {
			return $cached;
		}

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Aggregate query, table from $wpdb->prefix, result cached in transient above.
		$results = $wpdb->get_results(
			"SELECT segment, COUNT(*) as count FROM {$this->table} GROUP BY segment",
			OBJECT_K
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		$counts = array(
			'vip'      => 0,
			'trusted'  => 0,
			'normal'   => 0,
			'caution'  => 0,
			'risk'     => 0,
			'critical' => 0,
		);

		foreach ( $results as $segment => $row ) {
			$counts[ $segment ] = (int) $row->count;
		}

		set_transient( $cache_key, $counts, 5 * MINUTE_IN_SECONDS );

		return $counts;
	}

	/**
	 * Increment customer stats.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @param string $field      Field to increment.
	 * @param int    $amount     Amount to add.
	 * @return bool True on success.
	 */
	public function increment( string $email_hash, string $field, int $amount = 1 ): bool {
		global $wpdb;

		$allowed_fields = array(
			'total_orders', 'total_refunds', 'full_refunds', 'partial_refunds',
			'total_coupons_used', 'first_order_coupons', 'coupon_then_refund',
			'total_disputes', 'disputes_won', 'disputes_lost',
			'linked_accounts', 'cancelled_orders', 'order_edits', 'reviews_before_refund',
		);

		if ( ! in_array( $field, $allowed_fields, true ) ) {
			return false;
		}

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table from $wpdb->prefix, $field validated against allowlist above.
		$result = $wpdb->query( $wpdb->prepare(
			"UPDATE {$this->table} SET {$field} = {$field} + %d, updated_at = %s WHERE email_hash = %s",
			$amount,
			current_time( 'mysql' ),
			$email_hash
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		return $result !== false;
	}

	/**
	 * Update return rate for customer.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @return bool True on success.
	 */
	public function update_return_rate( string $email_hash ): bool {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$result = $wpdb->query( $wpdb->prepare(
			"UPDATE {$this->table}
			 SET return_rate = CASE
			     WHEN total_orders > 0 THEN (total_refunds / total_orders) * 100
			     ELSE 0
			 END,
			 updated_at = %s
			 WHERE email_hash = %s",
			current_time( 'mysql' ),
			$email_hash
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		return $result !== false;
	}

	/**
	 * Invalidate caches.
	 *
	 * @since 1.0.0
	 */
	public function invalidate_caches(): void {
		delete_transient( 'wstl_segment_counts' );
		delete_transient( 'wstl_high_risk_10' );
		delete_transient( 'wstl_high_risk_25' );
	}
}
