<?php
/**
 * Checkout blocking functionality.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Checkout blocker class.
 *
 * Prevents blocked customers from completing checkout.
 *
 * @since 1.0.0
 */
class TrustLens_Checkout_Blocker {

	/**
	 * Single instance.
	 *
	 * @var TrustLens_Checkout_Blocker|null
	 */
	private static ?TrustLens_Checkout_Blocker $instance = null;

	/**
	 * Get instance.
	 *
	 * @since 1.0.0
	 * @return TrustLens_Checkout_Blocker
	 */
	public static function instance(): TrustLens_Checkout_Blocker {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks(): void {
		// Only enable if blocking is turned on.
		if ( ! get_option( 'trustlens_enable_blocking', false ) ) {
			return;
		}

		// Validate checkout.
		add_action( 'woocommerce_checkout_process', array( $this, 'validate_checkout' ) );

		// Also check on add to cart for blocked customers.
		add_filter( 'woocommerce_add_to_cart_validation', array( $this, 'validate_add_to_cart' ), 10, 2 );
	}

	/**
	 * Validate checkout for blocked customers.
	 *
	 * @since 1.0.0
	 */
	public function validate_checkout(): void {
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout process handles nonce verification.
		$email = isset( $_POST['billing_email'] ) ? sanitize_email( wp_unslash( $_POST['billing_email'] ) ) : '';

		if ( empty( $email ) ) {
			return;
		}

		if ( $this->is_customer_blocked( $email ) ) {
			$message = get_option(
				'trustlens_block_message',
				__( 'We are unable to process your order at this time. Please contact support for assistance.', 'trustlens' )
			);

			wc_add_notice( $message, 'error' );

			// Log the blocked checkout attempt.
			$email_hash = wstl_get_email_hash( $email );
			wstl_log_event( $email_hash, 'checkout_blocked', array(
				'email' => $email,
				'ip'    => $this->get_customer_ip(),
			) );

			/**
			 * Fires when a blocked customer attempts checkout.
			 *
			 * @since 1.0.0
			 * @param string $email_hash Customer email hash.
			 * @param string $email      Customer email.
			 */
			do_action( 'trustlens/checkout_blocked', $email_hash, $email );
		}
	}

	/**
	 * Validate add to cart for blocked customers.
	 *
	 * @since 1.0.0
	 * @param bool $passed     Whether validation passed.
	 * @param int  $product_id Product ID being added.
	 * @return bool Whether to allow add to cart.
	 */
	public function validate_add_to_cart( bool $passed, int $product_id ): bool {
		// Only check for logged-in users.
		if ( ! is_user_logged_in() ) {
			return $passed;
		}

		$user = wp_get_current_user();
		$email = $user->user_email;

		if ( $this->is_customer_blocked( $email ) ) {
			$message = get_option(
				'trustlens_block_message',
				__( 'We are unable to process your order at this time. Please contact support for assistance.', 'trustlens' )
			);

			wc_add_notice( $message, 'error' );
			return false;
		}

		return $passed;
	}

	/**
	 * Check if customer is blocked.
	 *
	 * @since 1.0.0
	 * @param string $email Customer email.
	 * @return bool True if blocked.
	 */
	private function is_customer_blocked( string $email ): bool {
		$email_hash = wstl_get_email_hash( $email );
		$customer = wstl_get_customer( $email_hash );

		if ( ! $customer ) {
			return false;
		}

		// Skip if allowlisted.
		if ( $customer->is_allowlisted ) {
			return false;
		}

		return (bool) $customer->is_blocked;
	}

	/**
	 * Get customer IP address.
	 *
	 * @since 1.0.0
	 * @return string IP address.
	 */
	private function get_customer_ip(): string {
		if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
		} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
		} else {
			$ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
		}

		// Handle multiple IPs in X-Forwarded-For.
		if ( strpos( $ip, ',' ) !== false ) {
			$ips = explode( ',', $ip );
			$ip = trim( $ips[0] );
		}

		return $ip;
	}
}
