<?php
/**
 * Bulk Operations.
 *
 * @package TrustLens
 * @since   1.1.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Bulk Operations class.
 *
 * Handles bulk actions on customers.
 *
 * @since 1.1.0
 */
class TrustLens_Bulk_Operations {

	/**
	 * Block multiple customers.
	 *
	 * @since 1.1.0
	 * @param array  $email_hashes Array of customer email hashes.
	 * @param string $reason       Block reason.
	 * @return array Results with success/failure counts.
	 */
	public static function bulk_block( array $email_hashes, string $reason = '' ): array {
		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		foreach ( $email_hashes as $email_hash ) {
			$result = wstl_block_customer( $email_hash, $reason );
			if ( $result ) {
				$results['success']++;
			} else {
				$results['failed']++;
				$results['errors'][] = $email_hash;
			}
		}

		// Log bulk action.
		self::log_bulk_action( 'block', $email_hashes, $results, array( 'reason' => $reason ) );

		return $results;
	}

	/**
	 * Unblock multiple customers.
	 *
	 * @since 1.1.0
	 * @param array $email_hashes Array of customer email hashes.
	 * @return array Results with success/failure counts.
	 */
	public static function bulk_unblock( array $email_hashes ): array {
		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		foreach ( $email_hashes as $email_hash ) {
			$result = wstl_unblock_customer( $email_hash );
			if ( $result ) {
				$results['success']++;
			} else {
				$results['failed']++;
				$results['errors'][] = $email_hash;
			}
		}

		self::log_bulk_action( 'unblock', $email_hashes, $results );

		return $results;
	}

	/**
	 * Add multiple customers to allowlist.
	 *
	 * @since 1.1.0
	 * @param array $email_hashes Array of customer email hashes.
	 * @return array Results.
	 */
	public static function bulk_allowlist( array $email_hashes ): array {
		global $wpdb;

		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		$table = $wpdb->prefix . 'trustlens_customers';

		foreach ( $email_hashes as $email_hash ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
			$updated = $wpdb->update(
				$table,
				array(
					'is_allowlisted' => 1,
					'is_blocked'     => 0,
				),
				array( 'email_hash' => $email_hash ),
				array( '%d', '%d' ),
				array( '%s' )
			);

			if ( false !== $updated ) {
				$results['success']++;
			} else {
				$results['failed']++;
				$results['errors'][] = $email_hash;
			}
		}

		self::log_bulk_action( 'allowlist', $email_hashes, $results );

		return $results;
	}

	/**
	 * Remove multiple customers from allowlist.
	 *
	 * @since 1.1.0
	 * @param array $email_hashes Array of customer email hashes.
	 * @return array Results.
	 */
	public static function bulk_remove_allowlist( array $email_hashes ): array {
		global $wpdb;

		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		$table = $wpdb->prefix . 'trustlens_customers';

		foreach ( $email_hashes as $email_hash ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
			$updated = $wpdb->update(
				$table,
				array( 'is_allowlisted' => 0 ),
				array( 'email_hash' => $email_hash ),
				array( '%d' ),
				array( '%s' )
			);

			if ( false !== $updated ) {
				$results['success']++;
			} else {
				$results['failed']++;
				$results['errors'][] = $email_hash;
			}
		}

		self::log_bulk_action( 'remove_allowlist', $email_hashes, $results );

		return $results;
	}

	/**
	 * Recalculate scores for multiple customers.
	 *
	 * @since 1.1.0
	 * @param array $email_hashes Array of customer email hashes.
	 * @return array Results.
	 */
	public static function bulk_recalculate_scores( array $email_hashes ): array {
		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		$score_calculator = TrustLens_Score_Calculator::instance();

		foreach ( $email_hashes as $email_hash ) {
			try {
				$score_calculator->recalculate_score( $email_hash );
				$results['success']++;
			} catch ( Exception $e ) {
				$results['failed']++;
				$results['errors'][] = array(
					'email_hash' => $email_hash,
					'error'      => $e->getMessage(),
				);
			}
		}

		self::log_bulk_action( 'recalculate_scores', $email_hashes, $results );

		return $results;
	}

	/**
	 * Add tag to multiple customers.
	 *
	 * @since 1.1.0
	 * @param array  $email_hashes Array of customer email hashes.
	 * @param string $tag          Tag to add.
	 * @return array Results.
	 */
	public static function bulk_add_tag( array $email_hashes, string $tag ): array {
		global $wpdb;

		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		$table = $wpdb->prefix . 'trustlens_customers';

		foreach ( $email_hashes as $email_hash ) {
			$customer = wstl_get_customer( $email_hash );
			if ( ! $customer ) {
				$results['failed']++;
				$results['errors'][] = $email_hash;
				continue;
			}

			$tags = ! empty( $customer->tags ) ? json_decode( $customer->tags, true ) : array();
			if ( ! is_array( $tags ) ) {
				$tags = array();
			}

			if ( ! in_array( $tag, $tags, true ) ) {
				$tags[] = $tag;
			}

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
			$updated = $wpdb->update(
				$table,
				array( 'tags' => wp_json_encode( $tags ) ),
				array( 'email_hash' => $email_hash ),
				array( '%s' ),
				array( '%s' )
			);

			if ( false !== $updated ) {
				$results['success']++;
			} else {
				$results['failed']++;
				$results['errors'][] = $email_hash;
			}
		}

		self::log_bulk_action( 'add_tag', $email_hashes, $results, array( 'tag' => $tag ) );

		return $results;
	}

	/**
	 * Remove tag from multiple customers.
	 *
	 * @since 1.1.0
	 * @param array  $email_hashes Array of customer email hashes.
	 * @param string $tag          Tag to remove.
	 * @return array Results.
	 */
	public static function bulk_remove_tag( array $email_hashes, string $tag ): array {
		global $wpdb;

		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		$table = $wpdb->prefix . 'trustlens_customers';

		foreach ( $email_hashes as $email_hash ) {
			$customer = wstl_get_customer( $email_hash );
			if ( ! $customer ) {
				$results['failed']++;
				$results['errors'][] = $email_hash;
				continue;
			}

			$tags = ! empty( $customer->tags ) ? json_decode( $customer->tags, true ) : array();
			if ( ! is_array( $tags ) ) {
				$tags = array();
			}

			$tags = array_filter( $tags, function( $t ) use ( $tag ) {
				return $t !== $tag;
			} );

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
			$updated = $wpdb->update(
				$table,
				array( 'tags' => wp_json_encode( array_values( $tags ) ) ),
				array( 'email_hash' => $email_hash ),
				array( '%s' ),
				array( '%s' )
			);

			if ( false !== $updated ) {
				$results['success']++;
			} else {
				$results['failed']++;
				$results['errors'][] = $email_hash;
			}
		}

		self::log_bulk_action( 'remove_tag', $email_hashes, $results, array( 'tag' => $tag ) );

		return $results;
	}

	/**
	 * Delete multiple customers.
	 *
	 * @since 1.1.0
	 * @param array $email_hashes Array of customer email hashes.
	 * @return array Results.
	 */
	public static function bulk_delete( array $email_hashes ): array {
		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		foreach ( $email_hashes as $email_hash ) {
			$result = wstl_delete_customer( $email_hash );
			if ( $result ) {
				$results['success']++;
			} else {
				$results['failed']++;
				$results['errors'][] = $email_hash;
			}
		}

		self::log_bulk_action( 'delete', $email_hashes, $results );

		return $results;
	}

	/**
	 * Export multiple customers.
	 *
	 * @since 1.1.0
	 * @param array  $email_hashes Array of customer email hashes.
	 * @param string $format       Export format (csv, json).
	 * @return string|array Export data.
	 */
	public static function bulk_export( array $email_hashes, string $format = 'csv' ) {
		$customers = array();

		foreach ( $email_hashes as $email_hash ) {
			$customer = wstl_get_customer( $email_hash );
			if ( $customer ) {
				$customers[] = array(
					'email_hash'        => $customer->email_hash,
					'email'             => $customer->customer_email ?? '',
					'trust_score'       => (int) $customer->trust_score,
					'segment'           => $customer->segment,
					'total_orders'      => (int) $customer->total_orders,
					'total_refunds'     => (int) $customer->total_refunds,
					'return_rate'       => (float) $customer->return_rate,
					'total_order_value' => (float) $customer->total_order_value,
					'total_refund_value' => (float) $customer->total_refund_value,
					'is_blocked'        => (bool) $customer->is_blocked,
					'is_allowlisted'    => (bool) $customer->is_allowlisted,
					'created_at'        => $customer->created_at,
					'updated_at'        => $customer->updated_at,
				);
			}
		}

		if ( 'json' === $format ) {
			return $customers;
		}

		// CSV format.
		if ( empty( $customers ) ) {
			return '';
		}

		$csv = array();
		$csv[] = implode( ',', array_keys( $customers[0] ) );

		foreach ( $customers as $customer ) {
			$row = array();
			foreach ( $customer as $value ) {
				if ( is_bool( $value ) ) {
					$row[] = $value ? 'true' : 'false';
				} elseif ( is_numeric( $value ) ) {
					$row[] = $value;
				} else {
					$row[] = '"' . str_replace( '"', '""', $value ) . '"';
				}
			}
			$csv[] = implode( ',', $row );
		}

		return implode( "\n", $csv );
	}

	/**
	 * Get customers by segment for bulk operations.
	 *
	 * @since 1.1.0
	 * @param string $segment Segment name.
	 * @param int    $limit   Maximum customers to return.
	 * @return array Email hashes.
	 */
	public static function get_customers_by_segment( string $segment, int $limit = 1000 ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		return $wpdb->get_col( $wpdb->prepare(
			"SELECT email_hash FROM {$wpdb->prefix}trustlens_customers
			 WHERE segment = %s
			 LIMIT %d",
			$segment,
			$limit
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
	}

	/**
	 * Get customers by score range for bulk operations.
	 *
	 * @since 1.1.0
	 * @param int $min_score Minimum score.
	 * @param int $max_score Maximum score.
	 * @param int $limit     Maximum customers to return.
	 * @return array Email hashes.
	 */
	public static function get_customers_by_score_range( int $min_score, int $max_score, int $limit = 1000 ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		return $wpdb->get_col( $wpdb->prepare(
			"SELECT email_hash FROM {$wpdb->prefix}trustlens_customers
			 WHERE trust_score >= %d AND trust_score <= %d
			 LIMIT %d",
			$min_score,
			$max_score,
			$limit
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
	}

	/**
	 * Get customers by return rate for bulk operations.
	 *
	 * @since 1.1.0
	 * @param float $min_rate Minimum return rate.
	 * @param int   $limit    Maximum customers to return.
	 * @return array Email hashes.
	 */
	public static function get_customers_by_return_rate( float $min_rate, int $limit = 1000 ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		return $wpdb->get_col( $wpdb->prepare(
			"SELECT email_hash FROM {$wpdb->prefix}trustlens_customers
			 WHERE return_rate >= %f
			 ORDER BY return_rate DESC
			 LIMIT %d",
			$min_rate,
			$limit
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
	}

	/**
	 * Schedule a bulk operation for background processing.
	 *
	 * @since 1.1.0
	 * @param string $action       Action to perform.
	 * @param array  $email_hashes Customer email hashes.
	 * @param array  $args         Additional arguments.
	 * @return string|false Job ID or false on failure.
	 */
	public static function schedule_bulk_operation( string $action, array $email_hashes, array $args = array() ) {
		if ( ! function_exists( 'as_schedule_single_action' ) ) {
			return false;
		}

		$job_id = wp_generate_uuid4();

		// Store job data.
		set_transient(
			'trustlens_bulk_job_' . $job_id,
			array(
				'action'       => $action,
				'email_hashes' => $email_hashes,
				'args'         => $args,
				'status'       => 'pending',
				'created_at'   => current_time( 'mysql' ),
			),
			DAY_IN_SECONDS
		);

		// Schedule the action.
		as_schedule_single_action(
			time(),
			'trustlens/process_bulk_operation',
			array( 'job_id' => $job_id ),
			'trustlens'
		);

		return $job_id;
	}

	/**
	 * Process a scheduled bulk operation.
	 *
	 * @since 1.1.0
	 * @param string $job_id Job ID.
	 */
	public static function process_scheduled_operation( string $job_id ): void {
		$job = get_transient( 'trustlens_bulk_job_' . $job_id );

		if ( ! $job ) {
			return;
		}

		// Update status to processing.
		$job['status'] = 'processing';
		$job['started_at'] = current_time( 'mysql' );
		set_transient( 'trustlens_bulk_job_' . $job_id, $job, DAY_IN_SECONDS );

		$results = null;

		switch ( $job['action'] ) {
			case 'block':
				$results = self::bulk_block( $job['email_hashes'], $job['args']['reason'] ?? '' );
				break;

			case 'unblock':
				$results = self::bulk_unblock( $job['email_hashes'] );
				break;

			case 'allowlist':
				$results = self::bulk_allowlist( $job['email_hashes'] );
				break;

			case 'recalculate_scores':
				$results = self::bulk_recalculate_scores( $job['email_hashes'] );
				break;

			case 'add_tag':
				$results = self::bulk_add_tag( $job['email_hashes'], $job['args']['tag'] ?? '' );
				break;

			case 'delete':
				$results = self::bulk_delete( $job['email_hashes'] );
				break;
		}

		// Update job with results.
		$job['status'] = 'completed';
		$job['completed_at'] = current_time( 'mysql' );
		$job['results'] = $results;
		set_transient( 'trustlens_bulk_job_' . $job_id, $job, DAY_IN_SECONDS );
	}

	/**
	 * Get bulk operation job status.
	 *
	 * @since 1.1.0
	 * @param string $job_id Job ID.
	 * @return array|false Job data or false if not found.
	 */
	public static function get_job_status( string $job_id ) {
		return get_transient( 'trustlens_bulk_job_' . $job_id );
	}

	/**
	 * Log bulk action.
	 *
	 * @since 1.1.0
	 * @param string $action       Action performed.
	 * @param array  $email_hashes Affected customers.
	 * @param array  $results      Results.
	 * @param array  $extra        Extra data.
	 */
	private static function log_bulk_action( string $action, array $email_hashes, array $results, array $extra = array() ): void {
		/**
		 * Fires after a bulk action is completed.
		 *
		 * @since 1.1.0
		 * @param string $action       Action performed.
		 * @param array  $email_hashes Affected customers.
		 * @param array  $results      Results.
		 * @param array  $extra        Extra data.
		 */
		do_action( 'trustlens/bulk_action_completed', $action, $email_hashes, $results, $extra );

		// Log to error log in debug mode.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging intentionally only runs in development mode.
			error_log( sprintf(
				'TrustLens Bulk Action: %s - %d customers, %d success, %d failed',
				$action,
				count( $email_hashes ),
				$results['success'],
				$results['failed']
			) );
		}
	}
}

// Register Action Scheduler hook.
add_action( 'trustlens/process_bulk_operation', array( 'TrustLens_Bulk_Operations', 'process_scheduled_operation' ) );
