<?php
/**
 * Analytics and reporting.
 *
 * @package TrustLens
 * @since   1.1.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Analytics class.
 *
 * Provides data for charts and reports.
 *
 * @since 1.1.0
 */
class TrustLens_Analytics {

	/**
	 * Check if a table exists in the database.
	 *
	 * @since 1.2.0
	 * @param string $table Table name (without prefix).
	 * @return bool True if table exists.
	 */
	private static function table_exists( string $table ): bool {
		global $wpdb;

		$full_table = $wpdb->prefix . $table;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check, no user input.
		$result = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $full_table ) );

		return $result === $full_table;
	}

	/**
	 * Get segment distribution.
	 *
	 * @since 1.1.0
	 * @return array Segment counts.
	 */
	public static function get_segment_distribution(): array {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Simple aggregate query, table name from $wpdb->prefix is safe.
		$results = $wpdb->get_results(
			"SELECT segment, COUNT(*) as count
			 FROM {$wpdb->prefix}trustlens_customers
			 GROUP BY segment"
		);

		$segments = array(
			'vip'      => 0,
			'trusted'  => 0,
			'normal'   => 0,
			'caution'  => 0,
			'risk'     => 0,
			'critical' => 0,
		);

		foreach ( $results as $row ) {
			if ( isset( $segments[ $row->segment ] ) ) {
				$segments[ $row->segment ] = (int) $row->count;
			}
		}

		return $segments;
	}

	/**
	 * Get trust score trends over time.
	 *
	 * @since 1.1.0
	 * @param int $days Number of days to look back.
	 * @return array Daily averages.
	 */
	public static function get_score_trends( int $days = 30 ): array {
		global $wpdb;

		// Use only score_update signals so we show actual 0-100 trust scores, not module deltas.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT
				DATE(created_at) as date,
				AVG(signal_score) as avg_score,
				COUNT(DISTINCT email_hash) as customers
			 FROM {$wpdb->prefix}trustlens_signals
			 WHERE module_id = 'score_update'
			   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
			 GROUP BY DATE(created_at)
			 ORDER BY date ASC",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		$data = array();
		foreach ( $results as $row ) {
			$data[] = array(
				'date'      => $row->date,
				'avg_score' => round( (float) $row->avg_score, 1 ),
				'customers' => (int) $row->customers,
			);
		}

		return $data;
	}

	/**
	 * Get refund trends over time.
	 *
	 * @since 1.1.0
	 * @param int $days Number of days to look back.
	 * @return array Daily refund data.
	 */
	public static function get_refund_trends( int $days = 30 ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT
				DATE(created_at) as date,
				COUNT(*) as refund_count,
				SUM(CASE WHEN event_type = 'full_refund' THEN 1 ELSE 0 END) as full_refunds,
				SUM(CASE WHEN event_type = 'partial_refund' THEN 1 ELSE 0 END) as partial_refunds
			 FROM {$wpdb->prefix}trustlens_events
			 WHERE event_type IN ('full_refund', 'partial_refund')
			   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
			 GROUP BY DATE(created_at)
			 ORDER BY date ASC",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		$data = array();
		foreach ( $results as $row ) {
			$data[] = array(
				'date'            => $row->date,
				'total'           => (int) $row->refund_count,
				'full_refunds'    => (int) $row->full_refunds,
				'partial_refunds' => (int) $row->partial_refunds,
			);
		}

		return $data;
	}

	/**
	 * Get segment changes over time.
	 *
	 * @since 1.1.0
	 * @param int $days Number of days to look back.
	 * @return array Segment change data.
	 */
	public static function get_segment_changes( int $days = 30 ): array {
		global $wpdb;

		$segment_like = $wpdb->esc_like( 'segment_' ) . '%';

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT
				DATE(created_at) as date,
				event_type,
				COUNT(*) as count
			 FROM {$wpdb->prefix}trustlens_events
			 WHERE event_type LIKE %s
			   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
			 GROUP BY DATE(created_at), event_type
			 ORDER BY date ASC",
			$segment_like,
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		return $results;
	}

	/**
	 * Get top returners.
	 *
	 * @since 1.1.0
	 * @param int $limit Number of customers to return.
	 * @return array Top returners.
	 */
	public static function get_top_returners( int $limit = 10 ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		return $wpdb->get_results( $wpdb->prepare(
			"SELECT
				email_hash,
				customer_email,
				total_orders,
				total_refunds,
				return_rate,
				total_refund_value,
				segment,
				trust_score
			 FROM {$wpdb->prefix}trustlens_customers
			 WHERE total_refunds > 0
			 ORDER BY return_rate DESC, total_refund_value DESC
			 LIMIT %d",
			$limit
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
	}

	/**
	 * Get category abuse stats.
	 *
	 * @since 1.1.0
	 * @param int $limit Number of categories to return.
	 * @return array Category stats.
	 */
	public static function get_category_abuse_stats( int $limit = 10 ): array {
		global $wpdb;

		$table = $wpdb->prefix . 'trustlens_category_stats';

		// Check if table exists.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SHOW TABLES check, no user input.
		$table_exists = $wpdb->get_var( $wpdb->prepare(
			"SHOW TABLES LIKE %s",
			$table
		) );

		if ( ! $table_exists ) {
			return array();
		}

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, prepared query.
		return $wpdb->get_results( $wpdb->prepare(
			"SELECT
				category_slug,
				SUM(order_count) as total_orders,
				SUM(refund_count) as total_refunds,
				SUM(order_value) as total_order_value,
				SUM(refund_value) as total_refund_value,
				CASE WHEN SUM(order_count) > 0
					THEN ROUND((SUM(refund_count) / SUM(order_count)) * 100, 1)
					ELSE 0
				END as return_rate
			 FROM {$table}
			 GROUP BY category_slug
			 HAVING total_refunds > 0
			 ORDER BY return_rate DESC
			 LIMIT %d",
			$limit
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
	}

	/**
	 * Get event activity by hour.
	 *
	 * @since 1.1.0
	 * @param int $days Number of days to analyze.
	 * @return array Hourly activity.
	 */
	public static function get_hourly_activity( int $days = 7 ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT
				HOUR(created_at) as hour,
				COUNT(*) as event_count
			 FROM {$wpdb->prefix}trustlens_events
			 WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
			 GROUP BY HOUR(created_at)
			 ORDER BY hour ASC",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Fill in missing hours.
		$hourly = array_fill( 0, 24, 0 );
		foreach ( $results as $row ) {
			$hourly[ (int) $row->hour ] = (int) $row->event_count;
		}

		return $hourly;
	}

	/**
	 * Get summary statistics.
	 *
	 * @since 1.1.0
	 * @return array Summary stats.
	 */
	public static function get_summary_stats(): array {
		global $wpdb;

		$customers_table = $wpdb->prefix . 'trustlens_customers';
		$events_table = $wpdb->prefix . 'trustlens_events';

		// Customer stats.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Aggregate stats query, table name from $wpdb->prefix is safe.
		$customer_stats = $wpdb->get_row(
			"SELECT
				COUNT(*) as total_customers,
				AVG(trust_score) as avg_trust_score,
				AVG(return_rate) as avg_return_rate,
				SUM(is_blocked) as blocked_count,
				SUM(is_allowlisted) as allowlisted_count,
				SUM(total_orders) as total_orders,
				SUM(total_refunds) as total_refunds,
				SUM(total_order_value) as total_order_value,
				SUM(total_refund_value) as total_refund_value
			 FROM {$customers_table}"
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Recent activity (last 24 hours).
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, prepared query.
		$recent_activity = $wpdb->get_row( $wpdb->prepare(
			"SELECT
				COUNT(*) as events_24h,
				COUNT(DISTINCT email_hash) as active_customers_24h
			 FROM {$events_table}
			 WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d HOUR)",
			24
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// New high-risk customers this week.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, prepared query.
		$new_high_risk = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*)
			 FROM {$customers_table}
			 WHERE segment IN ('risk', 'critical')
			   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
			7
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		return array(
			'total_customers'       => (int) $customer_stats->total_customers,
			'avg_trust_score'       => round( (float) $customer_stats->avg_trust_score, 1 ),
			'avg_return_rate'       => round( (float) $customer_stats->avg_return_rate, 1 ),
			'blocked_count'         => (int) $customer_stats->blocked_count,
			'allowlisted_count'     => (int) $customer_stats->allowlisted_count,
			'total_orders'          => (int) $customer_stats->total_orders,
			'total_refunds'         => (int) $customer_stats->total_refunds,
			'total_order_value'     => (float) $customer_stats->total_order_value,
			'total_refund_value'    => (float) $customer_stats->total_refund_value,
			'events_24h'            => (int) $recent_activity->events_24h,
			'active_customers_24h'  => (int) $recent_activity->active_customers_24h,
			'new_high_risk_week'    => (int) $new_high_risk,
		);
	}

	/**
	 * Get data for dashboard charts (AJAX endpoint).
	 *
	 * @since 1.1.0
	 * @param string $chart_type Chart type.
	 * @param int    $days       Days to look back.
	 * @return array Chart data.
	 */
	public static function get_chart_data( string $chart_type, int $days = 30 ): array {
		switch ( $chart_type ) {
			case 'segments':
				return self::get_segment_distribution();

			case 'refunds':
				return self::get_refund_trends( $days );

			case 'scores':
				return self::get_score_trends( $days );

			case 'categories':
				return self::get_category_abuse_stats();

			case 'hourly':
				return self::get_hourly_activity( $days );

			case 'roi':
				return self::get_roi_data( $days );

			default:
				return array();
		}
	}

	/**
	 * Get ROI calculator data (Pro feature).
	 *
	 * Calculates money saved and at risk from TrustLens actions.
	 *
	 * @since 1.2.0
	 * @param int $days Number of days to analyze.
	 * @return array ROI data.
	 */
	public static function get_roi_data( int $days = 30 ): array {
		global $wpdb;

		$customers_table = $wpdb->prefix . 'trustlens_customers';
		$events_table = $wpdb->prefix . 'trustlens_events';
		$automation_logs = $wpdb->prefix . 'trustlens_automation_logs';

		// Money protected: Orders held for review that weren't refunded.
		$orders_held_value = self::calculate_orders_held_value( $days );

		// Money protected: Checkout blocks that prevented high-risk orders.
		$blocked_checkout_value = self::calculate_blocked_checkout_value( $days );

		// Money protected: Chargebacks prevented (estimated from flagged customers).
		$chargeback_prevention = self::estimate_chargeback_prevention( $days );

		// Money at risk: Unflagged orders from high-risk customers.
		$money_at_risk = self::calculate_money_at_risk( $days );

		// Total refund value from high-risk customers (actual losses).
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, prepared query.
		$high_risk_refunds = $wpdb->get_var( $wpdb->prepare(
			"SELECT SUM(c.total_refund_value)
			 FROM {$customers_table} c
			 WHERE c.segment IN ('risk', 'critical')
			   AND c.last_refund_date >= DATE_SUB(NOW(), INTERVAL %d DAY)",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Actions taken (check if table exists).
		$action_counts = array();
		if ( self::table_exists( 'trustlens_automation_logs' ) ) {
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, prepared query.
			$actions_taken = $wpdb->get_results( $wpdb->prepare(
				"SELECT action_type, COUNT(*) as count
				 FROM {$automation_logs}
				 WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
				 GROUP BY action_type",
				$days
			) );
			// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

			foreach ( $actions_taken as $action ) {
				$action_counts[ $action->action_type ] = (int) $action->count;
			}
		}

		// Calculate ROI percentage.
		$total_protected = $orders_held_value + $blocked_checkout_value + $chargeback_prevention;
		$total_at_risk = $money_at_risk + (float) $high_risk_refunds;
		$roi_percentage = $total_at_risk > 0 ? ( $total_protected / $total_at_risk ) * 100 : 0;

		return array(
			'period_days'              => $days,
			'money_protected'          => array(
				'orders_held'          => round( $orders_held_value, 2 ),
				'blocked_checkouts'    => round( $blocked_checkout_value, 2 ),
				'chargeback_prevention'=> round( $chargeback_prevention, 2 ),
				'total'                => round( $total_protected, 2 ),
			),
			'money_at_risk'            => array(
				'unflagged_orders'     => round( $money_at_risk, 2 ),
				'high_risk_refunds'    => round( (float) $high_risk_refunds, 2 ),
				'total'                => round( $total_at_risk, 2 ),
			),
			'roi_percentage'           => round( $roi_percentage, 1 ),
			'actions_taken'            => $action_counts,
			'summary'                  => array(
				'orders_held'          => $action_counts['hold_order'] ?? 0,
				'checkouts_blocked'    => $action_counts['block_checkout'] ?? 0,
				'accounts_required'    => $action_counts['require_account'] ?? 0,
				'customers_blocked'    => $action_counts['block_customer'] ?? 0,
			),
		);
	}

	/**
	 * Calculate value of orders held for review.
	 *
	 * @since 1.2.0
	 * @param int $days Days to analyze.
	 * @return float Value.
	 */
	private static function calculate_orders_held_value( int $days ): float {
		global $wpdb;

		// Get orders held by TrustLens that weren't fully refunded.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$held_value = $wpdb->get_var( $wpdb->prepare(
			"SELECT SUM(CAST(JSON_UNQUOTE(JSON_EXTRACT(event_data, '$.order_total')) AS DECIMAL(12,2)))
			 FROM {$wpdb->prefix}trustlens_events
			 WHERE event_type = 'order_held'
			   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Alternative: Check automation logs if table exists.
		if ( ! $held_value && self::table_exists( 'trustlens_automation_logs' ) ) {
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
			$held_orders = $wpdb->get_col( $wpdb->prepare(
				"SELECT DISTINCT order_id
				 FROM {$wpdb->prefix}trustlens_automation_logs
				 WHERE action_type = 'hold_order'
				   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
				$days
			) );
			// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

			$total = 0.0;
			foreach ( $held_orders as $order_id ) {
				$order = wc_get_order( $order_id );
				if ( $order && ! $order->has_status( array( 'refunded', 'cancelled' ) ) ) {
					$total += (float) $order->get_total();
				}
			}
			$held_value = $total;
		}

		return (float) $held_value;
	}

	/**
	 * Calculate estimated value of blocked checkouts.
	 *
	 * @since 1.2.0
	 * @param int $days Days to analyze.
	 * @return float Value.
	 */
	private static function calculate_blocked_checkout_value( int $days ): float {
		global $wpdb;

		// Count blocked checkouts.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$blocked_count = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*)
			 FROM {$wpdb->prefix}trustlens_events
			 WHERE event_type = 'checkout_blocked'
			   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Estimate value based on average order value of blocked customers.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Aggregate query, table name from $wpdb->prefix is safe.
		$avg_order = $wpdb->get_var(
			"SELECT AVG(total_order_value / NULLIF(total_orders, 0))
			 FROM {$wpdb->prefix}trustlens_customers
			 WHERE is_blocked = 1
			   AND total_orders > 0"
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		// Use store average if no blocked customer data.
		if ( ! $avg_order ) {
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Aggregate query, table name from $wpdb->prefix is safe.
			$avg_order = $wpdb->get_var(
				"SELECT AVG(total_order_value / NULLIF(total_orders, 0))
				 FROM {$wpdb->prefix}trustlens_customers
				 WHERE segment IN ('risk', 'critical')
				   AND total_orders > 0"
			);
			// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}

		// Default to $50 if no data.
		$avg_order = $avg_order ?: 50;

		// Estimate: blocked checkouts × avg order value × estimated fraud rate (60%).
		return (int) $blocked_count * (float) $avg_order * 0.6;
	}

	/**
	 * Estimate chargeback prevention value.
	 *
	 * @since 1.2.0
	 * @param int $days Days to analyze.
	 * @return float Estimated value.
	 */
	private static function estimate_chargeback_prevention( int $days ): float {
		global $wpdb;

		// Get customers with disputes who were flagged before dispute.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$flagged_disputes = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(DISTINCT e.email_hash)
			 FROM {$wpdb->prefix}trustlens_events e
			 JOIN {$wpdb->prefix}trustlens_customers c ON e.email_hash = c.email_hash
			 WHERE e.event_type = 'dispute_created'
			   AND e.created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
			   AND c.segment IN ('risk', 'critical')",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Get average dispute value.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$avg_dispute = $wpdb->get_var( $wpdb->prepare(
			"SELECT AVG(CAST(JSON_UNQUOTE(JSON_EXTRACT(event_data, '$.amount')) AS DECIMAL(12,2)))
			 FROM {$wpdb->prefix}trustlens_events
			 WHERE event_type = 'dispute_created'
			   AND created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		$avg_dispute = $avg_dispute ?: 100;

		// Estimate: flagged customers who had disputes × chargeback fee savings.
		// Industry average chargeback fee is ~$25 + the order value.
		return (int) $flagged_disputes * ( (float) $avg_dispute + 25 );
	}

	/**
	 * Calculate money at risk from unflagged high-risk orders.
	 *
	 * @since 1.2.0
	 * @param int $days Days to analyze.
	 * @return float Value at risk.
	 */
	private static function calculate_money_at_risk( int $days ): float {
		global $wpdb;

		// Get recent order value from high-risk customers who weren't blocked.
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$at_risk = $wpdb->get_var( $wpdb->prepare(
			"SELECT SUM(CAST(JSON_UNQUOTE(JSON_EXTRACT(e.event_data, '$.order_total')) AS DECIMAL(12,2)))
			 FROM {$wpdb->prefix}trustlens_events e
			 JOIN {$wpdb->prefix}trustlens_customers c ON e.email_hash = c.email_hash
			 WHERE e.event_type = 'order_created'
			   AND e.created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
			   AND c.segment IN ('caution', 'risk', 'critical')
			   AND c.is_blocked = 0",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		// Apply risk factor based on segment distribution.
		// Caution: 10% risk, Risk: 30% risk, Critical: 60% risk.
		if ( ! $at_risk ) {
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
			$at_risk = $wpdb->get_var( $wpdb->prepare(
				"SELECT SUM(
					CASE c.segment
						WHEN 'caution' THEN (c.total_order_value / NULLIF(c.total_orders, 0)) * 0.1
						WHEN 'risk' THEN (c.total_order_value / NULLIF(c.total_orders, 0)) * 0.3
						WHEN 'critical' THEN (c.total_order_value / NULLIF(c.total_orders, 0)) * 0.6
						ELSE 0
					END
				)
				 FROM {$wpdb->prefix}trustlens_customers c
				 WHERE c.segment IN ('caution', 'risk', 'critical')
				   AND c.is_blocked = 0
				   AND c.last_order_date >= DATE_SUB(NOW(), INTERVAL %d DAY)",
				$days
			) );
			// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
		}

		return (float) $at_risk;
	}

	/**
	 * Get ROI trends over time.
	 *
	 * @since 1.2.0
	 * @param int $days Days to analyze.
	 * @return array Daily ROI data.
	 */
	public static function get_roi_trends( int $days = 30 ): array {
		global $wpdb;

		// Check if table exists.
		if ( ! self::table_exists( 'trustlens_automation_logs' ) ) {
			return array();
		}

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT
				DATE(created_at) as date,
				SUM(CASE WHEN action_type = 'hold_order' THEN 1 ELSE 0 END) as orders_held,
				SUM(CASE WHEN action_type = 'block_checkout' THEN 1 ELSE 0 END) as checkouts_blocked,
				SUM(CASE WHEN action_type = 'require_account' THEN 1 ELSE 0 END) as accounts_required
			 FROM {$wpdb->prefix}trustlens_automation_logs
			 WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
			 GROUP BY DATE(created_at)
			 ORDER BY date ASC",
			$days
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		$data = array();
		foreach ( $results as $row ) {
			$data[] = array(
				'date'              => $row->date,
				'orders_held'       => (int) $row->orders_held,
				'checkouts_blocked' => (int) $row->checkouts_blocked,
				'accounts_required' => (int) $row->accounts_required,
			);
		}

		return $data;
	}

	/**
	 * Get monthly ROI summary for reporting.
	 *
	 * @since 1.2.0
	 * @param int $months Number of months to include.
	 * @return array Monthly summaries.
	 */
	public static function get_monthly_roi_summary( int $months = 6 ): array {
		$summaries = array();

		for ( $i = 0; $i < $months; $i++ ) {
			$start = strtotime( "-{$i} months", strtotime( 'first day of this month' ) );
			$end = strtotime( '+1 month', $start ) - 1;

			$days_in_month = (int) gmdate( 't', $start );
			$roi_data = self::get_roi_data( $days_in_month );

			$summaries[] = array(
				'month'           => gmdate( 'Y-m', $start ),
				'month_label'     => gmdate( 'F Y', $start ),
				'money_protected' => $roi_data['money_protected']['total'],
				'money_at_risk'   => $roi_data['money_at_risk']['total'],
				'roi_percentage'  => $roi_data['roi_percentage'],
				'actions_taken'   => array_sum( $roi_data['summary'] ),
			);
		}

		return array_reverse( $summaries );
	}
}
