<?php
/**
 * Dashboard page template.
 *
 * @package TrustLens\Admin\Views
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variables scoped to included view file.

$sync        = TrustLens_Historical_Sync::instance();
$sync_status = $sync->get_status();
$sync_progress = $sync->get_progress();

// Analytics data.
$segment_counts  = TrustLens_Analytics::get_segment_distribution();
$trends          = TrustLens_Analytics::get_score_trends( 30 );
$refund_trends   = TrustLens_Analytics::get_refund_trends( 30 );
$roi_data        = TrustLens_Analytics::get_roi_data( 30 );
$summary_stats   = TrustLens_Analytics::get_summary_stats();
$hourly_activity = TrustLens_Analytics::get_hourly_activity( 7 );
$top_returners   = TrustLens_Analytics::get_top_returners( 5 );
$category_abuse  = TrustLens_Analytics::get_category_abuse_stats( 5 );
$monthly_roi     = TrustLens_Analytics::get_monthly_roi_summary( 6 );

// Customer repository data.
$repo                = new TrustLens_Customer_Repository();
$high_risk_customers = $repo->get_high_risk( 10 );

// Module data.
$chargeback_stats = wstl_has_chargebacks() ? TrustLens_Module_Chargebacks::get_dispute_stats() : array();
$coupon_stats     = TrustLens_Module_Coupons::get_coupon_abuse_stats();

// Computed values.
$avg_score       = (int) round( $summary_stats['avg_trust_score'] );
$health_segment  = wstl_get_segment_from_score( $avg_score );
$total_customers = array_sum( $segment_counts );
$risk_count      = ( $segment_counts['risk'] ?? 0 ) + ( $segment_counts['critical'] ?? 0 );
$is_empty        = ( 0 === $total_customers );

// Chart data availability flags for metric empty states.
$has_trends_data = ! empty( $trends );
$has_refund_data = ! empty( $refund_trends );
$has_hourly_data = array_sum( array_map( 'intval', $hourly_activity ) ) > 0;
$has_monthly_roi_data = false;
foreach ( $monthly_roi as $monthly_point ) {
	if (
		( (float) ( $monthly_point['money_protected'] ?? 0 ) ) > 0 ||
		( (float) ( $monthly_point['money_at_risk'] ?? 0 ) ) > 0 ||
		( (int) ( $monthly_point['actions_taken'] ?? 0 ) ) > 0
	) {
		$has_monthly_roi_data = true;
		break;
	}
}

// Count available WooCommerce orders (only when dashboard is empty, to avoid unnecessary queries).
$wc_order_count = 0;
if ( $is_empty ) {
	$wc_order_count = count( wc_get_orders( array(
		'limit'  => -1,
		'return' => 'ids',
		'status' => array( 'completed', 'processing', 'refunded', 'cancelled' ),
	) ) );
}

// ROI.
$free_protected  = ( $roi_data['money_protected']['orders_held'] ?? 0 ) + ( $roi_data['money_protected']['blocked_checkouts'] ?? 0 );
$total_protected = wstl_has_chargebacks() ? ( $roi_data['money_protected']['total'] ?? 0 ) : $free_protected;
$total_at_risk   = $roi_data['money_at_risk']['total'] ?? 0;
$actions_taken   = array_sum( $roi_data['summary'] ?? array() );

// Time-based greeting.
$current_hour = (int) current_time( 'G' );
if ( $current_hour < 12 ) {
	$greeting = __( 'Good morning', 'trustlens' );
} elseif ( $current_hour < 18 ) {
	$greeting = __( 'Good afternoon', 'trustlens' );
} else {
	$greeting = __( 'Good evening', 'trustlens' );
}

// Pass chart data to the enqueued dashboard-charts.js.
$wstl_segments         = array( 'vip', 'trusted', 'normal', 'caution', 'risk', 'critical' );
$wstl_segment_labels   = array();
$wstl_segment_colors   = array();
foreach ( $wstl_segments as $seg ) {
	$wstl_segment_labels[ $seg ] = wstl_get_segment_label( $seg );
	$wstl_segment_colors[ $seg ] = wstl_get_segment_color( $seg );
}

wp_localize_script( 'trustlens-dashboard-charts', 'TrustLensDashboard', array(
	'trends'         => $trends,
	'segmentCounts'  => $segment_counts,
	'segmentLabels'  => $wstl_segment_labels,
	'segmentColors'  => $wstl_segment_colors,
	'refundTrends'   => $refund_trends,
	'hourlyActivity' => array_values( $hourly_activity ),
	'categoryAbuse'  => $category_abuse,
	'monthlyRoi'     => $monthly_roi,
	'i18n'           => array(
		'customers'         => __( 'Customers', 'trustlens' ),
		'averageScore'      => __( 'Average Score', 'trustlens' ),
		'threshold'         => __( 'Threshold', 'trustlens' ),
		'customersScored'   => __( 'customers scored', 'trustlens' ),
		'fullRefunds'       => __( 'Full Refunds', 'trustlens' ),
		'partialRefunds'    => __( 'Partial Refunds', 'trustlens' ),
		'events'            => __( 'Events', 'trustlens' ),
		'returnRatePercent' => __( 'Return Rate %', 'trustlens' ),
		'protected'         => __( 'Protected', 'trustlens' ),
		'atRisk'            => __( 'At Risk', 'trustlens' ),
	),
) );
?>

<div class="wrap wstl-wrap">
	<h1 class="wp-heading-inline"><?php esc_html_e( 'TrustLens Dashboard', 'trustlens' ); ?></h1>
	<hr class="wp-header-end">

	<div class="wstl-dashboard">

		<?php if ( $is_empty ) : ?>
		<!-- S1: Header Bar (Empty State) -->
		<div class="wstl-dashboard-section">
			<div class="wstl-header-bar">
				<div class="wstl-header-health">
					<div class="wstl-health-ring">
						<span class="wstl-health-value">&mdash;</span>
						<span class="wstl-health-label"><?php esc_html_e( 'Health', 'trustlens' ); ?></span>
					</div>
				</div>
				<div class="wstl-header-info">
					<p class="wstl-header-greeting"><?php echo esc_html( $greeting ); ?></p>
					<p class="wstl-header-subtitle">
						<?php if ( $wc_order_count > 0 ) : ?>
							<?php
							printf(
								/* translators: %s: order count */
								esc_html__( 'You have %s orders ready to be analyzed.', 'trustlens' ),
								'<strong>' . esc_html( number_format( $wc_order_count ) ) . '</strong>'
							);
							?>
						<?php else : ?>
							<?php esc_html_e( 'New orders will be analyzed automatically as they come in.', 'trustlens' ); ?>
						<?php endif; ?>
					</p>
					<p class="wstl-header-health-msg is-setup">
						<?php
						if ( $wc_order_count > 0 ) {
							esc_html_e( 'Run a historical sync to build customer trust profiles', 'trustlens' );
						} else {
							esc_html_e( 'Awaiting your first WooCommerce orders', 'trustlens' );
						}
						?>
					</p>
					<div class="wstl-header-metrics">
						<span class="wstl-header-chip"><?php echo esc_html( number_format( $wc_order_count ) ); ?> <?php esc_html_e( 'Orders Ready', 'trustlens' ); ?></span>
						<span class="wstl-header-chip"><?php echo esc_html( $sync_progress ); ?>% <?php esc_html_e( 'Sync Progress', 'trustlens' ); ?></span>
					</div>
				</div>
			</div>
		</div>

		<!-- Onboarding Card -->
		<div class="wstl-dashboard-section">
			<div class="wstl-onboarding-card">
				<div class="wstl-onboarding-icon">
					<span class="dashicons dashicons-chart-area"></span>
				</div>
				<h2><?php esc_html_e( 'Get Started with TrustLens', 'trustlens' ); ?></h2>
				<p class="wstl-onboarding-desc">
					<?php esc_html_e( 'Follow these steps to start monitoring customer trust across your store.', 'trustlens' ); ?>
				</p>
				<div class="wstl-onboarding-steps">
					<div class="wstl-onboarding-step">
						<div class="wstl-step-header">
							<span class="wstl-onboarding-step-num">1</span>
							<span class="dashicons dashicons-database-import wstl-step-icon"></span>
						</div>
						<h3><?php esc_html_e( 'Import Your Orders', 'trustlens' ); ?></h3>
						<p>
							<?php if ( $wc_order_count > 0 ) : ?>
								<?php
								printf(
									/* translators: %s: order count */
									esc_html__( '%s orders are waiting to be analyzed. Hit sync below and watch TrustLens build a trust profile for every customer.', 'trustlens' ),
									'<strong>' . esc_html( number_format( $wc_order_count ) ) . '</strong>'
								);
								?>
							<?php else : ?>
								<?php esc_html_e( 'As soon as WooCommerce orders come in, TrustLens will build a trust profile for every customer automatically.', 'trustlens' ); ?>
							<?php endif; ?>
						</p>
					</div>
					<div class="wstl-onboarding-step">
						<div class="wstl-step-header">
							<span class="wstl-onboarding-step-num">2</span>
							<span class="dashicons dashicons-admin-settings wstl-step-icon"></span>
						</div>
						<h3><?php esc_html_e( 'Customize Your Rules', 'trustlens' ); ?></h3>
						<p>
							<?php
							printf(
								/* translators: %1$s: opening link tag, %2$s: closing link tag */
								esc_html__( 'Adjust trust thresholds, turn on checkout blocking, and fine-tune detection modules in %1$sSettings%2$s.', 'trustlens' ),
								'<a href="' . esc_url( admin_url( 'admin.php?page=trustlens-settings' ) ) . '">',
								'</a>'
							);
							?>
						</p>
					</div>
					<div class="wstl-onboarding-step">
						<div class="wstl-step-header">
							<span class="wstl-onboarding-step-num">3</span>
							<span class="dashicons dashicons-shield wstl-step-icon"></span>
						</div>
						<h3><?php esc_html_e( 'Protect Your Store', 'trustlens' ); ?></h3>
						<p><?php esc_html_e( 'Your dashboard will light up with trust scores, segment trends, and flagged customers the moment the sync finishes.', 'trustlens' ); ?></p>
					</div>
				</div>
				<?php if ( wstl_show_pro_upsells() && wstl_get_upgrade_or_fallback_url() ) : ?>
				<p class="wstl-onboarding-pro-teaser">
					<?php
					printf(
						/* translators: %1$s: opening link tag, %2$s: closing link tag */
						esc_html__( 'Want to automate responses and get reports by email? %1$sSee Pro%2$s.', 'trustlens' ),
						'<a href="' . esc_url( wstl_get_upgrade_or_fallback_url() ) . '">',
						'</a>'
					);
					?>
				</p>
				<?php endif; ?>

				<!-- Sync Action -->
				<div class="wstl-onboarding-sync">
					<div id="wstl-sync-start-area" class="<?php echo 'running' === $sync_status['status'] ? 'wstl-modal-hidden' : ''; ?>">
						<button type="button" class="button button-primary button-hero" id="wstl-start-sync">
							<?php esc_html_e( 'Start Historical Sync', 'trustlens' ); ?>
						</button>
						<p class="wstl-onboarding-sync-note">
							<?php
							if ( $wc_order_count > 0 ) {
								esc_html_e( 'You\'ll see progress in real time — keep this tab open.', 'trustlens' );
							} else {
								esc_html_e( 'No eligible historical orders found yet. New WooCommerce orders are analyzed automatically as they come in.', 'trustlens' );
							}
							?>
						</p>
					</div>
					<div class="wstl-sync-progress <?php echo 'running' !== $sync_status['status'] ? 'wstl-modal-hidden' : ''; ?>" id="wstl-sync-progress-area">
						<div class="wstl-progress-bar">
							<div class="wstl-progress-fill" style="width: <?php echo esc_attr( $sync_progress ); ?>%;"></div>
						</div>
						<p class="wstl-sync-stats">
							<span id="wstl-sync-progress-text"><?php echo esc_html( $sync_progress ); ?>%</span> &mdash;
							<span id="wstl-sync-orders-text">
								<?php
								if ( 'running' === $sync_status['status'] ) {
									printf(
										/* translators: %1$s: processed orders, %2$s: total orders */
										esc_html__( '%1$s of %2$s orders processed', 'trustlens' ),
										number_format( $sync_status['processed_orders'] ),
										number_format( $sync_status['total_orders'] )
									);
								} else {
									printf(
										/* translators: %1$s: processed orders, %2$s: total orders */
										esc_html__( '%1$s of %2$s orders processed', 'trustlens' ),
										'0',
										number_format( $wc_order_count )
									);
								}
								?>
							</span>
						</p>
						<button type="button" class="button" id="wstl-stop-sync" <?php echo 'running' === $sync_status['status'] ? '' : 'style="display:none;"'; ?>>
							<?php esc_html_e( 'Stop Sync', 'trustlens' ); ?>
						</button>
					</div>
				</div>
			</div>
		</div>

		<?php else : ?>
		<!-- S1: Header Bar -->
		<div class="wstl-dashboard-section">
			<div class="wstl-header-bar">
				<div class="wstl-header-health">
					<div class="wstl-health-ring wstl-health-ring-<?php echo esc_attr( $health_segment ); ?>">
						<span class="wstl-health-value"><?php echo esc_html( $avg_score ); ?></span>
						<span class="wstl-health-label"><?php esc_html_e( 'Health', 'trustlens' ); ?></span>
					</div>
				</div>
				<div class="wstl-header-info">
					<p class="wstl-header-greeting"><?php echo esc_html( $greeting ); ?></p>
					<?php
					$health_class = 'is-good';
					$health_msg   = __( 'Your store is in great shape', 'trustlens' );
					if ( $risk_count > 0 && $avg_score < 40 ) {
						$health_class = 'is-danger';
						$health_msg   = __( 'Multiple risks detected — review flagged customers', 'trustlens' );
					} elseif ( $risk_count > 0 ) {
						$health_class = 'is-caution';
						$health_msg   = __( 'Some customers need attention', 'trustlens' );
					}
					?>
					<p class="wstl-header-health-msg <?php echo esc_attr( $health_class ); ?>">
						<?php echo esc_html( $health_msg ); ?>
					</p>
					<p class="wstl-header-subtitle">
						<?php
						printf(
							/* translators: %1$s: event count, %2$s: customer count */
							esc_html__( 'Last 24 hours: %1$s events from %2$s customers', 'trustlens' ),
							'<strong>' . esc_html( number_format( $summary_stats['events_24h'] ) ) . '</strong>',
							'<strong>' . esc_html( number_format( $summary_stats['active_customers_24h'] ) ) . '</strong>'
						);
						?>
					</p>
					<div class="wstl-header-metrics">
						<span class="wstl-header-chip"><?php echo esc_html( number_format( $summary_stats['total_customers'] ) ); ?> <?php esc_html_e( 'Customers', 'trustlens' ); ?></span>
						<span class="wstl-header-chip"><?php echo esc_html( number_format( $risk_count ) ); ?> <?php esc_html_e( 'Need Attention', 'trustlens' ); ?></span>
						<span class="wstl-header-chip"><?php echo esc_html( number_format( $summary_stats['events_24h'] ) ); ?> <?php esc_html_e( 'Events 24h', 'trustlens' ); ?></span>
					</div>
				</div>
				<div class="wstl-header-actions">
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers' ) ); ?>" class="button">
						<?php esc_html_e( 'View All Customers', 'trustlens' ); ?>
					</a>
				</div>
			</div>
		</div>

		<!-- S2: KPI Cards -->
		<div class="wstl-dashboard-section">
			<div class="wstl-kpi-grid">
				<div class="wstl-kpi-card">
					<div class="wstl-kpi-header">
						<span class="dashicons dashicons-groups wstl-kpi-icon"></span>
						<span class="wstl-kpi-label"><?php esc_html_e( 'Total Customers', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-kpi-value"><?php echo esc_html( number_format( $summary_stats['total_customers'] ) ); ?></div>
				</div>

				<div class="wstl-kpi-card wstl-kpi-card-<?php echo $avg_score >= 70 ? 'green' : ( $avg_score >= 40 ? 'amber' : 'red' ); ?>">
					<div class="wstl-kpi-header">
						<span class="dashicons dashicons-shield wstl-kpi-icon"></span>
						<span class="wstl-kpi-label"><?php esc_html_e( 'Avg Trust Score', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-kpi-value"><?php echo esc_html( $avg_score ); ?><span class="wstl-score-suffix">/100</span></div>
				</div>

				<div class="wstl-kpi-card <?php echo $summary_stats['new_high_risk_week'] > 0 ? 'wstl-kpi-card-red' : ''; ?>">
					<div class="wstl-kpi-header">
						<span class="dashicons dashicons-warning wstl-kpi-icon"></span>
						<span class="wstl-kpi-label"><?php esc_html_e( 'New High-Risk (7d)', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-kpi-value"><?php echo esc_html( number_format( $summary_stats['new_high_risk_week'] ) ); ?></div>
				</div>

				<div class="wstl-kpi-card wstl-kpi-card-blue">
					<div class="wstl-kpi-header">
						<span class="dashicons dashicons-clock wstl-kpi-icon"></span>
						<span class="wstl-kpi-label"><?php esc_html_e( 'Events (24h)', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-kpi-value"><?php echo esc_html( number_format( $summary_stats['events_24h'] ) ); ?></div>
				</div>

				<div class="wstl-kpi-card">
					<div class="wstl-kpi-header">
						<span class="dashicons dashicons-cart wstl-kpi-icon"></span>
						<span class="wstl-kpi-label"><?php esc_html_e( 'Total Orders', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-kpi-value"><?php echo esc_html( number_format( $summary_stats['total_orders'] ) ); ?></div>
				</div>

				<div class="wstl-kpi-card <?php echo $summary_stats['avg_return_rate'] > 15 ? 'wstl-kpi-card-amber' : ''; ?>">
					<div class="wstl-kpi-header">
						<span class="dashicons dashicons-undo wstl-kpi-icon"></span>
						<span class="wstl-kpi-label"><?php esc_html_e( 'Return Rate', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-kpi-value"><?php echo esc_html( number_format( $summary_stats['avg_return_rate'], 1 ) ); ?>%</div>
				</div>
			</div>
		</div>

		<!-- S3: Primary Charts -->
		<div class="wstl-dashboard-section">
			<div class="wstl-dashboard-grid-2">
				<div class="wstl-card wstl-chart-card">
					<div class="wstl-chart-header">
						<h2><?php esc_html_e( 'Trust Score Trends', 'trustlens' ); ?></h2>
						<span class="wstl-chart-period"><?php esc_html_e( 'Last 30 days', 'trustlens' ); ?></span>
					</div>
					<?php if ( $has_trends_data ) : ?>
						<div class="wstl-chart-container wstl-chart-height-lg">
							<canvas id="wstl-trends-chart"></canvas>
						</div>
					<?php else : ?>
						<div class="wstl-chart-empty-state wstl-chart-height-lg">
							<span class="dashicons dashicons-chart-line"></span>
							<h3><?php esc_html_e( 'No trend data yet', 'trustlens' ); ?></h3>
							<p><?php esc_html_e( 'Trust score trends will appear after score recalculations are recorded over time.', 'trustlens' ); ?></p>
						</div>
					<?php endif; ?>
				</div>

				<div class="wstl-card wstl-chart-card">
					<div class="wstl-chart-header">
						<h2><?php esc_html_e( 'Customer Segments', 'trustlens' ); ?></h2>
						<span class="wstl-chart-period"><?php echo esc_html( number_format( $total_customers ) ); ?> <?php esc_html_e( 'total', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-chart-container wstl-chart-height-lg">
						<canvas id="wstl-segment-chart"></canvas>
					</div>
				</div>
			</div>
		</div>

		<!-- S4: ROI Scorecard -->
		<div class="wstl-dashboard-section">
			<div class="wstl-roi-scorecard">
				<h2>
					<span class="dashicons dashicons-shield wstl-icon-title wstl-icon-title-success"></span>
					<?php esc_html_e( 'Revenue Protection Overview', 'trustlens' ); ?>
					<span class="wstl-inline-period"><?php esc_html_e( 'Last 30 days', 'trustlens' ); ?></span>
				</h2>
				<div class="wstl-roi-scorecard-grid">
					<div class="wstl-roi-scorecard-stat wstl-roi-scorecard-stat-protected">
						<span class="wstl-roi-scorecard-value"><?php echo wp_kses_post( wc_price( $total_protected ) ); ?></span>
						<span class="wstl-roi-scorecard-label"><?php esc_html_e( 'Money Protected', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-roi-scorecard-stat wstl-roi-scorecard-stat-risk">
						<span class="wstl-roi-scorecard-value"><?php echo wp_kses_post( wc_price( $total_at_risk ) ); ?></span>
						<span class="wstl-roi-scorecard-label"><?php esc_html_e( 'Money At Risk', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-roi-scorecard-stat wstl-roi-scorecard-stat-rate">
						<span class="wstl-roi-scorecard-value"><?php echo esc_html( $roi_data['roi_percentage'] ?? 0 ); ?>%</span>
						<span class="wstl-roi-scorecard-label"><?php esc_html_e( 'Protection Rate', 'trustlens' ); ?></span>
					</div>
					<div class="wstl-roi-scorecard-stat wstl-roi-scorecard-stat-actions">
						<span class="wstl-roi-scorecard-value"><?php echo esc_html( number_format( $actions_taken ) ); ?></span>
						<span class="wstl-roi-scorecard-label"><?php esc_html_e( 'Actions Taken', 'trustlens' ); ?></span>
					</div>
				</div>
				<div class="wstl-roi-detail-row">
					<div class="wstl-roi-detail">
						<?php esc_html_e( 'Orders Held:', 'trustlens' ); ?>
						<strong><?php echo esc_html( $roi_data['summary']['orders_held'] ?? 0 ); ?></strong>
					</div>
					<div class="wstl-roi-detail">
						<?php esc_html_e( 'Checkouts Blocked:', 'trustlens' ); ?>
						<strong><?php echo esc_html( $roi_data['summary']['checkouts_blocked'] ?? 0 ); ?></strong>
					</div>
					<?php if ( wstl_has_chargebacks() ) : ?>
						<div class="wstl-roi-detail">
							<?php esc_html_e( 'Chargeback Prevention:', 'trustlens' ); ?>
							<strong><?php echo wp_kses_post( wc_price( $roi_data['money_protected']['chargeback_prevention'] ?? 0 ) ); ?></strong>
						</div>
					<?php endif; ?>
					<div class="wstl-roi-detail">
						<?php esc_html_e( 'High-Risk Refunds:', 'trustlens' ); ?>
						<strong><?php echo wp_kses_post( wc_price( $roi_data['money_at_risk']['high_risk_refunds'] ?? 0 ) ); ?></strong>
					</div>
				</div>

				<?php if ( wstl_show_pro_upsells() ) : ?>
				<?php $upgrade_url = wstl_get_upgrade_or_fallback_url(); ?>
				<div class="wstl-roi-upsell-strip">
					<span class="dashicons dashicons-superhero-alt"></span>
					<div class="wstl-roi-upsell-strip-text">
						<strong><?php esc_html_e( 'Unlock full protection', 'trustlens' ); ?></strong>
						<?php esc_html_e( 'Stop fraud earlier — auto-block risky customers, get alerts in Slack or email, send scheduled reports, and track chargebacks. All in Pro.', 'trustlens' ); ?>
						<a href="<?php echo esc_url( $upgrade_url ); ?>" class="wstl-roi-upsell-strip-link">
							<?php esc_html_e( 'Upgrade to Pro', 'trustlens' ); ?> &rarr;
						</a>
					</div>
				</div>
				<?php endif; ?>
			</div>
		</div>

		<!-- S5: Secondary Charts -->
		<div class="wstl-dashboard-section">
			<div class="wstl-dashboard-grid-2">
				<div class="wstl-card wstl-chart-card">
					<div class="wstl-chart-header">
						<h2><?php esc_html_e( 'Refund Activity', 'trustlens' ); ?></h2>
						<span class="wstl-chart-period"><?php esc_html_e( 'Last 30 days', 'trustlens' ); ?></span>
					</div>
					<?php if ( $has_refund_data ) : ?>
						<div class="wstl-chart-container wstl-chart-height-md">
							<canvas id="wstl-refund-chart"></canvas>
						</div>
					<?php else : ?>
						<div class="wstl-chart-empty-state wstl-chart-height-md">
							<span class="dashicons dashicons-undo"></span>
							<h3><?php esc_html_e( 'No refund activity', 'trustlens' ); ?></h3>
							<p><?php esc_html_e( 'Refund trends will populate here when refunds are detected in this time window.', 'trustlens' ); ?></p>
						</div>
					<?php endif; ?>
				</div>

				<div class="wstl-card wstl-chart-card">
					<div class="wstl-chart-header">
						<h2><?php esc_html_e( 'Activity by Hour', 'trustlens' ); ?></h2>
						<span class="wstl-chart-period"><?php esc_html_e( 'Last 7 days', 'trustlens' ); ?></span>
					</div>
					<?php if ( $has_hourly_data ) : ?>
						<div class="wstl-chart-container wstl-chart-height-md">
							<canvas id="wstl-hourly-chart"></canvas>
						</div>
					<?php else : ?>
						<div class="wstl-chart-empty-state wstl-chart-height-md">
							<span class="dashicons dashicons-clock"></span>
							<h3><?php esc_html_e( 'No recent activity', 'trustlens' ); ?></h3>
							<p><?php esc_html_e( 'Hourly activity will show once customer events are logged in the last 7 days.', 'trustlens' ); ?></p>
						</div>
					<?php endif; ?>
				</div>
			</div>
		</div>

		<!-- S6: Insights Panel -->
		<div class="wstl-dashboard-section">
			<div class="wstl-dashboard-grid-2">
				<!-- Top Returners -->
				<div class="wstl-card">
					<div class="wstl-chart-header">
						<h2>
							<span class="dashicons dashicons-undo wstl-icon-title wstl-icon-title-danger"></span>
							<?php esc_html_e( 'Highest Return Rates', 'trustlens' ); ?>
						</h2>
					</div>
					<?php if ( empty( $top_returners ) ) : ?>
						<div class="wstl-empty-state">
							<span class="dashicons dashicons-yes-alt"></span>
							<p><?php esc_html_e( 'No refund activity detected yet.', 'trustlens' ); ?></p>
						</div>
					<?php else : ?>
						<table class="wstl-compact-table wstl-returners-table">
							<thead>
								<tr>
									<th><?php esc_html_e( 'Customer', 'trustlens' ); ?></th>
									<th><?php esc_html_e( 'Risk', 'trustlens' ); ?></th>
									<th class="wstl-num wstl-rate-col"><?php esc_html_e( 'Rate', 'trustlens' ); ?></th>
									<th class="wstl-num"><?php esc_html_e( 'Refund Value', 'trustlens' ); ?></th>
									<th><?php esc_html_e( 'Actions', 'trustlens' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ( $top_returners as $index => $returner ) : ?>
									<?php
									$return_rate = (float) $returner->return_rate;
									$rate_class = 'normal';
									if ( $return_rate >= 50 ) {
										$rate_class = 'critical';
									} elseif ( $return_rate >= 30 ) {
										$rate_class = 'high';
									} elseif ( $return_rate >= 15 ) {
										$rate_class = 'elevated';
									}
									?>
									<tr class="<?php echo $index < 2 ? 'is-priority' : ''; ?>">
										<td>
											<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers&customer=' . $returner->email_hash ) ); ?>" class="wstl-email-truncated">
												<?php echo esc_html( $returner->customer_email ?: __( 'Guest', 'trustlens' ) ); ?>
											</a>
											<?php if ( isset( $returner->total_refunds ) && isset( $returner->total_orders ) ) : ?>
												<div class="wstl-row-meta">
													<?php
													printf(
														/* translators: %1$d: refunds, %2$d: orders */
														esc_html__( '%1$d refunds / %2$d orders', 'trustlens' ),
														(int) $returner->total_refunds,
														(int) $returner->total_orders
													);
													?>
												</div>
											<?php endif; ?>
										</td>
										<td>
											<span class="wstl-segment-badge wstl-segment-<?php echo esc_attr( $returner->segment ); ?>">
												<?php echo esc_html( wstl_get_segment_label( $returner->segment ) ); ?>
											</span>
										</td>
										<td class="wstl-num wstl-rate-col">
											<span class="wstl-rate-chip wstl-rate-<?php echo esc_attr( $rate_class ); ?>">
												<?php echo esc_html( number_format( $return_rate, 0 ) ); ?>%
											</span>
										</td>
										<td class="wstl-num"><?php echo wp_kses_post( wc_price( $returner->total_refund_value ) ); ?></td>
										<td>
											<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers&customer=' . $returner->email_hash ) ); ?>" class="button button-small">
												<?php esc_html_e( 'View', 'trustlens' ); ?>
											</a>
										</td>
									</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
						<p class="wstl-table-footer">
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers&orderby=return_rate&order=desc' ) ); ?>">
								<?php esc_html_e( 'View All', 'trustlens' ); ?> &rarr;
							</a>
						</p>
					<?php endif; ?>
				</div>

				<!-- Category Abuse -->
				<div class="wstl-card wstl-chart-card">
					<div class="wstl-chart-header">
						<h2>
							<span class="dashicons dashicons-category wstl-icon-title wstl-icon-title-warn"></span>
							<?php esc_html_e( 'Return Rates by Category', 'trustlens' ); ?>
						</h2>
					</div>
					<?php if ( empty( $category_abuse ) ) : ?>
						<div class="wstl-empty-state">
							<span class="dashicons dashicons-yes-alt"></span>
							<p><?php esc_html_e( 'No category return data yet.', 'trustlens' ); ?></p>
						</div>
					<?php else : ?>
						<div class="wstl-chart-container wstl-chart-height-md">
							<canvas id="wstl-category-chart"></canvas>
						</div>
					<?php endif; ?>
				</div>
			</div>
		</div>

		<!-- S7: Detection & Monitoring -->
		<div class="wstl-dashboard-section">
			<div class="wstl-dashboard-grid-2">
				<!-- Detection Stats -->
				<div class="wstl-card">
					<h2>
						<span class="dashicons dashicons-search wstl-icon-title wstl-icon-title-info"></span>
						<?php esc_html_e( 'Detection Overview', 'trustlens' ); ?>
					</h2>

					<?php if ( wstl_has_chargebacks() && ! empty( $chargeback_stats ) ) : ?>
					<!-- Chargeback Stats -->
					<h3 class="wstl-subsection-title">
						<span class="dashicons dashicons-money-alt wstl-subsection-icon"></span>
						<?php esc_html_e( 'Chargeback Tracking', 'trustlens' ); ?>
					</h3>
					<div class="wstl-detection-stats wstl-detection-stats-spaced">
						<div class="wstl-detection-stat">
							<span class="wstl-detection-value"><?php echo esc_html( $chargeback_stats['total_disputes'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'Total Disputes', 'trustlens' ); ?></span>
						</div>
						<div class="wstl-detection-stat wstl-detection-stat-success">
							<span class="wstl-detection-value"><?php echo esc_html( $chargeback_stats['disputes_won'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'Won', 'trustlens' ); ?></span>
						</div>
						<div class="wstl-detection-stat wstl-detection-stat-danger">
							<span class="wstl-detection-value"><?php echo esc_html( $chargeback_stats['disputes_lost'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'Lost', 'trustlens' ); ?></span>
						</div>
						<div class="wstl-detection-stat">
							<span class="wstl-detection-value"><?php echo esc_html( $chargeback_stats['customers_with_disputes'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'Customers', 'trustlens' ); ?></span>
						</div>
					</div>
					<?php endif; ?>

					<!-- Coupon Abuse Stats -->
					<h3 class="wstl-subsection-title">
						<span class="dashicons dashicons-tag wstl-subsection-icon"></span>
						<?php esc_html_e( 'Coupon Abuse Detection', 'trustlens' ); ?>
					</h3>
					<div class="wstl-detection-stats">
						<div class="wstl-detection-stat">
							<span class="wstl-detection-value"><?php echo esc_html( $coupon_stats['total_coupons'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'Coupons Used', 'trustlens' ); ?></span>
						</div>
						<div class="wstl-detection-stat wstl-detection-stat-warning">
							<span class="wstl-detection-value"><?php echo esc_html( $coupon_stats['first_order_coupons'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'First-Order', 'trustlens' ); ?></span>
						</div>
						<div class="wstl-detection-stat wstl-detection-stat-danger">
							<span class="wstl-detection-value"><?php echo esc_html( $coupon_stats['coupon_refunds'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'Coupon + Refund', 'trustlens' ); ?></span>
						</div>
						<div class="wstl-detection-stat wstl-detection-stat-danger">
							<span class="wstl-detection-value"><?php echo esc_html( $coupon_stats['multi_account_abuse'] ); ?></span>
							<span class="wstl-detection-label"><?php esc_html_e( 'Multi-Account', 'trustlens' ); ?></span>
						</div>
					</div>
				</div>

				<!-- Monthly Protection Trend -->
				<div class="wstl-card wstl-chart-card">
					<div class="wstl-chart-header">
						<h2><?php esc_html_e( 'Protection Trend', 'trustlens' ); ?></h2>
						<span class="wstl-chart-period"><?php esc_html_e( '6 months', 'trustlens' ); ?></span>
					</div>
					<?php if ( $has_monthly_roi_data ) : ?>
						<div class="wstl-chart-container wstl-chart-height-sm">
							<canvas id="wstl-monthly-roi-chart"></canvas>
						</div>
					<?php else : ?>
						<div class="wstl-chart-empty-state wstl-chart-height-sm">
							<span class="dashicons dashicons-shield"></span>
							<h3><?php esc_html_e( 'No protection trend yet', 'trustlens' ); ?></h3>
							<p><?php esc_html_e( 'Protection trend appears after prevention actions and risk events accumulate.', 'trustlens' ); ?></p>
						</div>
					<?php endif; ?>
				</div>
			</div>
		</div>

		<!-- S8: Attention Required -->
		<div class="wstl-dashboard-section">
			<div class="wstl-card wstl-attention-card">
				<h2>
					<?php esc_html_e( 'Customers Requiring Attention', 'trustlens' ); ?>
					<?php if ( $risk_count > 0 ) : ?>
						<span class="wstl-attention-count"><?php echo esc_html( $risk_count ); ?></span>
						<span class="wstl-pulse-dot"></span>
					<?php endif; ?>
				</h2>

				<?php if ( empty( $high_risk_customers ) ) : ?>
					<div class="wstl-empty-state">
						<span class="dashicons dashicons-yes-alt"></span>
						<p><?php esc_html_e( 'No high-risk customers detected. Your store is looking good!', 'trustlens' ); ?></p>
					</div>
				<?php else : ?>
					<table class="wstl-table">
						<thead>
							<tr>
								<th><?php esc_html_e( 'Customer', 'trustlens' ); ?></th>
								<th><?php esc_html_e( 'Score', 'trustlens' ); ?></th>
								<th><?php esc_html_e( 'Return Rate', 'trustlens' ); ?></th>
								<th><?php esc_html_e( 'Refund Value', 'trustlens' ); ?></th>
								<th><?php esc_html_e( 'Actions', 'trustlens' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<?php foreach ( $high_risk_customers as $customer ) : ?>
								<tr>
									<td>
										<span class="wstl-segment-badge wstl-segment-<?php echo esc_attr( $customer->segment ); ?>">
											<?php echo esc_html( wstl_get_segment_label( $customer->segment ) ); ?>
										</span>
										<strong><?php echo esc_html( $customer->customer_email ?: __( 'Guest', 'trustlens' ) ); ?></strong>
										<?php if ( $customer->is_blocked ) : ?>
											<span class="wstl-badge wstl-badge-blocked"><?php esc_html_e( 'Blocked', 'trustlens' ); ?></span>
										<?php endif; ?>
									</td>
									<td>
										<span class="wstl-score wstl-score-<?php echo esc_attr( $customer->segment ); ?>">
											<?php echo esc_html( $customer->trust_score ); ?>
										</span>
									</td>
									<td><?php echo esc_html( number_format( $customer->return_rate, 0 ) ); ?>%</td>
									<td><?php echo wp_kses_post( wc_price( $customer->total_refund_value ?? 0 ) ); ?></td>
									<td>
										<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers&customer=' . $customer->email_hash ) ); ?>" class="button button-small">
											<?php esc_html_e( 'View', 'trustlens' ); ?>
										</a>
									</td>
								</tr>
							<?php endforeach; ?>
						</tbody>
					</table>

					<p class="wstl-table-footer">
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers&segment=risk' ) ); ?>">
							<?php esc_html_e( 'View All Risk Customers', 'trustlens' ); ?> &rarr;
						</a>
					</p>
				<?php endif; ?>
			</div>
		</div>

		<!-- S9: System Status (Collapsible) -->
		<div class="wstl-dashboard-section">
			<?php
			$sync_open = ( 'running' === $sync_status['status'] || ( 'idle' === $sync_status['status'] && 0 === $sync_status['total_orders'] ) );
			$status_class = 'wstl-collapsible-status-pending';
			if ( 'completed' === $sync_status['status'] || ( 'idle' === $sync_status['status'] && $sync_status['total_customers'] > 0 ) ) {
				$status_class = 'wstl-collapsible-status-synced';
			} elseif ( 'running' === $sync_status['status'] ) {
				$status_class = 'wstl-collapsible-status-running';
			}
			?>
			<details class="wstl-collapsible" <?php echo $sync_open ? 'open' : ''; ?>>
				<summary>
					<?php esc_html_e( 'Historical Data Sync', 'trustlens' ); ?>
					<span class="wstl-collapsible-status <?php echo esc_attr( $status_class ); ?>"></span>
				</summary>
				<div class="wstl-collapsible-content">
					<div id="wstl-sync-start-area" class="<?php echo 'running' === $sync_status['status'] ? 'wstl-modal-hidden' : ''; ?>">
						<?php if ( 'completed' === $sync_status['status'] ) : ?>
							<div class="wstl-sync-complete">
								<span class="dashicons dashicons-yes-alt wstl-icon-success wstl-icon-32"></span>
								<p>
									<?php
									printf(
										/* translators: %1$s: customer count, %2$s: order count */
										esc_html__( 'Sync complete! %1$s customers from %2$s orders.', 'trustlens' ),
										'<strong>' . number_format( $sync_status['total_customers'] ) . '</strong>',
										'<strong>' . number_format( $sync_status['processed_orders'] ) . '</strong>'
									);
									?>
								</p>
								<p class="wstl-mt-15">
									<button type="button" class="button" id="wstl-resync">
										<?php esc_html_e( 'Re-sync', 'trustlens' ); ?>
									</button>
								</p>
							</div>
						<?php elseif ( 'idle' === $sync_status['status'] && $sync_status['total_customers'] > 0 ) : ?>
							<div class="wstl-sync-idle">
								<p>
									<?php
									printf(
										/* translators: %s: customer count */
										esc_html__( 'Tracking %s customers.', 'trustlens' ),
										'<strong>' . number_format( $sync_status['total_customers'] ) . '</strong>'
									);
									?>
								</p>
								<button type="button" class="button button-primary" id="wstl-start-sync">
									<?php esc_html_e( 'Start Sync', 'trustlens' ); ?>
								</button>
							</div>
						<?php else : ?>
							<div class="wstl-sync-intro">
								<p><?php esc_html_e( 'Import your existing WooCommerce orders to build customer trust profiles based on historical data.', 'trustlens' ); ?></p>
								<p><strong><?php esc_html_e( 'This process runs in the background and will not slow down your site.', 'trustlens' ); ?></strong></p>
								<button type="button" class="button button-primary" id="wstl-start-sync">
									<?php esc_html_e( 'Start Historical Sync', 'trustlens' ); ?>
								</button>
							</div>
						<?php endif; ?>
					</div>
					<div class="wstl-sync-progress <?php echo 'running' !== $sync_status['status'] ? 'wstl-modal-hidden' : ''; ?>" id="wstl-sync-progress-area">
						<div class="wstl-progress-bar">
							<div class="wstl-progress-fill" style="width: <?php echo esc_attr( $sync_progress ); ?>%;"></div>
						</div>
						<p class="wstl-sync-stats">
							<span id="wstl-sync-progress-text"><?php echo esc_html( $sync_progress ); ?>%</span> &mdash;
							<span id="wstl-sync-orders-text">
								<?php
								printf(
									/* translators: %1$s: processed orders, %2$s: total orders */
									esc_html__( '%1$s of %2$s orders processed', 'trustlens' ),
									number_format( $sync_status['processed_orders'] ),
									number_format( $sync_status['total_orders'] )
								);
								?>
							</span>
						</p>
						<button type="button" class="button" id="wstl-stop-sync">
							<?php esc_html_e( 'Stop Sync', 'trustlens' ); ?>
						</button>
					</div>
				</div>
			</details>
		</div>
		<?php endif; // End if ( $is_empty ) / else. ?>

	</div><!-- .wstl-dashboard -->
</div><!-- .wrap -->

<?php // phpcs:enable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound ?>
