<?php
/**
 * Customer detail page template.
 *
 * @package TrustLens\Admin\Views
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variables scoped to included view file.

$color = wstl_get_segment_color( $customer->segment );

// Get linked accounts.
$links = TrustLens_Linked_Accounts::instance()->find_linked_accounts( $email_hash );
?>

<div class="wrap wstl-wrap">
	<h1 class="wp-heading-inline">
		<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers' ) ); ?>">&larr;</a>
		<?php esc_html_e( 'Customer Details', 'trustlens' ); ?>
	</h1>
	<hr class="wp-header-end">

	<div class="wstl-customer-detail">
		<!-- Header Card -->
		<div class="wstl-card wstl-customer-header" style="border-left: 4px solid <?php echo esc_attr( $color ); ?>;">
			<div class="wstl-customer-info">
				<h2>
					<?php echo esc_html( $customer->customer_email ?: __( 'Guest Customer', 'trustlens' ) ); ?>
					<span class="wstl-segment-badge wstl-segment-<?php echo esc_attr( $customer->segment ); ?>">
						<?php echo esc_html( wstl_get_segment_label( $customer->segment ) ); ?>
					</span>
					<?php if ( $customer->is_blocked ) : ?>
						<span class="wstl-badge wstl-badge-blocked"><?php esc_html_e( 'Blocked', 'trustlens' ); ?></span>
					<?php endif; ?>
					<?php if ( $customer->is_allowlisted ) : ?>
						<span class="wstl-badge wstl-badge-allowlist"><?php esc_html_e( 'Allowlisted', 'trustlens' ); ?></span>
					<?php endif; ?>
				</h2>
				<p class="wstl-customer-meta">
					<?php echo $customer->customer_type === 'user' ? esc_html__( 'Registered Customer', 'trustlens' ) : esc_html__( 'Guest', 'trustlens' ); ?>
					<?php if ( $customer->customer_id ) : ?>
						&middot;
						<a href="<?php echo esc_url( admin_url( 'user-edit.php?user_id=' . $customer->customer_id ) ); ?>">
							<?php esc_html_e( 'View User', 'trustlens' ); ?>
						</a>
					<?php endif; ?>
					<?php if ( $customer->first_order_date ) : ?>
						&middot;
						<?php
						printf(
							/* translators: %s: date */
							esc_html__( 'Customer since %s', 'trustlens' ),
							esc_html( date_i18n( get_option( 'date_format' ), strtotime( $customer->first_order_date ) ) )
						);
						?>
					<?php endif; ?>
				</p>
			</div>

			<div class="wstl-customer-score">
				<div class="wstl-score-circle" style="border-color: <?php echo esc_attr( $color ); ?>;">
					<span class="wstl-score-value"><?php echo esc_html( $customer->trust_score ); ?></span>
					<span class="wstl-score-label"><?php esc_html_e( 'Trust Score', 'trustlens' ); ?></span>
				</div>
			</div>

			<div class="wstl-customer-actions">
				<?php if ( $customer->is_blocked ) : ?>
					<button type="button" class="button wstl-action-unblock" data-hash="<?php echo esc_attr( $customer->email_hash ); ?>">
						<?php esc_html_e( 'Unblock', 'trustlens' ); ?>
					</button>
				<?php else : ?>
					<button type="button" class="button button-secondary wstl-action-block" data-hash="<?php echo esc_attr( $customer->email_hash ); ?>">
						<?php esc_html_e( 'Block Customer', 'trustlens' ); ?>
					</button>
				<?php endif; ?>

				<?php if ( $customer->is_allowlisted ) : ?>
					<button type="button" class="button wstl-action-remove-allowlist" data-hash="<?php echo esc_attr( $customer->email_hash ); ?>">
						<?php esc_html_e( 'Remove from Allowlist', 'trustlens' ); ?>
					</button>
				<?php else : ?>
					<button type="button" class="button wstl-action-allowlist" data-hash="<?php echo esc_attr( $customer->email_hash ); ?>">
						<?php esc_html_e( 'Add to Allowlist', 'trustlens' ); ?>
					</button>
				<?php endif; ?>

				<button type="button" class="button wstl-action-recalculate" data-hash="<?php echo esc_attr( $customer->email_hash ); ?>">
					<?php esc_html_e( 'Recalculate Score', 'trustlens' ); ?>
				</button>
			</div>
		</div>

		<!-- Stats Grid -->
		<div class="wstl-stats-row">
			<div class="wstl-card wstl-stat-card">
				<h3><?php esc_html_e( 'Orders', 'trustlens' ); ?></h3>
				<div class="wstl-stat-value"><?php echo esc_html( number_format( $customer->total_orders ) ); ?></div>
				<div class="wstl-stat-sub"><?php echo wp_kses_post( wc_price( $customer->total_order_value ) ); ?> <?php esc_html_e( 'total', 'trustlens' ); ?></div>
			</div>

			<div class="wstl-card wstl-stat-card">
				<h3><?php esc_html_e( 'Returns', 'trustlens' ); ?></h3>
				<div class="wstl-stat-value"><?php echo esc_html( number_format( $customer->total_refunds ) ); ?></div>
				<div class="wstl-stat-sub"><?php echo wp_kses_post( wc_price( $customer->total_refund_value ) ); ?> <?php esc_html_e( 'refunded', 'trustlens' ); ?></div>
			</div>

			<div class="wstl-card wstl-stat-card">
				<h3><?php esc_html_e( 'Return Rate', 'trustlens' ); ?></h3>
				<div class="wstl-stat-value <?php echo $customer->return_rate >= 40 ? 'wstl-stat-danger' : ''; ?>">
					<?php echo esc_html( number_format( $customer->return_rate, 1 ) ); ?>%
				</div>
				<div class="wstl-stat-sub">
					<?php echo esc_html( $customer->full_refunds ); ?> <?php esc_html_e( 'full', 'trustlens' ); ?> /
					<?php echo esc_html( $customer->partial_refunds ); ?> <?php esc_html_e( 'partial', 'trustlens' ); ?>
				</div>
			</div>

			<div class="wstl-card wstl-stat-card">
				<h3><?php esc_html_e( 'Cancelled', 'trustlens' ); ?></h3>
				<div class="wstl-stat-value"><?php echo esc_html( number_format( $customer->cancelled_orders ) ); ?></div>
				<div class="wstl-stat-sub"><?php esc_html_e( 'orders cancelled', 'trustlens' ); ?></div>
			</div>
		</div>

		<!-- Two Column Layout -->
		<div class="wstl-detail-row">
			<!-- Left Column -->
			<div class="wstl-detail-col">
				<!-- Signals -->
				<div class="wstl-card">
					<h3><?php esc_html_e( 'Trust Score Breakdown', 'trustlens' ); ?></h3>

					<?php if ( empty( $signals ) ) : ?>
						<p><?php esc_html_e( 'No signals recorded yet.', 'trustlens' ); ?></p>
					<?php else : ?>
						<table class="wstl-table wstl-signals-table">
							<thead>
								<tr>
									<th><?php esc_html_e( 'Module', 'trustlens' ); ?></th>
									<th><?php esc_html_e( 'Impact', 'trustlens' ); ?></th>
									<th><?php esc_html_e( 'Reason', 'trustlens' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ( $signals as $signal ) : ?>
									<tr>
										<td><?php echo esc_html( ucfirst( str_replace( '_', ' ', $signal->module_id ) ) ); ?></td>
										<td>
											<?php if ( $signal->signal_score > 0 ) : ?>
												<span class="wstl-signal-positive">+<?php echo esc_html( $signal->signal_score ); ?></span>
											<?php elseif ( $signal->signal_score < 0 ) : ?>
												<span class="wstl-signal-negative"><?php echo esc_html( $signal->signal_score ); ?></span>
											<?php else : ?>
												<span class="wstl-signal-neutral">0</span>
											<?php endif; ?>
										</td>
										<td><?php echo esc_html( $signal->signal_reason ); ?></td>
									</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
					<?php endif; ?>
				</div>

				<!-- Linked Accounts -->
				<?php if ( get_option( 'trustlens_module_linked_accounts_enabled', true ) ) : ?>
					<div class="wstl-card">
						<h3>
							<?php esc_html_e( 'Linked Accounts', 'trustlens' ); ?>
							<?php if ( ! empty( $links ) ) : ?>
								<span class="wstl-badge"><?php echo count( $links ); ?></span>
							<?php endif; ?>
						</h3>

						<?php if ( empty( $links ) ) : ?>
							<p class="wstl-muted"><?php esc_html_e( 'No linked accounts detected.', 'trustlens' ); ?></p>
						<?php else : ?>
							<div class="wstl-linked-accounts">
								<?php foreach ( $links as $link ) : ?>
									<?php
									$linked_customer = wstl_get_customer( $link['email_hash'] );
									if ( ! $linked_customer ) {
										continue;
									}
									$link_color = wstl_get_segment_color( $linked_customer->segment );
									$is_risky = in_array( $linked_customer->segment, array( 'risk', 'critical' ), true ) || $linked_customer->is_blocked;

									$type_labels = array(
										'device'      => __( 'Same Device', 'trustlens' ),
										'ip'          => __( 'Same IP', 'trustlens' ),
										'fingerprint' => __( 'Same Fingerprint', 'trustlens' ),
									);
									$type_names = array_map(
										function ( $type ) use ( $type_labels ) {
											return $type_labels[ $type ] ?? ucfirst( $type );
										},
										$link['match_types']
									);
									?>
									<div class="wstl-linked-account <?php echo $is_risky ? 'wstl-linked-risky' : ''; ?>">
										<div class="wstl-linked-info">
											<span class="wstl-segment-badge wstl-segment-<?php echo esc_attr( $linked_customer->segment ); ?>">
												<?php echo esc_html( wstl_get_segment_label( $linked_customer->segment ) ); ?>
											</span>
											<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers&customer=' . $link['email_hash'] ) ); ?>">
												<?php echo esc_html( $linked_customer->customer_email ?: __( 'Guest', 'trustlens' ) ); ?>
											</a>
											<?php if ( $linked_customer->is_blocked ) : ?>
												<span class="wstl-badge wstl-badge-blocked"><?php esc_html_e( 'Blocked', 'trustlens' ); ?></span>
											<?php endif; ?>
										</div>
										<div class="wstl-linked-meta">
											<span class="wstl-link-type" title="<?php esc_attr_e( 'Match types', 'trustlens' ); ?>">
												<?php echo esc_html( implode( ', ', $type_names ) ); ?>
											</span>
											<span class="wstl-link-score">
												<?php esc_html_e( 'Score:', 'trustlens' ); ?>
												<strong style="color: <?php echo esc_attr( $link_color ); ?>;">
													<?php echo esc_html( $linked_customer->trust_score ); ?>
												</strong>
											</span>
											<span class="wstl-link-matches">
												<?php
												printf(
													/* translators: %d: number of matching fingerprints */
													esc_html( _n( '%d match', '%d matches', $link['match_count'], 'trustlens' ) ),
													(int) $link['match_count']
												);
												?>
											</span>
										</div>
									</div>
								<?php endforeach; ?>
							</div>
						<?php endif; ?>
					</div>
				<?php endif; ?>
			</div>

			<!-- Right Column -->
			<div class="wstl-detail-col">
				<!-- Recent Events -->
				<div class="wstl-card">
					<h3><?php esc_html_e( 'Recent Activity', 'trustlens' ); ?></h3>

					<?php if ( empty( $events ) ) : ?>
						<p><?php esc_html_e( 'No activity recorded yet.', 'trustlens' ); ?></p>
					<?php else : ?>
						<div class="wstl-activity-feed">
							<?php foreach ( $events as $event ) : ?>
								<?php $event_data = json_decode( $event->event_data, true ); ?>
								<div class="wstl-activity-item wstl-activity-<?php echo esc_attr( $event->event_type ); ?>">
									<div class="wstl-activity-icon">
										<?php
										$icons = array(
											'order_completed' => 'cart',
											'refund'          => 'money-alt',
											'full_refund'     => 'money-alt',
											'partial_refund'  => 'money-alt',
											'order_cancelled' => 'no',
											'blocked'         => 'lock',
											'unblocked'       => 'unlock',
										);
										$icon = $icons[ $event->event_type ] ?? 'marker';
										?>
										<span class="dashicons dashicons-<?php echo esc_attr( $icon ); ?>"></span>
									</div>
									<div class="wstl-activity-content">
										<div class="wstl-activity-title">
											<?php echo esc_html( ucfirst( str_replace( '_', ' ', $event->event_type ) ) ); ?>
											<?php if ( $event->order_id ) : ?>
												- <a href="<?php echo esc_url( admin_url( 'post.php?post=' . $event->order_id . '&action=edit' ) ); ?>">
													#<?php echo esc_html( $event->order_id ); ?>
												</a>
											<?php endif; ?>
										</div>
										<div class="wstl-activity-meta">
											<?php echo esc_html( human_time_diff( strtotime( $event->created_at ), current_time( 'timestamp' ) ) ); ?>
											<?php esc_html_e( 'ago', 'trustlens' ); ?>
											<?php if ( isset( $event_data['refund_amount'] ) ) : ?>
												&middot; <?php echo wp_kses_post( wc_price( $event_data['refund_amount'] ) ); ?>
											<?php endif; ?>
										</div>
									</div>
								</div>
							<?php endforeach; ?>
						</div>
					<?php endif; ?>
				</div>

				<!-- Admin Notes -->
				<div class="wstl-card">
					<h3><?php esc_html_e( 'Admin Notes', 'trustlens' ); ?></h3>
					<?php if ( $customer->admin_notes ) : ?>
						<pre class="wstl-admin-notes"><?php echo esc_html( $customer->admin_notes ); ?></pre>
					<?php else : ?>
						<p class="wstl-muted"><?php esc_html_e( 'No notes yet.', 'trustlens' ); ?></p>
					<?php endif; ?>
					<textarea id="wstl-add-note" class="large-text" rows="2" placeholder="<?php esc_attr_e( 'Add a note...', 'trustlens' ); ?>"></textarea>
					<button type="button" class="button wstl-save-note wstl-mt-10" data-hash="<?php echo esc_attr( $customer->email_hash ); ?>">
						<?php esc_html_e( 'Save Note', 'trustlens' ); ?>
					</button>
				</div>
			</div>
		</div>
	</div>
</div>

<?php // phpcs:enable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound ?>
