/**
 * TrustLens Dashboard Charts
 *
 * Initializes all Chart.js charts on the dashboard page.
 * Reads data from the TrustLensDashboard object (wp_localize_script).
 *
 * @package TrustLens
 * @since   1.0.0
 */

/* global Chart, TrustLensDashboard */

(function() {
	'use strict';

	document.addEventListener('DOMContentLoaded', function() {
		if (typeof Chart === 'undefined' || typeof TrustLensDashboard === 'undefined') {
			return;
		}

		var data = TrustLensDashboard;
		var i18n = data.i18n;

		// Global Chart.js defaults.
		Chart.defaults.font.family = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
		Chart.defaults.color = '#50575e';
		Chart.defaults.plugins.tooltip.backgroundColor = '#1d2327';
		Chart.defaults.plugins.tooltip.cornerRadius = 6;
		Chart.defaults.plugins.tooltip.padding = 10;
		Chart.defaults.plugins.legend.labels.usePointStyle = true;
		Chart.defaults.plugins.legend.labels.pointStyle = 'circle';
		Chart.defaults.plugins.legend.labels.pointStyleWidth = 10;
		Chart.defaults.plugins.legend.labels.boxWidth = 10;
		Chart.defaults.plugins.legend.labels.boxHeight = 10;
		Chart.defaults.plugins.legend.labels.useBorderRadius = true;
		Chart.defaults.plugins.legend.labels.borderRadius = 999;
		Chart.defaults.elements.bar.borderRadius = 4;
		Chart.defaults.scale.grid.color = 'rgba(0, 0, 0, 0.04)';
		Chart.defaults.scale.border = { display: false };

		// Force all legend markers to render as true circles across chart types.
		var defaultGenerateLegendLabels = Chart.defaults.plugins.legend.labels.generateLabels;
		Chart.defaults.plugins.legend.labels.generateLabels = function(chart) {
			var labels = defaultGenerateLegendLabels(chart);
			return labels.map(function(label) {
				return Object.assign({}, label, {
					pointStyle: 'circle',
					lineWidth: 0,
					strokeStyle: label.fillStyle || label.strokeStyle,
					borderRadius: 999
				});
			});
		};

		// Center text plugin for doughnut chart.
		var centerTextPlugin = {
			id: 'centerText',
			afterDraw: function(chart) {
				if (chart.config.type !== 'doughnut') {
					return;
				}
				var width = chart.width;
				var height = chart.height;
				var ctx = chart.ctx;
				ctx.restore();

				var fontSize = (height / 114).toFixed(2);
				ctx.font = 'bold ' + fontSize + 'em -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif';
				ctx.textBaseline = 'middle';
				ctx.textAlign = 'center';
				ctx.fillStyle = '#1d2327';

				var total = chart.data.datasets[0].data.reduce(function(a, b) { return a + b; }, 0);
				ctx.fillText(total.toLocaleString(), width / 2, height / 2 - 8);

				ctx.font = '0.65em -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif';
				ctx.fillStyle = '#666';
				ctx.fillText(i18n.customers, width / 2, height / 2 + 14);
				ctx.save();
			}
		};
		Chart.register(centerTextPlugin);

		// Helper: format date string to short format.
		function formatDate(dateStr) {
			var d = new Date(dateStr + 'T00:00:00');
			var months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
			return months[d.getMonth()] + ' ' + d.getDate();
		}

		// ── Chart 1: Trust Score Trends ──
		var trendsCtx = document.getElementById('wstl-trends-chart');
		if (trendsCtx) {
			var trendsData = data.trends;
			var trendsGradient = trendsCtx.getContext('2d').createLinearGradient(0, 0, 0, 300);
			trendsGradient.addColorStop(0, 'rgba(102, 126, 234, 0.25)');
			trendsGradient.addColorStop(1, 'rgba(102, 126, 234, 0)');

			new Chart(trendsCtx.getContext('2d'), {
				type: 'line',
				data: {
					labels: trendsData.map(function(d) { return formatDate(d.date); }),
					datasets: [
						{
							label: i18n.averageScore,
							data: trendsData.map(function(d) { return d.avg_score; }),
							borderColor: '#667eea',
							backgroundColor: trendsGradient,
							fill: true,
							tension: 0.4,
							pointRadius: 0,
							pointHoverRadius: 6,
							pointHoverBackgroundColor: '#667eea',
							borderWidth: 2.5
						},
						{
							label: i18n.threshold,
							data: trendsData.map(function() { return 50; }),
							borderColor: 'rgba(0, 0, 0, 0.12)',
							borderDash: [6, 4],
							borderWidth: 1,
							pointRadius: 0,
							pointHoverRadius: 0,
							fill: false
						}
					]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					interaction: {
						intersect: false,
						mode: 'index'
					},
					scales: {
						y: {
							beginAtZero: true,
							max: 100,
							ticks: {
								stepSize: 25
							}
						},
						x: {
							ticks: {
								maxTicksLimit: 10
							}
						}
					},
					plugins: {
						legend: {
							display: false
						},
						tooltip: {
							callbacks: {
								afterBody: function(context) {
									var idx = context[0].dataIndex;
									var customers = trendsData[idx] ? trendsData[idx].customers : 0;
									return customers + ' ' + i18n.customersScored;
								}
							}
						}
					}
				}
			});
		}

		// ── Chart 2: Segment Distribution ──
		var segmentCtx = document.getElementById('wstl-segment-chart');
		if (segmentCtx) {
			var segments = ['vip', 'trusted', 'normal', 'caution', 'risk', 'critical'];
			var segmentLabels = segments.map(function(s) { return data.segmentLabels[s]; });
			var segmentData = segments.map(function(s) { return parseInt(data.segmentCounts[s], 10) || 0; });
			var segmentColors = segments.map(function(s) { return data.segmentColors[s]; });

			new Chart(segmentCtx.getContext('2d'), {
				type: 'doughnut',
				data: {
					labels: segmentLabels,
					datasets: [{
						data: segmentData,
						backgroundColor: segmentColors,
						borderWidth: 0,
						hoverOffset: 8
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					cutout: '65%',
					plugins: {
						legend: {
							position: 'bottom',
							labels: {
								padding: 16,
								usePointStyle: true,
								pointStyle: 'circle',
								pointStyleWidth: 10,
								boxWidth: 10,
								boxHeight: 10
							}
						}
					}
				}
			});
		}

		// ── Chart 3: Refund Trends (Stacked Bar) ──
		var refundCtx = document.getElementById('wstl-refund-chart');
		if (refundCtx) {
			var refundData = data.refundTrends;
			new Chart(refundCtx.getContext('2d'), {
				type: 'bar',
				data: {
					labels: refundData.map(function(d) { return formatDate(d.date); }),
					datasets: [
						{
							label: i18n.fullRefunds,
							data: refundData.map(function(d) { return d.full_refunds; }),
							backgroundColor: '#dc3545'
						},
						{
							label: i18n.partialRefunds,
							data: refundData.map(function(d) { return d.partial_refunds; }),
							backgroundColor: '#fd7e14'
						}
					]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					scales: {
						x: {
							stacked: true,
							ticks: { maxTicksLimit: 10 }
						},
						y: {
							stacked: true,
							beginAtZero: true
						}
					},
					plugins: {
						legend: {
							position: 'bottom',
							labels: {
								padding: 16,
								usePointStyle: true,
								pointStyle: 'circle',
								pointStyleWidth: 10,
								boxWidth: 10,
								boxHeight: 10
							}
						}
					}
				}
			});
		}

		// ── Chart 4: Hourly Activity ──
		var hourlyCtx = document.getElementById('wstl-hourly-chart');
		if (hourlyCtx) {
			var hourlyData = data.hourlyActivity;
			var hourLabels = [];
			var hourColors = [];
			var maxHourly = Math.max.apply(null, hourlyData) || 1;
			for (var i = 0; i < 24; i++) {
				var suffix = i < 12 ? 'am' : 'pm';
				var displayHour = i === 0 ? 12 : (i > 12 ? i - 12 : i);
				hourLabels.push(displayHour + suffix);
				var intensity = hourlyData[i] / maxHourly;
				var r = Math.round(102 + (118 - 102) * intensity);
				var g = Math.round(126 + (75 - 126) * intensity);
				var b = Math.round(234 + (162 - 234) * intensity);
				hourColors.push('rgba(' + r + ',' + g + ',' + b + ',' + (0.4 + 0.6 * intensity) + ')');
			}

			new Chart(hourlyCtx.getContext('2d'), {
				type: 'bar',
				data: {
					labels: hourLabels,
					datasets: [{
						label: i18n.events,
						data: hourlyData,
						backgroundColor: hourColors
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					scales: {
						y: { beginAtZero: true },
						x: {
							ticks: {
								maxTicksLimit: 8,
								callback: function(value, index) {
									return index % 3 === 0 ? this.getLabelForValue(value) : '';
								}
							}
						}
					},
					plugins: {
						legend: { display: false }
					}
				}
			});
		}

		// ── Chart 5: Category Abuse (Horizontal Bar) ──
		var categoryCtx = document.getElementById('wstl-category-chart');
		if (categoryCtx && data.categoryAbuse && data.categoryAbuse.length > 0) {
			var categoryData = data.categoryAbuse;
			var catLabels = categoryData.map(function(d) {
				var name = d.category_slug.replace(/-/g, ' ');
				return name.charAt(0).toUpperCase() + name.slice(1);
			});
			var catValues = categoryData.map(function(d) { return parseFloat(d.return_rate); });
			var catColors = catValues.map(function(v) {
				if (v >= 30) {
					return '#dc3545';
				}
				if (v >= 15) {
					return '#fd7e14';
				}
				return '#ffc107';
			});

			new Chart(categoryCtx.getContext('2d'), {
				type: 'bar',
				data: {
					labels: catLabels,
					datasets: [{
						label: i18n.returnRatePercent,
						data: catValues,
						backgroundColor: catColors,
						borderRadius: 4
					}]
				},
				options: {
					indexAxis: 'y',
					responsive: true,
					maintainAspectRatio: false,
					scales: {
						x: {
							beginAtZero: true,
							ticks: {
								callback: function(value) { return value + '%'; }
							}
						}
					},
					plugins: {
						legend: { display: false },
						tooltip: {
							callbacks: {
								label: function(context) {
									var d = categoryData[context.dataIndex];
									return context.parsed.x + '% (' + parseInt(d.total_refunds, 10) + '/' + parseInt(d.total_orders, 10) + ' orders)';
								}
							}
						}
					}
				}
			});
		}

		// ── Chart 6: Monthly ROI Trend ──
		var monthlyCtx = document.getElementById('wstl-monthly-roi-chart');
		if (monthlyCtx) {
			var monthlyData = data.monthlyRoi;
			var protectedGradient = monthlyCtx.getContext('2d').createLinearGradient(0, 0, 0, 280);
			protectedGradient.addColorStop(0, 'rgba(40, 167, 69, 0.2)');
			protectedGradient.addColorStop(1, 'rgba(40, 167, 69, 0)');

			var riskGradient = monthlyCtx.getContext('2d').createLinearGradient(0, 0, 0, 280);
			riskGradient.addColorStop(0, 'rgba(220, 53, 69, 0.15)');
			riskGradient.addColorStop(1, 'rgba(220, 53, 69, 0)');

			new Chart(monthlyCtx.getContext('2d'), {
				type: 'line',
				data: {
					labels: monthlyData.map(function(d) { return d.month_label; }),
					datasets: [
						{
							label: i18n.protected,
							data: monthlyData.map(function(d) { return d.money_protected; }),
							borderColor: '#28a745',
							backgroundColor: protectedGradient,
							fill: true,
							tension: 0.3,
							pointRadius: 4,
							pointHoverRadius: 6,
							borderWidth: 2.5
						},
						{
							label: i18n.atRisk,
							data: monthlyData.map(function(d) { return d.money_at_risk; }),
							borderColor: '#dc3545',
							backgroundColor: riskGradient,
							fill: true,
							tension: 0.3,
							pointRadius: 4,
							pointHoverRadius: 6,
							borderWidth: 2.5
						}
					]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					interaction: {
						intersect: false,
						mode: 'index'
					},
					scales: {
						y: {
							beginAtZero: true,
							ticks: {
								callback: function(value) {
									if (value >= 1000) {
										return '$' + (value / 1000).toFixed(0) + 'k';
									}
									return '$' + value;
								}
							}
						}
					},
					plugins: {
						legend: {
							position: 'bottom',
							labels: {
								padding: 16,
								usePointStyle: true,
								pointStyle: 'circle',
								pointStyleWidth: 10,
								boxWidth: 10,
								boxHeight: 10
							}
						},
						tooltip: {
							callbacks: {
								label: function(context) {
									return context.dataset.label + ': $' + context.parsed.y.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2});
								}
							}
						}
					}
				}
			});
		}
	});
})();
