/**
 * TrustLens Admin JavaScript
 *
 * @package TrustLens
 * @since   1.0.0
 */

(function($) {
	'use strict';

	/**
	 * TrustLens Admin object.
	 */
	var TrustLensAdmin = {

		/**
		 * Initialize.
		 */
		init: function() {
			this.bindEvents();
		},

		/**
		 * Extract a useful message from failed AJAX responses.
		 *
		 * @param {jqXHR} xhr jQuery XHR object.
		 * @return {string} Human-readable error.
		 */
		getAjaxErrorMessage: function(xhr) {
			if (xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
				return xhr.responseJSON.data.message;
			}

			if (xhr && xhr.responseText) {
				try {
					var parsed = JSON.parse(xhr.responseText);
					if (parsed && parsed.data && parsed.data.message) {
						return parsed.data.message;
					}
				} catch (e) {
					// Not JSON; fall through to generic message.
				}
			}

			return TrustLens.i18n.error;
		},

		/**
		 * Bind events.
		 */
		bindEvents: function() {
			var self = this;

			$(document).on('click', '.wstl-action-block', this.blockCustomer);
			$(document).on('click', '.wstl-action-unblock', this.unblockCustomer);
			$(document).on('click', '.wstl-action-allowlist', this.allowlistCustomer);
			$(document).on('click', '.wstl-action-remove-allowlist', this.removeAllowlist);

			// Sync controls.
			$(document).on('click', '#wstl-start-sync', this.startSync);
			$(document).on('click', '#wstl-stop-sync', this.stopSync);
			$(document).on('click', '#wstl-resync', this.startSync);
			$(document).on('click', '#wstl-reset-sync', this.resetSync);
			$(document).on('click', '#wstl-reset-data', this.resetData);

			// API key generation.
			$(document).on('click', '#wstl-generate-api-key, #wstl-regenerate-api-key', this.generateApiKey);

			// Bulk actions.
			$(document).on('click', '#wstl-select-all, #wstl-select-all-bottom', this.toggleSelectAll);
			$(document).on('click', '#wstl-bulk-apply, #wstl-bulk-apply-bottom', this.applyBulkAction);
			$(document).on('click', '#wstl-export-csv', this.exportCSV);

			// Customer detail actions.
			$(document).on('click', '.wstl-action-recalculate', this.recalculateScore);
			$(document).on('click', '.wstl-save-note', this.saveNote);

			// Automation rules.
			$(document).on('click', '#wstl-add-automation-rule, #wstl-add-automation-rule-empty', function() { self.openAutomationModal(); });
			$(document).on('click', '.wstl-edit-rule', function() {
				var ruleId = $(this).closest('tr').data('rule-id');
				self.openAutomationModal(ruleId);
			});
			$(document).on('click', '.wstl-delete-rule', this.deleteAutomationRule);
			$(document).on('click', '#wstl-save-rule', this.saveAutomationRule);
			$(document).on('submit', '#wstl-rule-form', this.saveAutomationRule);
			$(document).on('change', '.wstl-rule-enabled', this.toggleAutomationRuleEnabled);
			$(document).on('click', '#wstl-add-condition', this.addAutomationCondition);
			$(document).on('click', '.wstl-remove-condition', this.removeAutomationCondition);

			// Webhooks.
			$(document).on('click', '#wstl-add-webhook', function() { self.openWebhookModal(); });
			$(document).on('click', '.wstl-test-webhook', this.testWebhook);
			$(document).on('click', '.wstl-delete-webhook', this.deleteWebhook);
			$(document).on('click', '#wstl-save-webhook', this.saveWebhook);

			// Scheduled reports.
			$(document).on('click', '#wstl-add-report-schedule', function() { self.openReportModal(); });
			$(document).on('click', '.wstl-send-report-now', this.sendReportNow);
			$(document).on('click', '.wstl-delete-schedule', this.deleteReportSchedule);
			$(document).on('click', '#wstl-save-schedule', this.saveReportSchedule);

			// Test notification.
			$(document).on('click', '#wstl-send-test-notification', this.sendTestNotification);

			// Test data (Settings > Data).
			$(document).on('click', '#wstl-seed-test-data', this.seedTestData);
			$(document).on('click', '#wstl-remove-test-data', this.removeTestData);

			// Modal close.
			$(document).on('click', '.wstl-modal-close, .wstl-modal-overlay, .wstl-modal-cancel', this.closeModal);
			$(document).on('keydown', function(e) {
				if (e.key === 'Escape') {
					self.closeModal();
				}
			});

			// Resume AJAX batch processing if sync is running.
			if ($('.wstl-sync-progress:visible').length) {
				this.processNextBatch();
			}
		},

		/**
		 * Block customer.
		 *
		 * @param {Event} e Click event.
		 */
		blockCustomer: function(e) {
			e.preventDefault();

			var $button = $(this);
			var hash = $button.data('hash');

			if (!confirm(TrustLens.i18n.confirm_block)) {
				return;
			}

			$button.prop('disabled', true).text('...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_block_customer',
					nonce: TrustLens.nonce,
					email_hash: hash
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Block');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Block');
				}
			});
		},

		/**
		 * Unblock customer.
		 *
		 * @param {Event} e Click event.
		 */
		unblockCustomer: function(e) {
			e.preventDefault();

			var $button = $(this);
			var hash = $button.data('hash');

			$button.prop('disabled', true).text('...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_unblock_customer',
					nonce: TrustLens.nonce,
					email_hash: hash
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Unblock');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Unblock');
				}
			});
		},

		/**
		 * Add to allowlist.
		 *
		 * @param {Event} e Click event.
		 */
		allowlistCustomer: function(e) {
			e.preventDefault();

			var $button = $(this);
			var hash = $button.data('hash');

			if (!confirm(TrustLens.i18n.confirm_allowlist)) {
				return;
			}

			$button.prop('disabled', true).text('...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_allowlist_customer',
					nonce: TrustLens.nonce,
					email_hash: hash
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Add to Allowlist');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Add to Allowlist');
				}
			});
		},

		/**
		 * Remove from allowlist.
		 *
		 * @param {Event} e Click event.
		 */
		removeAllowlist: function(e) {
			e.preventDefault();

			var $button = $(this);
			var hash = $button.data('hash');

			$button.prop('disabled', true).text('...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_remove_allowlist',
					nonce: TrustLens.nonce,
					email_hash: hash
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Remove from Allowlist');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Remove from Allowlist');
				}
			});
		},

		/**
		 * Start historical sync.
		 *
		 * @param {Event} e Click event.
		 */
		startSync: function(e) {
			e.preventDefault();

			var self = TrustLensAdmin;
			var $button = $(this);
			if (!$button.data('original-text')) {
				$button.data('original-text', $button.text());
			}
			$button.prop('disabled', true).text(TrustLens.i18n.starting || 'Starting...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_start_sync',
					nonce: TrustLens.nonce
				},
				success: function(response) {
					if (response.success) {
						var totalOrders = response.data.status.total_orders;
						var ordersText = '0 of ' + self.formatNumber(totalOrders) + ' orders processed';
						var fullText = 'Processed ' + ordersText + ' (0%)';

						// Switch UI: hide start area, show progress area (works on Dashboard and Data tab).
						$('#wstl-sync-start-area').hide();
						$('#wstl-sync-progress-area').show();
						$('#wstl-start-sync').hide();
						$('#wstl-stop-sync').show();

						// Initialize progress bar and text.
						$('.wstl-progress-fill').css('width', '0%');
						$('#wstl-sync-progress-text').text('0%');
						var $ordersEl = $('#wstl-sync-orders-text');
						if ($ordersEl.length) {
							$ordersEl.text($ordersEl.is('span') ? ordersText : fullText);
						}

						// Start processing batches.
						self.processNextBatch();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text($button.data('original-text') || TrustLens.i18n.start_sync || 'Start Historical Sync');
					}
				},
				error: function(xhr) {
					alert(self.getAjaxErrorMessage(xhr));
					$button.prop('disabled', false).text($button.data('original-text') || TrustLens.i18n.start_sync || 'Start Historical Sync');
				}
			});
		},

		/**
		 * Stop historical sync.
		 *
		 * @param {Event} e Click event.
		 */
		stopSync: function(e) {
			e.preventDefault();

			var self = TrustLensAdmin;
			var $button = $(this);
			$button.prop('disabled', true).text(TrustLens.i18n.stopping || 'Stopping...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_stop_sync',
					nonce: TrustLens.nonce
				},
				success: function(response) {
					location.reload();
				},
				error: function(xhr) {
					alert(self.getAjaxErrorMessage(xhr));
					$button.prop('disabled', false).text(TrustLens.i18n.stop_sync || 'Stop Sync');
				}
			});
		},

		/**
		 * Reset sync data.
		 *
		 * @param {Event} e Click event.
		 */
		resetSync: function(e) {
			e.preventDefault();

			if (!confirm(TrustLens.i18n.confirm_reset || 'Are you sure you want to delete all TrustLens data? This cannot be undone.')) {
				return;
			}

			var self = TrustLensAdmin;
			var $button = $(this);
			$button.prop('disabled', true).text(TrustLens.i18n.resetting || 'Resetting...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_reset_sync',
					nonce: TrustLens.nonce
				},
				success: function(response) {
					location.reload();
				},
				error: function(xhr) {
					alert(self.getAjaxErrorMessage(xhr));
					$button.prop('disabled', false).text(TrustLens.i18n.reset || 'Reset');
				}
			});
		},

		/**
		 * Process the next sync batch via AJAX.
		 * Calls itself recursively until sync completes.
		 */
		processNextBatch: function() {
			var self = this;

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_process_sync_batch',
					nonce: TrustLens.nonce
				},
				success: function(response) {
					if (response.success) {
						var status = response.data.status;
						var progress = response.data.progress;
						var ordersText = self.formatNumber(status.processed_orders) + ' of ' +
							self.formatNumber(status.total_orders) + ' orders processed';
						var fullText = 'Processed ' + ordersText + ' (' + progress + '%)';

						// Update progress bar and text (Data tab has single #wstl-sync-orders-text <p>, dashboard has two spans).
						$('.wstl-progress-fill').css('width', progress + '%');
						$('#wstl-sync-progress-text').text(progress + '%');
						var $ordersEl = $('#wstl-sync-orders-text');
						if ($ordersEl.length) {
							$ordersEl.text($ordersEl.is('span') ? ordersText : fullText);
						}

						if (status.status === 'completed') {
							// Show success message, then reload to full dashboard.
							var customersText = self.formatNumber(status.total_customers) + ' customer' + (status.total_customers !== 1 ? 's' : '');
							var $successMsg = $(
								'<div class="wstl-sync-success">' +
									'<span class="dashicons dashicons-yes-alt"></span>' +
									'<div>' +
										'<strong>' + (TrustLens.i18n.sync_complete || 'Sync complete!') + '</strong>' +
										'<p>' + customersText + ' profiled \u2014 loading your dashboard\u2026</p>' +
									'</div>' +
								'</div>'
							);
							$('#wstl-sync-progress-area').after($successMsg);
							$('#wstl-stop-sync').hide();

							setTimeout(function() {
								location.reload();
							}, 2500);
						} else {
							// Process next batch.
							self.processNextBatch();
						}
					} else {
						self.handleSyncBatchError(response.data.message || TrustLens.i18n.error);
					}
				},
				error: function(xhr) {
					self.handleSyncBatchError(self.getAjaxErrorMessage(xhr));
				}
			});
		},

		/**
		 * Recover sync UI when batch processing fails.
		 *
		 * @param {string} message Error message.
		 */
		handleSyncBatchError: function(message) {
			alert(message || TrustLens.i18n.error);

			$('#wstl-sync-progress-area').hide().addClass('wstl-modal-hidden');
			$('#wstl-sync-start-area').show().removeClass('wstl-modal-hidden');
			$('#wstl-stop-sync').hide().prop('disabled', false).text(TrustLens.i18n.stop_sync || 'Stop Sync');
			$('#wstl-start-sync').show().prop('disabled', false).text(TrustLens.i18n.start_sync || 'Start Sync');
		},


		/**
		 * Format number with commas.
		 *
		 * @param {number} num Number to format.
		 * @return {string} Formatted number.
		 */
		formatNumber: function(num) {
			return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
		},

		/**
		 * Generate API key.
		 *
		 * @param {Event} e Click event.
		 */
		generateApiKey: function(e) {
			e.preventDefault();

			var $button = $(this);
			var isRegenerate = $button.attr('id') === 'wstl-regenerate-api-key';

			if (isRegenerate && !confirm('Regenerating the API key will invalidate the existing key. Continue?')) {
				return;
			}

			$button.prop('disabled', true).text('Generating...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_generate_api_key',
					nonce: TrustLens.nonce
				},
				success: function(response) {
					if (response.success) {
						var key = response.data.api_key;
						if (navigator.clipboard && navigator.clipboard.writeText) {
							navigator.clipboard.writeText(key).then(function() {
								alert(response.data.message + '\n\nThe key has been copied to your clipboard.');
								location.reload();
							}).catch(function() {
								prompt('Your new API key (copy it now - it will not be shown again):', key);
								location.reload();
							});
						} else {
							prompt('Your new API key (copy it now - it will not be shown again):', key);
							location.reload();
						}
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text(isRegenerate ? 'Regenerate' : 'Generate API Key');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text(isRegenerate ? 'Regenerate' : 'Generate API Key');
				}
			});
		},

		/**
		 * Reset all data.
		 *
		 * @param {Event} e Click event.
		 */
		resetData: function(e) {
			e.preventDefault();

			if (!confirm('WARNING: This will permanently delete ALL TrustLens data including customers, events, signals, and sync status. This action cannot be undone.\n\nAre you absolutely sure?')) {
				return;
			}

			// Double confirmation for destructive action.
			if (!confirm('Last chance! Type "DELETE" in your mind and click OK to proceed.')) {
				return;
			}

			var $button = $(this);
			$button.prop('disabled', true).text('Deleting...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_reset_data',
					nonce: TrustLens.nonce
				},
				success: function(response) {
					if (response.success) {
						alert('All data has been deleted.');
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Delete All TrustLens Data');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Delete All TrustLens Data');
				}
			});
		},

		/**
		 * Seed test data (Settings > Data).
		 *
		 * @param {Event} e Click event.
		 */
		seedTestData: function(e) {
			e.preventDefault();

			var $btn = $(this);
			var $spinner = $('#wstl-test-data-spinner');
			var $result = $('#wstl-test-data-result');

			if (!confirm(TrustLens.i18n.confirm_seed || 'Generate 60 test customers? This will add sample data to your database.')) {
				return;
			}

			$btn.prop('disabled', true);
			$spinner.addClass('is-active');
			$result.html('');

			$.post(TrustLens.ajaxUrl, {
				action: 'wstl_seed_test_data',
				nonce: TrustLens.nonce
			}, function(response) {
				$spinner.removeClass('is-active');
				if (response.success) {
					$result.html('<span class="wstl-test-data-success"><span class="dashicons dashicons-yes"></span> ' + response.data.message + '</span>');
					setTimeout(function() {
						location.reload();
					}, 1500);
				} else {
					$btn.prop('disabled', false);
					$result.html('<span class="wstl-test-data-error"><span class="dashicons dashicons-no"></span> ' + (response.data.message || 'Error') + '</span>');
				}
			}).fail(function() {
				$spinner.removeClass('is-active');
				$btn.prop('disabled', false);
				$result.html('<span class="wstl-test-data-error"><span class="dashicons dashicons-no"></span> ' + (TrustLens.i18n.request_failed || 'Request failed.') + '</span>');
			});
		},

		/**
		 * Remove test data (Settings > Data).
		 *
		 * @param {Event} e Click event.
		 */
		removeTestData: function(e) {
			e.preventDefault();

			var $btn = $(this);
			var $spinner = $('#wstl-test-data-spinner');
			var $result = $('#wstl-test-data-result');

			if (!confirm(TrustLens.i18n.confirm_remove || 'Remove all test customers? This action cannot be undone.')) {
				return;
			}

			$btn.prop('disabled', true);
			$spinner.addClass('is-active');
			$result.html('');

			$.post(TrustLens.ajaxUrl, {
				action: 'wstl_remove_test_data',
				nonce: TrustLens.nonce
			}, function(response) {
				$spinner.removeClass('is-active');
				if (response.success) {
					$result.html('<span class="wstl-test-data-success"><span class="dashicons dashicons-yes"></span> ' + response.data.message + '</span>');
					setTimeout(function() {
						location.reload();
					}, 1500);
				} else {
					$btn.prop('disabled', false);
					$result.html('<span class="wstl-test-data-error"><span class="dashicons dashicons-no"></span> ' + (response.data.message || 'Error') + '</span>');
				}
			}).fail(function() {
				$spinner.removeClass('is-active');
				$btn.prop('disabled', false);
				$result.html('<span class="wstl-test-data-error"><span class="dashicons dashicons-no"></span> ' + (TrustLens.i18n.request_failed || 'Request failed.') + '</span>');
			});
		},

		/**
		 * Toggle select all checkboxes.
		 *
		 * @param {Event} e Click event.
		 */
		toggleSelectAll: function(e) {
			var isChecked = $(this).prop('checked');
			$('.wstl-customer-checkbox').prop('checked', isChecked);
			$('#wstl-select-all, #wstl-select-all-bottom').prop('checked', isChecked);
		},

		/**
		 * Apply bulk action.
		 *
		 * @param {Event} e Click event.
		 */
		applyBulkAction: function(e) {
			e.preventDefault();

			var action = $('#wstl-bulk-action-selector').val() || $('#wstl-bulk-action-selector-bottom').val();
			var customers = [];

			$('.wstl-customer-checkbox:checked').each(function() {
				customers.push($(this).val());
			});

			if (!action) {
				alert('Please select a bulk action.');
				return;
			}

			if (customers.length === 0) {
				alert('Please select at least one customer.');
				return;
			}

			// Confirmation for destructive actions.
			if (action === 'delete' && !confirm('Are you sure you want to delete ' + customers.length + ' customer(s)? This cannot be undone.')) {
				return;
			}

			if (action === 'block' && !confirm('Are you sure you want to block ' + customers.length + ' customer(s)?')) {
				return;
			}

			var $button = $(this);
			var $spinner = $button.siblings('.spinner');
			var $result = $('#wstl-bulk-result');

			$button.prop('disabled', true);
			$spinner.addClass('is-active');
			$result.removeClass('success error').text('');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_bulk_action',
					nonce: TrustLens.nonce,
					bulk_action: action,
					customers: customers
				},
				success: function(response) {
					$spinner.removeClass('is-active');
					$button.prop('disabled', false);

					if (response.success) {
						$result.addClass('success').text(response.data.message);
						setTimeout(function() {
							location.reload();
						}, 1500);
					} else {
						$result.addClass('error').text(response.data.message || TrustLens.i18n.error);
					}
				},
				error: function() {
					$spinner.removeClass('is-active');
					$button.prop('disabled', false);
					$result.addClass('error').text(TrustLens.i18n.error);
				}
			});
		},

		/**
		 * Export customers to CSV.
		 *
		 * @param {Event} e Click event.
		 */
		exportCSV: function(e) {
			e.preventDefault();

			var $button = $(this);
			$button.prop('disabled', true).text('Exporting...');

			// Get current filters from URL.
			var urlParams = new URLSearchParams(window.location.search);

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_export_csv',
					nonce: TrustLens.nonce,
					segment: urlParams.get('segment') || '',
					search: urlParams.get('s') || ''
				},
				success: function(response) {
					$button.prop('disabled', false).text('Export CSV');

					if (response.success) {
						// Create download link.
						var blob = new Blob([response.data.csv], { type: 'text/csv' });
						var url = window.URL.createObjectURL(blob);
						var a = document.createElement('a');
						a.href = url;
						a.download = response.data.filename;
						document.body.appendChild(a);
						a.click();
						window.URL.revokeObjectURL(url);
						document.body.removeChild(a);
					} else {
						alert(response.data.message || TrustLens.i18n.error);
					}
				},
				error: function() {
					$button.prop('disabled', false).text('Export CSV');
					alert(TrustLens.i18n.error);
				}
			});
		},

		/**
		 * Recalculate customer score.
		 *
		 * @param {Event} e Click event.
		 */
		recalculateScore: function(e) {
			e.preventDefault();

			var $button = $(this);
			var hash = $button.data('hash');

			$button.prop('disabled', true).text('Calculating...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_recalculate_score',
					nonce: TrustLens.nonce,
					email_hash: hash
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Recalculate Score');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Recalculate Score');
				}
			});
		},

		/**
		 * Save admin note.
		 *
		 * @param {Event} e Click event.
		 */
		saveNote: function(e) {
			e.preventDefault();

			var $button = $(this);
			var hash = $button.data('hash');
			var note = $('#wstl-add-note').val().trim();

			if (!note) {
				alert('Please enter a note.');
				return;
			}

			$button.prop('disabled', true).text('Saving...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_save_note',
					nonce: TrustLens.nonce,
					email_hash: hash,
					note: note
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Add Note');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Add Note');
				}
			});
		},

		/**
		 * Open automation rule modal.
		 *
		 * @param {number} ruleId Optional rule ID for editing.
		 */
		openAutomationModal: function(ruleId) {
			var $modal = $('#wstl-rule-modal');

			// Reset form.
			$modal.find('form')[0].reset();
			$modal.find('#wstl-rule-id').val('');

			if (ruleId) {
				var $row = $('tr[data-rule-id="' + ruleId + '"]');
				$modal.find('#wstl-rule-id').val(ruleId);
				var ruleData = $row.find('.wstl-edit-rule').data('rule');
				if (ruleData && typeof ruleData === 'object') {
					$modal.find('#wstl-rule-name').val(ruleData.name || '');
					$modal.find('#wstl-rule-trigger').val(ruleData.trigger || 'score_updated');
					$modal.find('#wstl-rule-action').val(ruleData.action || 'send_email');
					$modal.find('#wstl-rule-action-value').val(ruleData.action_value || '');
					$modal.find('#wstl-rule-enabled').prop('checked', !!ruleData.enabled);
					// Conditions: repopulate first row; extra rows would need to be re-added by add-condition
					var conditions = ruleData.conditions || [];
					var $container = $modal.find('#wstl-rule-conditions');
					$container.find('.wstl-condition-row').slice(1).remove();
					var $first = $container.find('.wstl-condition-row').first();
					if (conditions.length > 0) {
						$first.find('.wstl-condition-field').val(conditions[0].field || 'trust_score');
						$first.find('.wstl-condition-operator').val(conditions[0].operator || '<');
						$first.find('.wstl-condition-value').val(conditions[0].value || '');
					}
					for (var i = 1; i < conditions.length; i++) {
						// Clone first row and append (simplified; add-condition button may add more)
						var $clone = $first.clone();
						$clone.find('.wstl-condition-field').val(conditions[i].field || 'trust_score');
						$clone.find('.wstl-condition-operator').val(conditions[i].operator || '<');
						$clone.find('.wstl-condition-value').val(conditions[i].value || '');
						$clone.find('select, input').each(function() {
							var $el = $(this);
							var name = $el.attr('name');
							if (name) $el.attr('name', name.replace(/\[0\]/, '[' + i + ']'));
						});
						$container.append($clone);
					}
				}
				$modal.find('#wstl-rule-modal-title').text(TrustLens.i18n.edit_rule || 'Edit Automation Rule');
			} else {
				$modal.find('#wstl-rule-modal-title').text(TrustLens.i18n.add_rule || 'Add Automation Rule');
			}

			$modal.addClass('is-open');
		},

		/**
		 * Save automation rule.
		 *
		 * @param {Event} e Click event.
		 */
		saveAutomationRule: function(e) {
			e.preventDefault();

			var $button = $(this).closest('form').find('#wstl-save-rule');
			var $form = $('#wstl-rule-form');

			var conditions = [];
			$form.find('#wstl-rule-conditions .wstl-condition-row').each(function() {
				var field = $(this).find('.wstl-condition-field').val();
				var op = $(this).find('.wstl-condition-operator').val();
				var val = $(this).find('.wstl-condition-value').val();
				if (field) {
					conditions.push({ field: field, operator: op, value: val });
				}
			});

			$button.prop('disabled', true).text(TrustLens.i18n.saving || 'Saving...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_save_automation_rule',
					nonce: TrustLens.nonce,
					rule_id: $form.find('#wstl-rule-id').val(),
					name: $form.find('#wstl-rule-name').val(),
					trigger: $form.find('#wstl-rule-trigger').val(),
					action: $form.find('#wstl-rule-action').val(),
					action_value: $form.find('#wstl-rule-action-value').val(),
					enabled: $form.find('#wstl-rule-enabled').is(':checked') ? 1 : 0,
					conditions: conditions
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text(TrustLens.i18n.save_rule || 'Save Rule');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text(TrustLens.i18n.save_rule || 'Save Rule');
				}
			});
		},

		/**
		 * Delete automation rule.
		 *
		 * @param {Event} e Click event.
		 */
		deleteAutomationRule: function(e) {
			e.preventDefault();

			if (!confirm(TrustLens.i18n.confirm_delete_rule || 'Are you sure you want to delete this automation rule?')) {
				return;
			}

			var $button = $(this);
			var ruleId = $button.closest('tr').data('rule-id');

			$button.prop('disabled', true);

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_delete_automation_rule',
					nonce: TrustLens.nonce,
					rule_id: ruleId
				},
				success: function(response) {
					if (response.success) {
						$button.closest('tr').fadeOut(function() {
							$(this).remove();
						});
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false);
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false);
				}
			});
		},

		/**
		 * Add a condition row to the rule form.
		 */
		addAutomationCondition: function() {
			var $container = $('#wstl-rule-conditions');
			var $first = $container.find('.wstl-condition-row').first();
			var index = $container.find('.wstl-condition-row').length;
			var $clone = $first.clone();
			$clone.find('input, select').val('');
			$clone.find('select.wstl-condition-field').val('trust_score');
			$clone.find('select.wstl-condition-operator').val('<');
			$clone.find('[name^="conditions["]').each(function() {
				var $el = $(this);
				$el.attr('name', $el.attr('name').replace(/\[\d+\]/, '[' + index + ']'));
			});
			$container.append($clone);
		},

		/**
		 * Remove a condition row (keep at least one).
		 */
		removeAutomationCondition: function() {
			var $container = $('#wstl-rule-conditions');
			if ($container.find('.wstl-condition-row').length <= 1) return;
			$(this).closest('.wstl-condition-row').remove();
			$container.find('.wstl-condition-row').each(function(i) {
				$(this).find('[name^="conditions["]').each(function() {
					var $el = $(this);
					$el.attr('name', $el.attr('name').replace(/\[\d+\]/, '[' + i + ']'));
				});
			});
		},

		/**
		 * Toggle automation rule enabled state.
		 */
		toggleAutomationRuleEnabled: function() {
			var $checkbox = $(this);
			var $row = $checkbox.closest('tr');
			var ruleId = $row.data('rule-id');
			if (!ruleId) return;

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_update_automation_rule_enabled',
					nonce: TrustLens.nonce,
					rule_id: ruleId,
					enabled: $checkbox.is(':checked') ? 1 : 0
				},
				success: function(response) {
					if (!response.success) {
						alert(response.data.message || TrustLens.i18n.error);
						$checkbox.prop('checked', !$checkbox.is(':checked'));
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$checkbox.prop('checked', !$checkbox.is(':checked'));
				}
			});
		},

		/**
		 * Open webhook modal.
		 */
		openWebhookModal: function() {
			var $modal = $('#wstl-webhook-modal');
			$modal.find('form')[0].reset();
			$modal.addClass('is-open');
		},

		/**
		 * Save webhook.
		 *
		 * @param {Event} e Click event.
		 */
		saveWebhook: function(e) {
			e.preventDefault();

			var $button = $(this);
			var $form = $button.closest('form');

			// Get selected events.
			var events = [];
			$form.find('input[name="webhook_events[]"]:checked').each(function() {
				events.push($(this).val());
			});

			if (events.length === 0) {
				alert('Please select at least one event.');
				return;
			}

			$button.prop('disabled', true).text('Saving...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_add_webhook',
					nonce: TrustLens.nonce,
					name: $form.find('#wstl-webhook-name').val(),
					url: $form.find('#wstl-webhook-url').val(),
					events: events,
					enabled: $form.find('#wstl-webhook-enabled').is(':checked') ? 1 : 0
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Save Webhook');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Save Webhook');
				}
			});
		},

		/**
		 * Test webhook.
		 *
		 * @param {Event} e Click event.
		 */
		testWebhook: function(e) {
			e.preventDefault();

			var $button = $(this);
			var webhookId = $button.data('webhook-id');

			$button.prop('disabled', true).text('Testing...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_test_webhook',
					nonce: TrustLens.nonce,
					webhook_id: webhookId
				},
				success: function(response) {
					$button.prop('disabled', false).text('Test');

					if (response.success) {
						alert('Webhook test sent successfully! Response code: ' + response.data.response_code);
					} else {
						alert('Webhook test failed: ' + (response.data.message || TrustLens.i18n.error));
					}
				},
				error: function() {
					$button.prop('disabled', false).text('Test');
					alert(TrustLens.i18n.error);
				}
			});
		},

		/**
		 * Delete webhook.
		 *
		 * @param {Event} e Click event.
		 */
		deleteWebhook: function(e) {
			e.preventDefault();

			if (!confirm('Are you sure you want to delete this webhook?')) {
				return;
			}

			var $button = $(this);
			var webhookId = $button.data('webhook-id');

			$button.prop('disabled', true);

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_delete_webhook',
					nonce: TrustLens.nonce,
					webhook_id: webhookId
				},
				success: function(response) {
					if (response.success) {
						$button.closest('tr').fadeOut(function() {
							$(this).remove();
						});
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false);
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false);
				}
			});
		},

		/**
		 * Open report schedule modal.
		 */
		openReportModal: function() {
			var $modal = $('#wstl-report-modal');
			$modal.find('form')[0].reset();
			$modal.addClass('is-open');
		},

		/**
		 * Save report schedule.
		 *
		 * @param {Event} e Click event.
		 */
		saveReportSchedule: function(e) {
			e.preventDefault();

			var $button = $(this);
			var $form = $button.closest('form');

			$button.prop('disabled', true).text('Saving...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_add_report_schedule',
					nonce: TrustLens.nonce,
					name: $form.find('#wstl-report-name').val(),
					type: $form.find('#wstl-report-type').val(),
					frequency: $form.find('#wstl-report-frequency').val(),
					recipients: $form.find('#wstl-report-recipients').val(),
					enabled: $form.find('#wstl-report-enabled').is(':checked') ? 1 : 0
				},
				success: function(response) {
					if (response.success) {
						location.reload();
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false).text('Save Schedule');
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false).text('Save Schedule');
				}
			});
		},

		/**
		 * Send report now.
		 *
		 * @param {Event} e Click event.
		 */
		sendReportNow: function(e) {
			e.preventDefault();

			var $button = $(this);
			var scheduleId = $button.data('schedule-id');

			$button.prop('disabled', true).text('Sending...');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_send_report_now',
					nonce: TrustLens.nonce,
					schedule_id: scheduleId
				},
				success: function(response) {
					$button.prop('disabled', false).text('Send Now');

					if (response.success) {
						alert('Report sent successfully!');
					} else {
						alert('Failed to send report: ' + (response.data.message || TrustLens.i18n.error));
					}
				},
				error: function() {
					$button.prop('disabled', false).text('Send Now');
					alert(TrustLens.i18n.error);
				}
			});
		},

		/**
		 * Send test notification email.
		 *
		 * @param {Event} e Click event.
		 */
		sendTestNotification: function(e) {
			e.preventDefault();

			var $button = $(this);
			var $spinner = $button.next('.spinner');
			var $result = $('#wstl-test-notification-result');

			$button.prop('disabled', true);
			$spinner.addClass('is-active');
			$result.text('').removeClass('success error');

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_send_test_notification',
					nonce: TrustLens.nonce
				},
				timeout: 15000,
				success: function(response) {
					$button.prop('disabled', false);
					$spinner.removeClass('is-active');

					if (response.success) {
						$result.text(response.data.message).addClass('success');
					} else {
						$result.text(response.data.message || TrustLens.i18n.error).addClass('error');
					}
				},
				error: function(xhr, status) {
					$button.prop('disabled', false);
					$spinner.removeClass('is-active');
					if (status === 'timeout') {
						$result.text(TrustLens.i18n.test_notification_timeout || 'Request timed out. Your server may not be sending mail (e.g. SMTP not configured). Try an SMTP plugin or check your server mail settings.').addClass('error');
					} else {
						$result.text(TrustLens.i18n.error).addClass('error');
					}
				}
			});
		},

		/**
		 * Delete report schedule.
		 *
		 * @param {Event} e Click event.
		 */
		deleteReportSchedule: function(e) {
			e.preventDefault();

			if (!confirm('Are you sure you want to delete this scheduled report?')) {
				return;
			}

			var $button = $(this);
			var scheduleId = $button.data('schedule-id');

			$button.prop('disabled', true);

			$.ajax({
				url: TrustLens.ajaxUrl,
				type: 'POST',
				data: {
					action: 'wstl_delete_report_schedule',
					nonce: TrustLens.nonce,
					schedule_id: scheduleId
				},
				success: function(response) {
					if (response.success) {
						$button.closest('tr').fadeOut(function() {
							$(this).remove();
						});
					} else {
						alert(response.data.message || TrustLens.i18n.error);
						$button.prop('disabled', false);
					}
				},
				error: function() {
					alert(TrustLens.i18n.error);
					$button.prop('disabled', false);
				}
			});
		},

		/**
		 * Close modal.
		 *
		 * @param {Event} e Click event.
		 */
		closeModal: function(e) {
			if (e) {
				// Only close if clicking overlay, close button, or cancel button.
				if (!$(e.target).hasClass('wstl-modal-overlay') && !$(e.target).hasClass('wstl-modal-close') && !$(e.target).hasClass('wstl-modal-cancel') && !$(e.target).closest('.wstl-modal-close').length && !$(e.target).closest('.wstl-modal-cancel').length) {
					return;
				}
			}
			$('.wstl-modal').removeClass('is-open');
		}
	};

	// Initialize on document ready.
	$(document).ready(function() {
		TrustLensAdmin.init();
	});

})(jQuery);
