<?php
/**
 * Admin functionality.
 *
 * @package TrustLens\Admin
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Admin class.
 *
 * @since 1.0.0
 */
class TrustLens_Admin {
	/**
	 * User meta key: auto-update notice dismissed.
	 *
	 * @var string
	 */
	private const AUTO_UPDATE_NOTICE_DISMISSED_META = 'trustlens_auto_update_notice_dismissed';

	/**
	 * User meta key: review notice dismissed permanently.
	 *
	 * @var string
	 */
	private const REVIEW_NOTICE_DISMISSED_META = 'trustlens_review_notice_dismissed';

	/**
	 * User meta key: review notice remind later timestamp.
	 *
	 * @var string
	 */
	private const REVIEW_NOTICE_REMIND_LATER_META = 'trustlens_review_notice_remind_later';

	/**
	 * Minimum days installed before review notice is eligible.
	 *
	 * @var int
	 */
	private const REVIEW_NOTICE_MIN_DAYS_INSTALLED = 7;

	/**
	 * Minimum engaged customers before review notice is eligible.
	 *
	 * @var int
	 */
	private const REVIEW_NOTICE_MIN_CUSTOMERS = 3;

	/**
	 * Single instance.
	 *
	 * @var TrustLens_Admin|null
	 */
	private static ?TrustLens_Admin $instance = null;

	/**
	 * Get instance.
	 *
	 * @since 1.0.0
	 * @return TrustLens_Admin
	 */
	public static function instance(): TrustLens_Admin {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks(): void {
		add_action( 'admin_init', array( $this, 'activation_redirect' ) );
		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'admin_notices', array( $this, 'render_promotional_notices' ) );
		add_action( 'wp_dashboard_setup', array( $this, 'register_dashboard_widget' ) );

		// AJAX handlers.
		add_action( 'wp_ajax_wstl_block_customer', array( $this, 'ajax_block_customer' ) );
		add_action( 'wp_ajax_wstl_unblock_customer', array( $this, 'ajax_unblock_customer' ) );
		add_action( 'wp_ajax_wstl_allowlist_customer', array( $this, 'ajax_allowlist_customer' ) );
		add_action( 'wp_ajax_wstl_remove_allowlist', array( $this, 'ajax_remove_allowlist' ) );

		// Sync AJAX handlers.
		add_action( 'wp_ajax_wstl_start_sync', array( $this, 'ajax_start_sync' ) );
		add_action( 'wp_ajax_wstl_process_sync_batch', array( $this, 'ajax_process_sync_batch' ) );
		add_action( 'wp_ajax_wstl_stop_sync', array( $this, 'ajax_stop_sync' ) );
		add_action( 'wp_ajax_wstl_reset_sync', array( $this, 'ajax_reset_sync' ) );

		// API key generation.
		add_action( 'wp_ajax_wstl_generate_api_key', array( $this, 'ajax_generate_api_key' ) );

		// Phase 3 AJAX handlers.
		add_action( 'wp_ajax_wstl_bulk_action', array( $this, 'ajax_bulk_action' ) );
		add_action( 'wp_ajax_wstl_recalculate_score', array( $this, 'ajax_recalculate_score' ) );
		add_action( 'wp_ajax_wstl_save_note', array( $this, 'ajax_save_note' ) );
		add_action( 'wp_ajax_wstl_export_csv', array( $this, 'ajax_export_csv' ) );
		if ( wstl_has_automation() && wstl_can_use_pro() ) {
			add_action( 'wp_ajax_wstl_save_automation_rule', array( $this, 'ajax_save_automation_rule' ) );
			add_action( 'wp_ajax_wstl_delete_automation_rule', array( $this, 'ajax_delete_automation_rule' ) );
			add_action( 'wp_ajax_wstl_update_automation_rule_enabled', array( $this, 'ajax_update_automation_rule_enabled' ) );
		}
		if ( wstl_has_webhooks() && wstl_can_use_pro() ) {
			add_action( 'wp_ajax_wstl_add_webhook', array( $this, 'ajax_add_webhook' ) );
			add_action( 'wp_ajax_wstl_delete_webhook', array( $this, 'ajax_delete_webhook' ) );
			add_action( 'wp_ajax_wstl_test_webhook', array( $this, 'ajax_test_webhook' ) );
		}
		if ( wstl_has_scheduled_reports() && wstl_can_use_pro() ) {
			add_action( 'wp_ajax_wstl_add_report_schedule', array( $this, 'ajax_add_report_schedule' ) );
			add_action( 'wp_ajax_wstl_delete_report_schedule', array( $this, 'ajax_delete_report_schedule' ) );
			add_action( 'wp_ajax_wstl_send_report_now', array( $this, 'ajax_send_report_now' ) );
		}
		add_action( 'wp_ajax_wstl_reset_data', array( $this, 'ajax_reset_data' ) );
		add_action( 'wp_ajax_wstl_send_test_notification', array( $this, 'ajax_send_test_notification' ) );
		add_action( 'wp_ajax_wstl_dismiss_auto_update_notice', array( $this, 'ajax_dismiss_auto_update_notice' ) );
		add_action( 'wp_ajax_wstl_enable_auto_updates', array( $this, 'ajax_enable_auto_updates' ) );
		add_action( 'wp_ajax_wstl_dismiss_review_notice', array( $this, 'ajax_dismiss_review_notice' ) );
		add_action( 'wp_ajax_wstl_remind_later_review_notice', array( $this, 'ajax_remind_later_review_notice' ) );

		// Order metabox.
		add_action( 'add_meta_boxes', array( $this, 'register_order_metabox' ) );

		// Settings.
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Redirect to dashboard on plugin activation.
	 *
	 * @since 1.0.0
	 */
	public function activation_redirect(): void {
		if ( ! get_transient( 'trustlens_activated' ) ) {
			return;
		}

		delete_transient( 'trustlens_activated' );

		// Don't redirect on multi-site bulk activation, AJAX, or CLI.
		if ( wp_doing_ajax() || is_network_admin() || isset( $_GET['activate-multi'] ) || ( defined( 'WP_CLI' ) && WP_CLI ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for bulk activation flag.
			return;
		}

		wp_safe_redirect( admin_url( 'admin.php?page=trustlens' ) );
		exit;
	}

	/**
	 * Register admin menu.
	 *
	 * @since 1.0.0
	 */
	public function register_menu(): void {
		add_menu_page(
			__( 'TrustLens', 'trustlens' ),
			__( 'TrustLens', 'trustlens' ),
			'manage_woocommerce',
			'trustlens',
			array( $this, 'render_dashboard_page' ),
			'dashicons-visibility',
			56
		);

		add_submenu_page(
			'trustlens',
			__( 'Dashboard', 'trustlens' ),
			__( 'Dashboard', 'trustlens' ),
			'manage_woocommerce',
			'trustlens',
			array( $this, 'render_dashboard_page' )
		);

		add_submenu_page(
			'trustlens',
			__( 'Customers', 'trustlens' ),
			__( 'Customers', 'trustlens' ),
			'manage_woocommerce',
			'trustlens-customers',
			array( $this, 'render_customers_page' )
		);

		add_submenu_page(
			'trustlens',
			__( 'Notifications', 'trustlens' ),
			__( 'Notifications', 'trustlens' ),
			'manage_woocommerce',
			'trustlens-notifications',
			array( $this, 'render_notifications_page' )
		);

		add_submenu_page(
			'trustlens',
			__( 'Automation', 'trustlens' ),
			__( 'Automation', 'trustlens' ),
			'manage_woocommerce',
			'trustlens-automation',
			array( $this, 'render_automation_page' )
		);

		add_submenu_page(
			'trustlens',
			__( 'Settings', 'trustlens' ),
			__( 'Settings', 'trustlens' ),
			'manage_woocommerce',
			'trustlens-settings',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Enqueue admin assets.
	 *
	 * WordPress.org: All admin JS/CSS must be added via wp_enqueue_script/style or
	 * wp_add_inline_script/wp_add_inline_style. Do not output raw <style> or <script> in view files.
	 *
	 * @since 1.0.0
	 * @param string $hook Current admin page.
	 */
	public function enqueue_assets( string $hook ): void {
		// Only on TrustLens pages.
		if ( strpos( $hook, 'trustlens' ) === false && $hook !== 'index.php' ) {
			return;
		}

		$admin_css_path = TRUSTLENS_PLUGIN_DIR . 'admin/css/admin.css';
		$dashboard_css_path = TRUSTLENS_PLUGIN_DIR . 'admin/css/dashboard.css';
		$customers_css_path = TRUSTLENS_PLUGIN_DIR . 'admin/css/customers.css';
		$customer_detail_css_path = TRUSTLENS_PLUGIN_DIR . 'admin/css/customer-detail.css';
		$settings_css_path = TRUSTLENS_PLUGIN_DIR . 'admin/css/settings.css';
		$notifications_css_path = TRUSTLENS_PLUGIN_DIR . 'admin/css/notifications.css';
		$pro_upsell_css_path = TRUSTLENS_PLUGIN_DIR . 'admin/css/pro-upsell.css';
		$admin_js_path  = TRUSTLENS_PLUGIN_DIR . 'admin/js/admin.js';
		$admin_css_ver  = file_exists( $admin_css_path ) ? (string) filemtime( $admin_css_path ) : TRUSTLENS_VERSION;
		$dashboard_css_ver = file_exists( $dashboard_css_path ) ? (string) filemtime( $dashboard_css_path ) : TRUSTLENS_VERSION;
		$customers_css_ver = file_exists( $customers_css_path ) ? (string) filemtime( $customers_css_path ) : TRUSTLENS_VERSION;
		$customer_detail_css_ver = file_exists( $customer_detail_css_path ) ? (string) filemtime( $customer_detail_css_path ) : TRUSTLENS_VERSION;
		$settings_css_ver = file_exists( $settings_css_path ) ? (string) filemtime( $settings_css_path ) : TRUSTLENS_VERSION;
		$notifications_css_ver = file_exists( $notifications_css_path ) ? (string) filemtime( $notifications_css_path ) : TRUSTLENS_VERSION;
		$pro_upsell_css_ver = file_exists( $pro_upsell_css_path ) ? (string) filemtime( $pro_upsell_css_path ) : TRUSTLENS_VERSION;
		$admin_js_ver   = file_exists( $admin_js_path ) ? (string) filemtime( $admin_js_path ) : TRUSTLENS_VERSION;

		wp_enqueue_style(
			'trustlens-admin',
			TRUSTLENS_PLUGIN_URL . 'admin/css/admin.css',
			array(),
			$admin_css_ver
		);

		// Freemius may render admin pointers on plugin pages; ensure core pointer assets are present.
		wp_enqueue_style( 'wp-pointer' );
		wp_enqueue_script( 'wp-pointer' );

		// Load shared Pro upsell styles across TrustLens admin pages.
		wp_enqueue_style(
			'trustlens-pro-upsell',
			TRUSTLENS_PLUGIN_URL . 'admin/css/pro-upsell.css',
			array( 'trustlens-admin' ),
			$pro_upsell_css_ver
		);

		// Load dashboard-specific styles only on the dashboard page (not automation, which has its own enqueue).
		if ( strpos( $hook, 'page_trustlens' ) !== false && strpos( $hook, 'customers' ) === false && strpos( $hook, 'settings' ) === false && strpos( $hook, 'notifications' ) === false && strpos( $hook, 'automation' ) === false ) {
			wp_enqueue_style(
				'trustlens-dashboard',
				TRUSTLENS_PLUGIN_URL . 'admin/css/dashboard.css',
				array( 'trustlens-admin' ),
				$dashboard_css_ver
			);
		}

		// Load customers page styles only on the customers list/detail page.
		if ( strpos( $hook, 'trustlens-customers' ) !== false ) {
			wp_enqueue_style(
				'trustlens-customers',
				TRUSTLENS_PLUGIN_URL . 'admin/css/customers.css',
				array( 'trustlens-admin' ),
				$customers_css_ver
			);

			// Load customer detail-only styles for the single customer view.
			if ( isset( $_GET['customer'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display-only query parameter.
				wp_enqueue_style(
					'trustlens-customer-detail',
					TRUSTLENS_PLUGIN_URL . 'admin/css/customer-detail.css',
					array( 'trustlens-admin', 'trustlens-customers' ),
					$customer_detail_css_ver
				);
			}
		}

		// Load settings-specific styles only on TrustLens settings pages.
		if ( strpos( $hook, 'trustlens-settings' ) !== false ) {
			wp_enqueue_style(
				'trustlens-settings',
				TRUSTLENS_PLUGIN_URL . 'admin/css/settings.css',
				array( 'trustlens-admin' ),
				$settings_css_ver
			);
		}

		// Load notifications-specific styles only on TrustLens notifications pages.
		if ( strpos( $hook, 'trustlens-notifications' ) !== false ) {
			wp_enqueue_style(
				'trustlens-notifications',
				TRUSTLENS_PLUGIN_URL . 'admin/css/notifications.css',
				array( 'trustlens-admin' ),
				$notifications_css_ver
			);
		}

		// Load dashboard + settings styles on Automation page only when Pro UI is shown (not for upsell-only).
		if ( strpos( $hook, 'trustlens-automation' ) !== false
			&& wstl_has_automation()
			&& wstl_can_use_pro()
			&& file_exists( TRUSTLENS_PLUGIN_DIR . 'admin/views/automation-pro.php' ) ) {
			wp_enqueue_style(
				'trustlens-dashboard',
				TRUSTLENS_PLUGIN_URL . 'admin/css/dashboard.css',
				array( 'trustlens-admin' ),
				$dashboard_css_ver
			);
			wp_enqueue_style(
				'trustlens-settings',
				TRUSTLENS_PLUGIN_URL . 'admin/css/settings.css',
				array( 'trustlens-admin' ),
				$settings_css_ver
			);
		}

		// Data settings tab: dynamic sync progress width via wp_add_inline_style (no inline style= in view).
		if ( strpos( $hook, 'trustlens-settings' ) !== false && isset( $_GET['tab'] ) && 'data' === $_GET['tab'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- display only
			$sync = TrustLens_Historical_Sync::instance();
			$sync_status = $sync->get_status();
			if ( 'running' === $sync_status['status'] ) {
				$progress = $sync->get_progress();
				wp_add_inline_style( 'trustlens-admin', '#wstl-sync-status .wstl-progress-fill { width: ' . (float) $progress . '% !important; }' );
			}
		}

		// Enqueue Chart.js and dashboard charts for dashboard page (bundled locally for WordPress.org compliance).
		// Note for reviewers: Chart.js is updated with each plugin release. A newer Chart.js version may be released
		// during the review period; we will update the bundled library in the next plugin release if needed.
		if ( strpos( $hook, 'page_trustlens' ) !== false && strpos( $hook, 'customers' ) === false && strpos( $hook, 'settings' ) === false && strpos( $hook, 'notifications' ) === false && strpos( $hook, 'automation' ) === false ) {
			wp_enqueue_script(
				'chartjs',
				TRUSTLENS_PLUGIN_URL . 'admin/vendor/chart-js/chart.umd.min.js',
				array(),
				'4.5.1',
				true
			);

			wp_enqueue_script(
				'trustlens-dashboard-charts',
				TRUSTLENS_PLUGIN_URL . 'admin/js/dashboard-charts.js',
				array( 'chartjs' ),
				TRUSTLENS_VERSION,
				true
			);
		}

		wp_enqueue_script(
			'trustlens-admin',
			TRUSTLENS_PLUGIN_URL . 'admin/js/admin.js',
			array( 'jquery' ),
			$admin_js_ver,
			true
		);

		wp_localize_script( 'trustlens-admin', 'TrustLens', array(
			'ajaxUrl' => admin_url( 'admin-ajax.php' ),
			'nonce'   => wp_create_nonce( 'trustlens_admin' ),
			'i18n'    => array(
				'confirm_block'       => __( 'Are you sure you want to block this customer?', 'trustlens' ),
				'confirm_allowlist'   => __( 'Add this customer to the allowlist?', 'trustlens' ),
				'confirm_reset'       => __( 'Are you sure you want to delete all TrustLens data? This cannot be undone.', 'trustlens' ),
				'confirm_delete'      => __( 'Are you sure you want to delete the selected customers? This cannot be undone.', 'trustlens' ),
				'confirm_bulk'        => __( 'Apply this action to the selected customers?', 'trustlens' ),
				'select_customers'    => __( 'Please select at least one customer.', 'trustlens' ),
				'select_action'       => __( 'Please select an action.', 'trustlens' ),
				'success'             => __( 'Action completed successfully.', 'trustlens' ),
				'error'                     => __( 'An error occurred. Please try again.', 'trustlens' ),
				'test_notification_timeout' => __( 'Request timed out. Your server may not be sending mail (e.g. SMTP not configured). Try an SMTP plugin or check your server mail settings.', 'trustlens' ),
				'starting'            => __( 'Starting...', 'trustlens' ),
				'stopping'            => __( 'Stopping...', 'trustlens' ),
				'resetting'           => __( 'Resetting...', 'trustlens' ),
				'start_sync'          => __( 'Start Sync', 'trustlens' ),
				'stop_sync'           => __( 'Stop Sync', 'trustlens' ),
				'sync_complete'       => __( 'Sync complete!', 'trustlens' ),
				'processing'          => __( 'Processing...', 'trustlens' ),
				'customers_processed' => __( 'customers processed', 'trustlens' ),
				'confirm_seed'        => __( 'Generate 60 test customers? This will add sample data to your database.', 'trustlens' ),
				'confirm_remove'     => __( 'Remove all test customers? This action cannot be undone.', 'trustlens' ),
				'request_failed'     => __( 'Request failed.', 'trustlens' ),
				'add_rule'           => __( 'Add Automation Rule', 'trustlens' ),
				'edit_rule'          => __( 'Edit Automation Rule', 'trustlens' ),
				'save_rule'          => __( 'Save Rule', 'trustlens' ),
				'saving'             => __( 'Saving...', 'trustlens' ),
				'confirm_delete_rule' => __( 'Are you sure you want to delete this automation rule?', 'trustlens' ),
			),
		) );
	}

	/**
	 * Render dashboard page.
	 *
	 * @since 1.0.0
	 */
	public function render_dashboard_page(): void {
		include TRUSTLENS_PLUGIN_DIR . 'admin/views/dashboard.php';
	}

	/**
	 * Render customers page.
	 *
	 * @since 1.0.0
	 */
	public function render_customers_page(): void {
		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Display-only query parameters for admin page listing, no data changes.
		// Handle single customer view.
		if ( isset( $_GET['customer'] ) ) {
			$this->render_customer_detail_page();
			return;
		}

		$repo = new TrustLens_Customer_Repository();

		$per_page = 20;
		$page = isset( $_GET['paged'] ) ? absint( wp_unslash( $_GET['paged'] ) ) : 1;
		$segment = isset( $_GET['segment'] ) ? sanitize_text_field( wp_unslash( $_GET['segment'] ) ) : '';
		$search = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';
		$orderby = isset( $_GET['orderby'] ) ? sanitize_text_field( wp_unslash( $_GET['orderby'] ) ) : 'trust_score';
		$order = isset( $_GET['order'] ) ? sanitize_text_field( wp_unslash( $_GET['order'] ) ) : 'ASC';

		$customers = $repo->get_customers( array(
			'per_page' => $per_page,
			'page'     => $page,
			'segment'  => $segment,
			'search'   => $search,
			'orderby'  => $orderby,
			'order'    => $order,
		) );

		$total = $repo->count( array(
			'segment' => $segment,
			'search'  => $search,
		) );

		$total_pages = ceil( $total / $per_page );
		$segment_counts = $repo->get_segment_counts();

		include TRUSTLENS_PLUGIN_DIR . 'admin/views/customers.php';
		// phpcs:enable WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Render customer detail page.
	 *
	 * @since 1.0.0
	 */
	private function render_customer_detail_page(): void {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display-only query parameter for admin page, no data changes.
		$email_hash = isset( $_GET['customer'] ) ? sanitize_text_field( wp_unslash( $_GET['customer'] ) ) : '';
		$customer = wstl_get_customer( $email_hash );

		if ( ! $customer ) {
			wp_die( esc_html__( 'Customer not found.', 'trustlens' ) );
		}

		// Get signals.
		$signals = TrustLens_Score_Calculator::get_signals( $email_hash );

		// Get recent events.
		global $wpdb;
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		$events = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}trustlens_events
			 WHERE email_hash = %s
			 ORDER BY created_at DESC
			 LIMIT 20",
			$email_hash
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter

		include TRUSTLENS_PLUGIN_DIR . 'admin/views/customer-detail.php';
	}

	/**
	 * Render notifications page.
	 *
	 * @since 1.1.0
	 */
	public function render_notifications_page(): void {
		include TRUSTLENS_PLUGIN_DIR . 'admin/views/notifications.php';
	}

	/**
	 * Render settings page.
	 *
	 * @since 1.0.0
	 */
	public function render_settings_page(): void {
		include TRUSTLENS_PLUGIN_DIR . 'admin/views/settings.php';
	}

	/**
	 * Render automation page.
	 *
	 * @since 1.0.0
	 */
	public function render_automation_page(): void {
		include TRUSTLENS_PLUGIN_DIR . 'admin/views/automation.php';
	}

	/**
	 * Render admin promotional notices (auto-update + review prompts).
	 *
	 * @since 1.1.1
	 */
	public function render_promotional_notices(): void {
		if ( ! $this->is_relevant_notice_screen() ) {
			return;
		}

		$this->maybe_render_auto_update_notice();
		$this->maybe_render_review_notice();
	}

	/**
	 * Check if the current admin screen is relevant for notice rendering.
	 *
	 * @since 1.1.1
	 * @return bool
	 */
	private function is_relevant_notice_screen(): bool {
		$screen = get_current_screen();
		if ( ! $screen ) {
			return false;
		}

		if ( 'plugins' === $screen->id ) {
			return true;
		}

		return strpos( $screen->id, 'trustlens' ) !== false;
	}

	/**
	 * Render auto-update notice when eligible.
	 *
	 * @since 1.1.1
	 */
	private function maybe_render_auto_update_notice(): void {
		if ( ! current_user_can( 'update_plugins' ) ) {
			return;
		}

		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return;
		}

		if ( get_user_meta( $user_id, self::AUTO_UPDATE_NOTICE_DISMISSED_META, true ) ) {
			return;
		}

		if ( $this->is_trustlens_auto_update_enabled() || ! $this->are_plugin_auto_updates_available() ) {
			return;
		}

		$nonce = wp_create_nonce( 'wstl_auto_update_notice' );
		?>
		<div class="notice notice-info is-dismissible" id="wstl-auto-update-notice" data-nonce="<?php echo esc_attr( $nonce ); ?>">
			<p><strong><?php esc_html_e( 'Keep TrustLens up to date automatically', 'trustlens' ); ?></strong></p>
			<p><?php esc_html_e( 'Enable auto-updates to receive the latest fixes and security improvements automatically.', 'trustlens' ); ?></p>
			<p>
				<button type="button" class="button button-primary" id="wstl-enable-auto-updates"><?php esc_html_e( 'Enable Auto-Updates', 'trustlens' ); ?></button>
				<button type="button" class="button button-link" id="wstl-dismiss-auto-update-notice"><?php esc_html_e( "Don't show again", 'trustlens' ); ?></button>
			</p>
		</div>
		<script>
		(function() {
			var notice = document.getElementById('wstl-auto-update-notice');
			if (!notice) return;
			var nonce = notice.getAttribute('data-nonce');
			function postAction(action) {
				var body = new URLSearchParams();
				body.append('action', action);
				body.append('nonce', nonce);
				fetch(ajaxurl, { method: 'POST', credentials: 'same-origin', headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' }, body: body.toString() })
					.finally(function() { notice.style.display = 'none'; });
			}
			var enableBtn = document.getElementById('wstl-enable-auto-updates');
			if (enableBtn) {
				enableBtn.addEventListener('click', function() { postAction('wstl_enable_auto_updates'); });
			}
			var dismissBtn = document.getElementById('wstl-dismiss-auto-update-notice');
			if (dismissBtn) {
				dismissBtn.addEventListener('click', function() { postAction('wstl_dismiss_auto_update_notice'); });
			}
			notice.addEventListener('click', function(e) {
				if (e.target && e.target.classList && e.target.classList.contains('notice-dismiss')) {
					postAction('wstl_dismiss_auto_update_notice');
				}
			});
		})();
		</script>
		<?php
	}

	/**
	 * Render review notice when eligible.
	 *
	 * @since 1.1.1
	 */
	private function maybe_render_review_notice(): void {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return;
		}

		if ( get_user_meta( $user_id, self::REVIEW_NOTICE_DISMISSED_META, true ) ) {
			return;
		}

		$remind_later_until = (int) get_user_meta( $user_id, self::REVIEW_NOTICE_REMIND_LATER_META, true );
		if ( $remind_later_until > time() ) {
			return;
		}

		$installed_at = (int) get_option( 'trustlens_installed_timestamp', 0 );
		if ( $installed_at <= 0 ) {
			return;
		}

		$days_installed = (int) floor( ( time() - $installed_at ) / DAY_IN_SECONDS );
		if ( $days_installed < self::REVIEW_NOTICE_MIN_DAYS_INSTALLED ) {
			return;
		}

		$sync_status = TrustLens_Historical_Sync::instance()->get_status();
		$engaged_customers = (int) ( $sync_status['total_customers'] ?? 0 );
		if ( $engaged_customers < self::REVIEW_NOTICE_MIN_CUSTOMERS ) {
			return;
		}

		$dismiss_nonce = wp_create_nonce( 'wstl_review_notice' );
		$review_url = 'https://wordpress.org/support/plugin/trustlens/reviews/#new-post';
		?>
		<div class="notice notice-success is-dismissible" id="wstl-review-notice" data-nonce="<?php echo esc_attr( $dismiss_nonce ); ?>">
			<p><strong><?php esc_html_e( 'Enjoying TrustLens?', 'trustlens' ); ?></strong></p>
			<p><?php esc_html_e( 'If TrustLens is helping you protect your store, a quick review helps us a lot.', 'trustlens' ); ?></p>
			<p>
				<a class="button button-primary" href="<?php echo esc_url( $review_url ); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Leave a Review', 'trustlens' ); ?></a>
				<button type="button" class="button" id="wstl-review-remind-later"><?php esc_html_e( 'Remind me later', 'trustlens' ); ?></button>
				<button type="button" class="button button-link" id="wstl-review-dismiss"><?php esc_html_e( "Don't show again", 'trustlens' ); ?></button>
			</p>
		</div>
		<script>
		(function() {
			var notice = document.getElementById('wstl-review-notice');
			if (!notice) return;
			var nonce = notice.getAttribute('data-nonce');
			function postAction(action) {
				var body = new URLSearchParams();
				body.append('action', action);
				body.append('nonce', nonce);
				fetch(ajaxurl, { method: 'POST', credentials: 'same-origin', headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' }, body: body.toString() })
					.finally(function() { notice.style.display = 'none'; });
			}
			var remindBtn = document.getElementById('wstl-review-remind-later');
			if (remindBtn) {
				remindBtn.addEventListener('click', function() { postAction('wstl_remind_later_review_notice'); });
			}
			var dismissBtn = document.getElementById('wstl-review-dismiss');
			if (dismissBtn) {
				dismissBtn.addEventListener('click', function() { postAction('wstl_dismiss_review_notice'); });
			}
			notice.addEventListener('click', function(e) {
				if (e.target && e.target.classList && e.target.classList.contains('notice-dismiss')) {
					postAction('wstl_dismiss_review_notice');
				}
			});
		})();
		</script>
		<?php
	}

	/**
	 * Check whether auto-updates are enabled for TrustLens.
	 *
	 * @since 1.1.1
	 * @return bool
	 */
	private function is_trustlens_auto_update_enabled(): bool {
		$auto_updates = get_option( 'auto_update_plugins', array() );
		return is_array( $auto_updates ) && in_array( TRUSTLENS_PLUGIN_BASENAME, $auto_updates, true );
	}

	/**
	 * Check whether plugin auto-updates are available on this site.
	 *
	 * @since 1.1.1
	 * @return bool
	 */
	private function are_plugin_auto_updates_available(): bool {
		if ( defined( 'AUTOMATIC_UPDATER_DISABLED' ) && AUTOMATIC_UPDATER_DISABLED ) {
			return false;
		}
		// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core WP hook.
		return (bool) apply_filters( 'plugins_auto_update_enabled', true );
	}

	/**
	 * Register settings.
	 *
	 * @since 1.0.0
	 */
	public function register_settings(): void {
		// General settings.
		register_setting(
			'trustlens_settings',
			'trustlens_min_orders',
			array( 'sanitize_callback' => array( $this, 'sanitize_min_orders' ) )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_enable_blocking',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_enable_dashboard',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_enable_order_warning',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_block_message',
			array( 'sanitize_callback' => 'sanitize_textarea_field' )
		);
		if ( wstl_has_payment_method_controls() && wstl_can_use_pro() ) {
			register_setting(
				'trustlens_settings',
				'trustlens_enable_payment_method_controls',
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_segments',
				array( 'sanitize_callback' => array( $this, 'sanitize_payment_method_control_segments' ) )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_gateways',
				array( 'sanitize_callback' => array( $this, 'sanitize_payment_method_control_gateways' ) )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_min_total',
				array( 'sanitize_callback' => array( $this, 'sanitize_payment_method_control_min_total' ) )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_message',
				array( 'sanitize_callback' => 'sanitize_textarea_field' )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_velocity_enabled',
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_velocity_count',
				array( 'sanitize_callback' => array( $this, 'sanitize_payment_method_control_velocity_count' ) )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_velocity_hours',
				array( 'sanitize_callback' => array( $this, 'sanitize_payment_method_control_velocity_hours' ) )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_payment_method_control_linked_risk_enabled',
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
		}

		// Threshold settings.
		register_setting(
			'trustlens_settings',
			'trustlens_returns_high_threshold',
			array( 'sanitize_callback' => array( $this, 'sanitize_returns_high_threshold' ) )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_returns_critical_threshold',
			array( 'sanitize_callback' => array( $this, 'sanitize_returns_critical_threshold' ) )
		);

		// Module settings.
		register_setting(
			'trustlens_settings',
			'trustlens_module_returns_enabled',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_module_orders_enabled',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_module_coupons_enabled',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_module_categories_enabled',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_module_linked_accounts_enabled',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_linked_accounts_penalty',
			array( 'sanitize_callback' => array( $this, 'sanitize_linked_accounts_penalty' ) )
		);
		if ( wstl_has_chargebacks() ) {
			register_setting(
				'trustlens_settings',
				'trustlens_module_chargebacks_enabled',
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_chargebacks_auto_block',
				array( 'sanitize_callback' => array( $this, 'sanitize_chargebacks_auto_block' ) )
			);
		}

		// Coupon abuse options (used by Coupon Abuse Detection module).
		register_setting(
			'trustlens_settings',
			'trustlens_coupons_block_linked_abuse',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
		register_setting(
			'trustlens_settings',
			'trustlens_coupons_max_first_order',
			array( 'sanitize_callback' => array( $this, 'sanitize_coupons_max_first_order' ) )
		);

		// Notification settings (registered under trustlens_notifications group).
		// Global + free notification booleans.
		$notification_booleans = array(
			'trustlens_enable_notifications',
			'trustlens_notify_blocked_checkout',
			'trustlens_notify_welcome_summary',
			'trustlens_notify_weekly_summary',
		);

		// Pro notification settings — only register when Pro feature code is in build (WordPress.org compliant).
		if ( wstl_can_use_pro() ) {
			$notification_booleans = array_merge( $notification_booleans, array(
				'trustlens_notify_high_risk_order',
				'trustlens_notify_segment_critical',
				'trustlens_enable_daily_digest',
				'trustlens_notify_high_value_order',
				'trustlens_notify_repeat_refunder',
				'trustlens_notify_velocity',
				'trustlens_notify_score_recovery',
				'trustlens_notify_new_customer_risk',
				'trustlens_notify_monthly_report',
				'trustlens_notify_chargeback_filed',
			) );
		}

		foreach ( $notification_booleans as $option ) {
			register_setting(
				'trustlens_notifications',
				$option,
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
		}

		register_setting(
			'trustlens_notifications',
			'trustlens_notification_email',
			array( 'sanitize_callback' => 'sanitize_email' )
		);

		// Pro notification integer settings.
		if ( wstl_can_use_pro() ) {
			$notification_integers = array(
				'trustlens_notify_high_value_threshold',
				'trustlens_notify_high_value_score',
				'trustlens_notify_repeat_refunder_count',
				'trustlens_notify_repeat_refunder_days',
				'trustlens_notify_velocity_count',
				'trustlens_notify_velocity_hours',
				'trustlens_notify_new_customer_risk_score',
			);
			foreach ( $notification_integers as $option ) {
				register_setting(
					'trustlens_notifications',
					$option,
					array( 'sanitize_callback' => 'absint' )
				);
			}
		}

		// Automation settings (Pro only — not registered in free build).
		if ( wstl_has_automation() && wstl_can_use_pro() ) {
			register_setting(
				'trustlens_settings',
				'trustlens_enable_automation',
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
		}

		// Webhook settings (Pro only — not registered in free build).
		if ( wstl_has_webhooks() && wstl_can_use_pro() ) {
			register_setting(
				'trustlens_settings',
				'trustlens_enable_webhooks',
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_webhook_events',
				array( 'sanitize_callback' => array( $this, 'sanitize_webhook_events' ) )
			);
		}

		// Scheduled reports settings (Pro only — not registered in free build).
		if ( wstl_has_scheduled_reports() && wstl_can_use_pro() ) {
			register_setting(
				'trustlens_settings',
				'trustlens_enable_scheduled_reports',
				array( 'sanitize_callback' => 'rest_sanitize_boolean' )
			);
			register_setting(
				'trustlens_settings',
				'trustlens_report_email',
				array( 'sanitize_callback' => 'sanitize_email' )
			);
		}

		// Data management.
		register_setting(
			'trustlens_settings',
			'trustlens_remove_data_on_uninstall',
			array( 'sanitize_callback' => 'rest_sanitize_boolean' )
		);
	}

	/**
	 * Sanitize webhook events array.
	 *
	 * @since 1.0.0
	 * @param mixed $value Value to sanitize.
	 * @return array Sanitized array.
	 */
	public function sanitize_webhook_events( $value ): array {
		if ( ! is_array( $value ) ) {
			return array();
		}

		return array_map( 'sanitize_text_field', $value );
	}

	/**
	 * Sanitize minimum orders (General tab). Clamp to 1–20 to match form and prevent invalid values.
	 *
	 * @since 1.0.0
	 * @param mixed $value Value to sanitize.
	 * @return int Sanitized value.
	 */
	public function sanitize_min_orders( $value ): int {
		$v = absint( $value );
		return max( 1, min( 20, $v ) );
	}

	/**
	 * Sanitize high risk threshold (General tab). Clamp to 10–90.
	 *
	 * @since 1.0.0
	 * @param mixed $value Value to sanitize.
	 * @return int Sanitized value.
	 */
	public function sanitize_returns_high_threshold( $value ): int {
		$v = absint( $value );
		return max( 10, min( 90, $v ) );
	}

	/**
	 * Sanitize critical risk threshold (General tab). Clamp to 20–100 and ensure >= high threshold.
	 *
	 * @since 1.0.0
	 * @param mixed $value Value to sanitize.
	 * @return int Sanitized value.
	 */
	public function sanitize_returns_critical_threshold( $value ): int {
		$v = absint( $value );
		$v = max( 20, min( 100, $v ) );
		// Ensure critical >= high so scoring bands make sense (critical is the higher bar).
		$high = 40;
		if ( isset( $_POST['trustlens_returns_high_threshold'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Sanitize callback, nonce verified by options.php.
			$high = $this->sanitize_returns_high_threshold( wp_unslash( $_POST['trustlens_returns_high_threshold'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
		} else {
			$high = (int) get_option( 'trustlens_returns_high_threshold', 40 );
		}
		return max( $v, $high );
	}

	/**
	 * Sanitize linked accounts penalty (Modules tab). Clamp to 0–20 to match form.
	 *
	 * @since 1.0.0
	 * @param mixed $value Value to sanitize.
	 * @return int Sanitized value.
	 */
	public function sanitize_linked_accounts_penalty( $value ): int {
		$v = absint( $value );
		return max( 0, min( 20, $v ) );
	}

	/**
	 * Sanitize max first-order coupons (Modules tab). Clamp to 1–20 to match form.
	 *
	 * @since 1.0.0
	 * @param mixed $value Value to sanitize.
	 * @return int Sanitized value.
	 */
	public function sanitize_coupons_max_first_order( $value ): int {
		$v = absint( $value );
		return max( 1, min( 20, $v ) );
	}

	/**
	 * Sanitize targeted segments for payment method controls.
	 *
	 * @since 1.0.6
	 * @param mixed $value Value to sanitize.
	 * @return array
	 */
	public function sanitize_payment_method_control_segments( $value ): array {
		if ( ! is_array( $value ) ) {
			return array( 'risk', 'critical' );
		}

		$allowed  = array( 'vip', 'trusted', 'normal', 'caution', 'risk', 'critical' );
		$segments = array_values(
			array_intersect(
				$allowed,
				array_map( 'sanitize_key', $value )
			)
		);

		return empty( $segments ) ? array( 'risk', 'critical' ) : $segments;
	}

	/**
	 * Sanitize targeted gateway IDs for payment method controls.
	 *
	 * @since 1.0.6
	 * @param mixed $value Value to sanitize.
	 * @return array
	 */
	public function sanitize_payment_method_control_gateways( $value ): array {
		if ( ! is_array( $value ) ) {
			return array();
		}

		$gateways = array_values(
			array_filter(
				array_map( 'sanitize_key', $value )
			)
		);

		if ( ! class_exists( 'WC_Payment_Gateways' ) ) {
			return $gateways;
		}

		$available = array();
		foreach ( WC_Payment_Gateways::instance()->payment_gateways() as $gateway ) {
			if ( isset( $gateway->id ) ) {
				$available[] = sanitize_key( (string) $gateway->id );
			}
		}

		return array_values( array_intersect( $available, $gateways ) );
	}

	/**
	 * Sanitize minimum order total for payment method controls.
	 *
	 * @since 1.0.6
	 * @param mixed $value Value to sanitize.
	 * @return float
	 */
	public function sanitize_payment_method_control_min_total( $value ): float {
		$amount = is_numeric( $value ) ? (float) $value : 0.0;
		$amount = max( 0.0, min( 9999999.0, $amount ) );
		return round( $amount, 2 );
	}

	/**
	 * Sanitize velocity order threshold for payment method controls.
	 *
	 * @since 1.0.6
	 * @param mixed $value Value to sanitize.
	 * @return int
	 */
	public function sanitize_payment_method_control_velocity_count( $value ): int {
		$v = absint( $value );
		return max( 2, min( 50, $v ) );
	}

	/**
	 * Sanitize velocity window hours for payment method controls.
	 *
	 * @since 1.0.6
	 * @param mixed $value Value to sanitize.
	 * @return int
	 */
	public function sanitize_payment_method_control_velocity_hours( $value ): int {
		$v = absint( $value );
		return max( 1, min( 168, $v ) );
	}

	/**
	 * Sanitize chargebacks auto-block threshold (Chargebacks tab). 0 = disabled, 1–20 allowed.
	 *
	 * @since 1.2.0
	 * @param mixed $value Value to sanitize.
	 * @return int Sanitized value.
	 */
	public function sanitize_chargebacks_auto_block( $value ): int {
		$v = absint( $value );
		return max( 0, min( 20, $v ) );
	}

	/**
	 * Register dashboard widget.
	 *
	 * @since 1.0.0
	 */
	public function register_dashboard_widget(): void {
		if ( ! get_option( 'trustlens_enable_dashboard', true ) ) {
			return;
		}

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		wp_add_dashboard_widget(
			'trustlens_dashboard_widget',
			__( 'TrustLens - Customer Trust', 'trustlens' ),
			array( $this, 'render_dashboard_widget' )
		);
	}

	/**
	 * Render dashboard widget.
	 *
	 * @since 1.0.0
	 */
	public function render_dashboard_widget(): void {
		$repo = new TrustLens_Customer_Repository();
		$high_risk = $repo->get_high_risk( 5 );
		$counts = $repo->get_segment_counts();
		$risk_count = $counts['risk'] + $counts['critical'];

		?>
		<div class="wstl-dashboard-widget">
			<?php if ( $risk_count > 0 ) : ?>
				<p class="wstl-widget-alert">
					<?php
					echo esc_html(
						sprintf(
							/* translators: %d: number of high-risk customers */
							_n(
								'%d customer requires attention',
								'%d customers require attention',
								$risk_count,
								'trustlens'
							),
							$risk_count
						)
					);
					?>
				</p>
			<?php else : ?>
				<p class="wstl-widget-ok">
					<?php esc_html_e( 'No high-risk customers detected.', 'trustlens' ); ?>
				</p>
			<?php endif; ?>

			<?php if ( ! empty( $high_risk ) ) : ?>
				<ul class="wstl-widget-list">
					<?php foreach ( $high_risk as $customer ) : ?>
						<li>
							<span class="wstl-segment-badge wstl-segment-<?php echo esc_attr( $customer->segment ); ?>">
								<?php echo esc_html( wstl_get_segment_label( $customer->segment ) ); ?>
							</span>
							<strong><?php echo esc_html( $customer->customer_email ?: __( 'Guest', 'trustlens' ) ); ?></strong>
							<br>
							<small>
								<?php
								printf(
									/* translators: %1$s: return rate, %2$s: refund value */
									esc_html__( '%1$s%% return rate', 'trustlens' ) . ' &middot; ' . esc_html__( '%2$s refunded', 'trustlens' ),
									esc_html( number_format( $customer->return_rate, 0 ) ),
									wp_kses_post( wc_price( $customer->total_refund_value ) )
								);
								?>
							</small>
						</li>
					<?php endforeach; ?>
				</ul>
			<?php endif; ?>

			<p class="wstl-widget-footer">
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens' ) ); ?>">
					<?php esc_html_e( 'View Dashboard', 'trustlens' ); ?> &rarr;
				</a>
			</p>
		</div>
		<?php
	}

	/**
	 * Register order metabox.
	 *
	 * @since 1.0.0
	 */
	public function register_order_metabox(): void {
		if ( ! get_option( 'trustlens_enable_order_warning', true ) ) {
			return;
		}

		$screen = class_exists( '\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController' )
			? wc_get_page_screen_id( 'shop-order' )
			: 'shop_order';

		add_meta_box(
			'trustlens_customer_trust',
			__( 'Customer Trust', 'trustlens' ),
			array( $this, 'render_order_metabox' ),
			$screen,
			'side',
			'high'
		);
	}

	/**
	 * Render order metabox.
	 *
	 * @since 1.0.0
	 * @param WP_Post|WC_Order $post_or_order Post or order object.
	 */
	public function render_order_metabox( $post_or_order ): void {
		$order = $post_or_order instanceof WC_Order ? $post_or_order : wc_get_order( $post_or_order->ID );

		if ( ! $order ) {
			return;
		}

		$email_hash = wstl_get_email_hash( $order->get_billing_email() );
		$customer = wstl_get_customer( $email_hash );

		if ( ! $customer ) {
			echo '<p>' . esc_html__( 'No trust data available for this customer.', 'trustlens' ) . '</p>';
			return;
		}

		$color = wstl_get_segment_color( $customer->segment );
		$is_risky = in_array( $customer->segment, array( 'risk', 'critical' ), true );

		?>
		<div class="wstl-order-metabox" style="<?php echo $is_risky ? 'background: #fff5f5; border-left: 4px solid ' . esc_attr( $color ) . ';' : ''; ?> padding: 10px; margin: -6px -12px -12px;">
			<p style="margin: 0 0 8px;">
				<span class="wstl-segment-badge" style="background: <?php echo esc_attr( $color ); ?>; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 600;">
					<?php echo esc_html( wstl_get_segment_label( $customer->segment ) ); ?>
				</span>
				<strong style="margin-left: 8px;">Score: <?php echo esc_html( $customer->trust_score ); ?>/100</strong>
			</p>

			<p style="margin: 0; font-size: 12px; color: #666;">
				<?php
				echo esc_html(
					sprintf(
						/* translators: %1$d: orders, %2$d: returns, %3$s: return rate */
						__( '%1$d orders &middot; %2$d returns &middot; %3$s%% rate', 'trustlens' ),
						$customer->total_orders,
						$customer->total_refunds,
						number_format( $customer->return_rate, 0 )
					)
				);
				?>
			</p>

			<?php if ( $is_risky ) : ?>
				<p style="margin: 8px 0 0; padding: 8px; background: #fff; border: 1px solid #ddd; font-size: 12px;">
					<strong style="color: <?php echo esc_attr( $color ); ?>;">
						<?php esc_html_e( 'Review this order carefully.', 'trustlens' ); ?>
					</strong>
				</p>
			<?php endif; ?>

			<p style="margin: 8px 0 0;">
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=trustlens-customers&customer=' . $email_hash ) ); ?>" class="wstl-link-small">
					<?php esc_html_e( 'View Full Profile', 'trustlens' ); ?> &rarr;
				</a>
			</p>
		</div>
		<?php
	}

	/**
	 * AJAX: Block customer.
	 *
	 * @since 1.0.0
	 */
	public function ajax_block_customer(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$email_hash = sanitize_text_field( wp_unslash( $_POST['email_hash'] ?? '' ) );

		if ( empty( $email_hash ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid customer.', 'trustlens' ) ) );
		}

		$result = wstl_block_customer( $email_hash, __( 'Blocked via admin', 'trustlens' ) );

		if ( $result ) {
			wp_send_json_success( array( 'message' => __( 'Customer blocked.', 'trustlens' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to block customer.', 'trustlens' ) ) );
		}
	}

	/**
	 * AJAX: Unblock customer.
	 *
	 * @since 1.0.0
	 */
	public function ajax_unblock_customer(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$email_hash = sanitize_text_field( wp_unslash( $_POST['email_hash'] ?? '' ) );

		if ( empty( $email_hash ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid customer.', 'trustlens' ) ) );
		}

		$result = wstl_unblock_customer( $email_hash );

		if ( $result ) {
			wp_send_json_success( array( 'message' => __( 'Customer unblocked.', 'trustlens' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to unblock customer.', 'trustlens' ) ) );
		}
	}

	/**
	 * AJAX: Add to allowlist.
	 *
	 * @since 1.0.0
	 */
	public function ajax_allowlist_customer(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$email_hash = sanitize_text_field( wp_unslash( $_POST['email_hash'] ?? '' ) );

		if ( empty( $email_hash ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid customer.', 'trustlens' ) ) );
		}

		$result = wstl_allowlist_customer( $email_hash );

		if ( $result ) {
			wp_send_json_success( array( 'message' => __( 'Customer added to allowlist.', 'trustlens' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to update customer.', 'trustlens' ) ) );
		}
	}

	/**
	 * AJAX: Remove from allowlist.
	 *
	 * @since 1.0.0
	 */
	public function ajax_remove_allowlist(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$email_hash = sanitize_text_field( wp_unslash( $_POST['email_hash'] ?? '' ) );

		if ( empty( $email_hash ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid customer.', 'trustlens' ) ) );
		}

		$result = wstl_remove_from_allowlist( $email_hash );

		if ( $result ) {
			wp_send_json_success( array( 'message' => __( 'Customer removed from allowlist.', 'trustlens' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to update customer.', 'trustlens' ) ) );
		}
	}

	/**
	 * AJAX: Start historical sync.
	 *
	 * @since 1.0.0
	 */
	public function ajax_start_sync(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		try {
			$sync = TrustLens_Historical_Sync::instance();
			$started = $sync->start( false );

			if ( $started ) {
				wp_send_json_success( array(
					'message' => __( 'Sync started.', 'trustlens' ),
					'status'  => $sync->get_status(),
				) );
			} else {
				$status = $sync->get_status();
				if ( 'running' === $status['status'] ) {
					wp_send_json_error( array( 'message' => __( 'Sync is already running.', 'trustlens' ) ) );
				} else {
					wp_send_json_error( array( 'message' => $status['error'] ?? __( 'Failed to start sync.', 'trustlens' ) ) );
				}
			}
		} catch ( \Throwable $e ) {
			wp_send_json_error(
				array(
					'message' => sprintf(
						/* translators: %s: technical error message */
						__( 'Failed to start sync: %s', 'trustlens' ),
						$e->getMessage()
					),
				),
				500
			);
		}
	}

	/**
	 * AJAX: Process a single sync batch and return progress.
	 *
	 * @since 1.0.0
	 */
	public function ajax_process_sync_batch(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		try {
			$sync   = TrustLens_Historical_Sync::instance();
			$result = $sync->process_ajax_batch();

			wp_send_json_success( array(
				'status'   => $result,
				'progress' => $result['progress'] ?? 0,
			) );
		} catch ( \Throwable $e ) {
			wp_send_json_error(
				array(
					'message' => sprintf(
						/* translators: %s: technical error message */
						__( 'Sync batch failed: %s', 'trustlens' ),
						$e->getMessage()
					),
				),
				500
			);
		}
	}

	/**
	 * AJAX: Stop historical sync.
	 *
	 * @since 1.0.0
	 */
	public function ajax_stop_sync(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$sync = TrustLens_Historical_Sync::instance();
		$sync->stop();

		wp_send_json_success( array(
			'message' => __( 'Sync stopped.', 'trustlens' ),
			'status'  => $sync->get_status(),
		) );
	}

	/**
	 * AJAX: Reset sync and clear data.
	 *
	 * @since 1.0.0
	 */
	public function ajax_reset_sync(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$sync = TrustLens_Historical_Sync::instance();
		$sync->reset();

		wp_send_json_success( array(
			'message' => __( 'All data has been reset.', 'trustlens' ),
			'status'  => $sync->get_status(),
		) );
	}

	/**
	 * AJAX: Generate API key.
	 *
	 * @since 1.0.0
	 */
	public function ajax_generate_api_key(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		// Generate a secure API key.
		$api_key = wp_generate_password( 64, false, false );
		$hashed_key = hash( 'sha256', $api_key );

		// Store the hashed key (we show the plain key once, then only store hash).
		update_option( 'trustlens_api_key', $hashed_key );

		wp_send_json_success( array(
			'message' => __( 'API key generated. Copy it now - it will not be shown again.', 'trustlens' ),
			'api_key' => $api_key,
		) );
	}

	/**
	 * AJAX: Reset all data (danger zone).
	 *
	 * @since 1.1.0
	 */
	public function ajax_reset_data(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		global $wpdb;

		// Delete all data from TrustLens tables.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- TRUNCATE requires direct query, table name from $wpdb->prefix is safe.
		$wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}trustlens_customers" );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}trustlens_events" );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}trustlens_signals" );

		// Reset sync status.
		delete_option( 'trustlens_sync_status' );

		wp_send_json_success( array( 'message' => __( 'All data has been deleted.', 'trustlens' ) ) );
	}

	/**
	 * AJAX: Bulk action on customers.
	 *
	 * @since 1.1.0
	 */
	public function ajax_bulk_action(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$action = sanitize_text_field( wp_unslash( $_POST['bulk_action'] ?? '' ) );
		$customers = isset( $_POST['customers'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['customers'] ) ) : array();

		if ( empty( $action ) || empty( $customers ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid request.', 'trustlens' ) ) );
		}

		$bulk = TrustLens_Bulk_Operations::instance();
		$result = $bulk->execute( $action, $customers );

		if ( $result['success'] ) {
			wp_send_json_success( array(
				'message'   => $result['message'],
				'processed' => $result['processed'],
			) );
		} else {
			wp_send_json_error( array( 'message' => $result['message'] ) );
		}
	}

	/**
	 * AJAX: Recalculate customer score.
	 *
	 * @since 1.1.0
	 */
	public function ajax_recalculate_score(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$email_hash = sanitize_text_field( wp_unslash( $_POST['email_hash'] ?? '' ) );

		if ( empty( $email_hash ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid customer.', 'trustlens' ) ) );
		}

		$calculator = TrustLens_Score_Calculator::instance();
		$result = $calculator->recalculate_score( $email_hash );

		if ( $result ) {
			wp_send_json_success( array(
				'message' => __( 'Score recalculated.', 'trustlens' ),
				'score'   => $result['score'],
				'segment' => $result['segment'],
			) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to recalculate score.', 'trustlens' ) ) );
		}
	}

	/**
	 * AJAX: Save admin note.
	 *
	 * @since 1.1.0
	 */
	public function ajax_save_note(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$email_hash = sanitize_text_field( wp_unslash( $_POST['email_hash'] ?? '' ) );
		$note = sanitize_textarea_field( wp_unslash( $_POST['note'] ?? '' ) );

		if ( empty( $email_hash ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid customer.', 'trustlens' ) ) );
		}

		$customer = wstl_get_customer( $email_hash );
		$existing_notes = $customer ? $customer->admin_notes : '';
		$timestamp = current_time( 'mysql' );
		$new_notes = $existing_notes . "\n[{$timestamp}] " . $note;

		$result = wstl_update_customer( $email_hash, array( 'admin_notes' => trim( $new_notes ) ) );

		if ( $result ) {
			wp_send_json_success( array( 'message' => __( 'Note saved.', 'trustlens' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to save note.', 'trustlens' ) ) );
		}
	}

	/**
	 * AJAX: Export customers to CSV.
	 *
	 * @since 1.1.0
	 */
	public function ajax_export_csv(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$segment = sanitize_text_field( wp_unslash( $_POST['segment'] ?? '' ) );
		$search = sanitize_text_field( wp_unslash( $_POST['search'] ?? '' ) );

		$repo = new TrustLens_Customer_Repository();
		$customers = $repo->get_customers( array(
			'per_page' => 10000,
			'segment'  => $segment,
			'search'   => $search,
		) );

		// Build CSV string.
		$csv_rows = array();

		// Header row.
		$csv_rows[] = $this->array_to_csv_row( array( 'Email', 'Score', 'Segment', 'Orders', 'Refunds', 'Return Rate', 'Refund Value', 'Blocked', 'Allowlisted' ) );

		// Data rows.
		foreach ( $customers as $customer ) {
			$csv_rows[] = $this->array_to_csv_row( array(
				$customer->customer_email ?: 'Guest',
				$customer->trust_score,
				$customer->segment,
				$customer->total_orders,
				$customer->total_refunds,
				number_format( $customer->return_rate, 1 ) . '%',
				$customer->total_refund_value,
				$customer->is_blocked ? 'Yes' : 'No',
				$customer->is_allowlisted ? 'Yes' : 'No',
			) );
		}

		$csv_string = implode( "\n", $csv_rows );

		$filename = 'trustlens-customers-' . gmdate( 'Y-m-d' ) . '.csv';

		wp_send_json_success( array(
			'csv'      => $csv_string,
			'filename' => $filename,
		) );
	}

	/**
	 * AJAX: Save automation rule.
	 *
	 * @since 1.1.0
	 */
	public function ajax_save_automation_rule(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Automation' ) ) {
			wp_send_json_error( array( 'message' => __( 'Automation rules are available in the Pro version.', 'trustlens' ) ) );
		}

		$rule_id = sanitize_text_field( wp_unslash( $_POST['rule_id'] ?? '' ) );
		$raw_conditions = isset( $_POST['conditions'] ) && is_array( $_POST['conditions'] ) ? wp_unslash( $_POST['conditions'] ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized in sanitize_conditions().
		$rule = array(
			'id'           => $rule_id ?: wp_generate_uuid4(),
			'name'         => sanitize_text_field( wp_unslash( $_POST['name'] ?? '' ) ),
			'trigger'      => sanitize_text_field( wp_unslash( $_POST['trigger'] ?? '' ) ),
			'conditions'   => $this->sanitize_conditions( $raw_conditions ),
			'action'       => sanitize_text_field( wp_unslash( $_POST['action'] ?? '' ) ),
			'action_value' => sanitize_text_field( wp_unslash( $_POST['action_value'] ?? '' ) ),
			'enabled'      => ! empty( $_POST['enabled'] ),
		);

		$rules = get_option( 'trustlens_automation_rules', array() );

		if ( $rule_id ) {
			$replaced = false;
			foreach ( $rules as $i => $existing ) {
				if ( ( $existing['id'] ?? '' ) === $rule_id ) {
					$rules[ $i ] = $rule;
					$replaced = true;
					break;
				}
			}
			if ( ! $replaced ) {
				wp_send_json_error( array( 'message' => __( 'Rule not found. It may have been deleted.', 'trustlens' ) ) );
			}
		} else {
			$rules[] = $rule;
		}

		update_option( 'trustlens_automation_rules', $rules );

		wp_send_json_success( array( 'message' => __( 'Rule saved.', 'trustlens' ), 'rule' => $rule ) );
	}

	/**
	 * Sanitize conditions array.
	 *
	 * @since 1.1.0
	 * @param array $conditions Raw conditions.
	 * @return array Sanitized conditions.
	 */
	private function sanitize_conditions( array $conditions ): array {
		$sanitized = array();
		foreach ( $conditions as $condition ) {
			if ( ! empty( $condition['field'] ) ) {
				$sanitized[] = array(
					'field'    => sanitize_text_field( $condition['field'] ?? '' ),
					'operator' => sanitize_text_field( $condition['operator'] ?? '' ),
					'value'    => sanitize_text_field( $condition['value'] ?? '' ),
				);
			}
		}
		return $sanitized;
	}

	/**
	 * AJAX: Delete automation rule.
	 *
	 * @since 1.1.0
	 */
	public function ajax_delete_automation_rule(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Automation' ) ) {
			wp_send_json_error( array( 'message' => __( 'Automation rules are available in the Pro version.', 'trustlens' ) ) );
		}

		$rule_id = sanitize_text_field( wp_unslash( $_POST['rule_id'] ?? '' ) );
		$rules = get_option( 'trustlens_automation_rules', array() );

		$rules = array_filter( $rules, function( $rule ) use ( $rule_id ) {
			return ( $rule['id'] ?? '' ) !== $rule_id;
		} );

		update_option( 'trustlens_automation_rules', array_values( $rules ) );

		wp_send_json_success( array( 'message' => __( 'Rule deleted.', 'trustlens' ) ) );
	}

	/**
	 * AJAX: Update automation rule enabled state.
	 *
	 * @since 1.0.0
	 */
	public function ajax_update_automation_rule_enabled(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Automation' ) ) {
			wp_send_json_error( array( 'message' => __( 'Automation rules are available in the Pro version.', 'trustlens' ) ) );
		}

		$rule_id = sanitize_text_field( wp_unslash( $_POST['rule_id'] ?? '' ) );
		$enabled = ! empty( $_POST['enabled'] );

		if ( '' === $rule_id ) {
			wp_send_json_error( array( 'message' => __( 'Rule ID required.', 'trustlens' ) ) );
		}

		$rules = get_option( 'trustlens_automation_rules', array() );
		$found = false;
		foreach ( $rules as $i => $rule ) {
			if ( ( $rule['id'] ?? '' ) === $rule_id ) {
				$rules[ $i ]['enabled'] = $enabled;
				$found = true;
				break;
			}
		}

		if ( ! $found ) {
			wp_send_json_error( array( 'message' => __( 'Rule not found.', 'trustlens' ) ) );
		}

		update_option( 'trustlens_automation_rules', $rules );
		wp_send_json_success( array( 'message' => __( 'Rule updated.', 'trustlens' ) ) );
	}

	/**
	 * AJAX: Add webhook endpoint.
	 *
	 * @since 1.1.0
	 */
	public function ajax_add_webhook(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Webhooks' ) ) {
			wp_send_json_error( array( 'message' => __( 'Webhooks are available in the Pro version.', 'trustlens' ) ) );
		}

		$url = esc_url_raw( wp_unslash( $_POST['url'] ?? '' ) );
		$secret = sanitize_text_field( wp_unslash( $_POST['secret'] ?? '' ) );
		$enabled = ! empty( $_POST['enabled'] );

		if ( empty( $url ) ) {
			wp_send_json_error( array( 'message' => __( 'URL is required.', 'trustlens' ) ) );
		}

		$id = TrustLens_Webhooks::add_endpoint( array(
			'url'     => $url,
			'secret'  => $secret,
			'enabled' => $enabled,
		) );

		wp_send_json_success( array( 'message' => __( 'Webhook added.', 'trustlens' ), 'id' => $id ) );
	}

	/**
	 * AJAX: Delete webhook endpoint.
	 *
	 * @since 1.1.0
	 */
	public function ajax_delete_webhook(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Webhooks' ) ) {
			wp_send_json_error( array( 'message' => __( 'Webhooks are available in the Pro version.', 'trustlens' ) ) );
		}

		$id = sanitize_text_field( wp_unslash( $_POST['endpoint_id'] ?? '' ) );
		TrustLens_Webhooks::remove_endpoint( $id );

		wp_send_json_success( array( 'message' => __( 'Webhook deleted.', 'trustlens' ) ) );
	}

	/**
	 * AJAX: Test webhook endpoint.
	 *
	 * @since 1.1.0
	 */
	public function ajax_test_webhook(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Webhooks' ) ) {
			wp_send_json_error( array( 'message' => __( 'Webhooks are available in the Pro version.', 'trustlens' ) ) );
		}

		$url = esc_url_raw( wp_unslash( $_POST['url'] ?? '' ) );
		$secret = sanitize_text_field( wp_unslash( $_POST['secret'] ?? '' ) );

		$result = TrustLens_Webhooks::test_endpoint( $url, $secret );

		if ( $result['success'] ) {
			wp_send_json_success( array( 'message' => __( 'Webhook test successful.', 'trustlens' ) ) );
		} else {
			wp_send_json_error( array( 'message' => $result['error'] ?? __( 'Webhook test failed.', 'trustlens' ) ) );
		}
	}

	/**
	 * AJAX: Add report schedule.
	 *
	 * @since 1.1.0
	 */
	public function ajax_add_report_schedule(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Scheduled_Reports' ) ) {
			wp_send_json_error( array( 'message' => __( 'Scheduled reports are available in the Pro version.', 'trustlens' ) ) );
		}

		$frequency = sanitize_text_field( wp_unslash( $_POST['frequency'] ?? 'weekly' ) );
		$day       = absint( wp_unslash( $_POST['day'] ?? 1 ) );
		$time      = sanitize_text_field( wp_unslash( $_POST['time'] ?? '09:00' ) );

		$schedule = array(
			'id'        => wp_generate_uuid4(),
			'type'      => sanitize_text_field( wp_unslash( $_POST['type'] ?? 'summary' ) ),
			'frequency' => $frequency,
			'day'       => $day,
			'time'      => $time,
			'enabled'   => ! empty( $_POST['enabled'] ),
			'next_run'  => $this->calculate_next_run( array( 'frequency' => $frequency, 'day' => $day, 'time' => $time ) ),
		);

		$schedules = get_option( 'trustlens_report_schedules', array() );
		$schedules[] = $schedule;
		update_option( 'trustlens_report_schedules', $schedules );

		wp_send_json_success( array( 'message' => __( 'Schedule added.', 'trustlens' ), 'schedule' => $schedule ) );
	}

	/**
	 * Calculate next run time.
	 *
	 * @since 1.1.0
	 * @param array $data Schedule data.
	 * @return int Unix timestamp.
	 */
	private function calculate_next_run( array $data ): int {
		$frequency = sanitize_text_field( $data['frequency'] ?? 'weekly' );
		$time = sanitize_text_field( $data['time'] ?? '09:00' );
		$day = absint( $data['day'] ?? 1 );

		$now = current_time( 'timestamp' );

		if ( 'daily' === $frequency ) {
			$next = strtotime( 'today ' . $time, $now );
			if ( $next <= $now ) {
				$next = strtotime( 'tomorrow ' . $time, $now );
			}
		} elseif ( 'monthly' === $frequency ) {
			$next = strtotime( 'first day of next month ' . $time, $now );
		} else {
			$days = array( 1 => 'Monday', 2 => 'Tuesday', 3 => 'Wednesday', 4 => 'Thursday', 5 => 'Friday', 6 => 'Saturday', 7 => 'Sunday' );
			$day_name = $days[ $day ] ?? 'Monday';
			$next = strtotime( 'next ' . $day_name . ' ' . $time, $now );
		}

		return $next;
	}

	/**
	 * AJAX: Delete report schedule.
	 *
	 * @since 1.1.0
	 */
	public function ajax_delete_report_schedule(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Scheduled_Reports' ) ) {
			wp_send_json_error( array( 'message' => __( 'Scheduled reports are available in the Pro version.', 'trustlens' ) ) );
		}

		$schedule_id = sanitize_text_field( wp_unslash( $_POST['schedule_id'] ?? '' ) );
		$schedules = get_option( 'trustlens_report_schedules', array() );

		$schedules = array_filter( $schedules, function( $s ) use ( $schedule_id ) {
			return ( $s['id'] ?? '' ) !== $schedule_id;
		} );

		update_option( 'trustlens_report_schedules', array_values( $schedules ) );

		wp_send_json_success( array( 'message' => __( 'Schedule deleted.', 'trustlens' ) ) );
	}

	/**
	 * AJAX: Send report now.
	 *
	 * @since 1.1.0
	 */
	public function ajax_send_report_now(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		if ( ! wstl_can_use_pro() || ! class_exists( 'TrustLens_Scheduled_Reports' ) ) {
			wp_send_json_error( array( 'message' => __( 'Scheduled reports are available in the Pro version.', 'trustlens' ) ) );
		}

		$schedule_id = sanitize_text_field( wp_unslash( $_POST['schedule_id'] ?? '' ) );

		$reports = TrustLens_Scheduled_Reports::instance();
		$result = $reports->send_report_by_id( $schedule_id );

		if ( $result ) {
			wp_send_json_success( array( 'message' => __( 'Report sent.', 'trustlens' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to send report.', 'trustlens' ) ) );
		}
	}

	/**
	 * Handle test notification AJAX request.
	 *
	 * @since 1.1.0
	 */
	public function ajax_send_test_notification(): void {
		check_ajax_referer( 'trustlens_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'trustlens' ) ) );
		}

		$to = get_option( 'trustlens_notification_email', get_option( 'admin_email' ) );
		if ( empty( $to ) ) {
			$to = get_option( 'admin_email' );
		}

		$subject = sprintf(
			/* translators: %s: site name */
			__( '[TrustLens] Test Notification from %s', 'trustlens' ),
			get_bloginfo( 'name' )
		);

		$message = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head>';
		$message .= '<body style="font-family: -apple-system, BlinkMacSystemFont, \'Segoe UI\', Roboto, sans-serif; font-size: 14px; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;">';
		$message .= '<div style="background: linear-gradient(135deg, #2271b1, #135e96); padding: 20px; border-radius: 5px 5px 0 0;">';
		$message .= '<h1 style="margin: 0; color: #fff; font-size: 24px;">' . esc_html__( 'Test Notification', 'trustlens' ) . '</h1>';
		$message .= '</div>';
		$message .= '<div style="background: #fff; padding: 20px; border: 1px solid #ddd; border-top: none; border-radius: 0 0 5px 5px;">';
		$message .= '<p>' . esc_html__( 'This is a test notification from TrustLens. If you received this email, your notification settings are working correctly.', 'trustlens' ) . '</p>';
		$message .= '<p style="margin-top: 20px;">';
		$message .= '<a href="' . esc_url( admin_url( 'admin.php?page=trustlens-notifications' ) ) . '" style="background: #2271b1; color: #fff; padding: 10px 20px; text-decoration: none; border-radius: 3px;">' . esc_html__( 'Go to Notification Settings', 'trustlens' ) . '</a>';
		$message .= '</p>';
		$message .= '</div>';
		$message .= '<p style="margin-top: 20px; font-size: 12px; color: #666; text-align: center;">';
		$message .= sprintf(
			/* translators: %s: site name */
			esc_html__( 'This email was sent by TrustLens on %s', 'trustlens' ),
			get_bloginfo( 'name' )
		);
		$message .= '</p>';
		$message .= '</body></html>';

		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			'From: ' . get_bloginfo( 'name' ) . ' <' . get_option( 'admin_email' ) . '>',
		);

		$sent = wp_mail( $to, $subject, $message, $headers );

		if ( $sent ) {
			wp_send_json_success( array(
				'message' => sprintf(
					/* translators: %s: email address */
					__( 'Test email sent to %s', 'trustlens' ),
					$to
				),
			) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to send test email. Check your server mail settings.', 'trustlens' ) ) );
		}
	}

	/**
	 * Handle auto-update notice dismissal.
	 *
	 * @since 1.1.1
	 */
	public function ajax_dismiss_auto_update_notice(): void {
		check_ajax_referer( 'wstl_auto_update_notice', 'nonce' );

		if ( ! current_user_can( 'update_plugins' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'trustlens' ) ) );
		}

		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid user.', 'trustlens' ) ) );
		}

		update_user_meta( $user_id, self::AUTO_UPDATE_NOTICE_DISMISSED_META, time() );

		wp_send_json_success( array( 'message' => __( 'Notice dismissed.', 'trustlens' ) ) );
	}

	/**
	 * Enable plugin auto-updates for TrustLens.
	 *
	 * @since 1.1.1
	 */
	public function ajax_enable_auto_updates(): void {
		check_ajax_referer( 'wstl_auto_update_notice', 'nonce' );

		if ( ! current_user_can( 'update_plugins' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'trustlens' ) ) );
		}

		$auto_updates = get_option( 'auto_update_plugins', array() );
		if ( ! is_array( $auto_updates ) ) {
			$auto_updates = array();
		}

		if ( ! in_array( TRUSTLENS_PLUGIN_BASENAME, $auto_updates, true ) ) {
			$auto_updates[] = TRUSTLENS_PLUGIN_BASENAME;
			update_option( 'auto_update_plugins', array_values( array_unique( $auto_updates ) ) );
		}

		$user_id = get_current_user_id();
		if ( $user_id ) {
			update_user_meta( $user_id, self::AUTO_UPDATE_NOTICE_DISMISSED_META, time() );
		}

		wp_send_json_success( array( 'message' => __( 'Auto-updates enabled.', 'trustlens' ) ) );
	}

	/**
	 * Dismiss review notice permanently.
	 *
	 * @since 1.1.1
	 */
	public function ajax_dismiss_review_notice(): void {
		check_ajax_referer( 'wstl_review_notice', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) && ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'trustlens' ) ) );
		}

		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid user.', 'trustlens' ) ) );
		}

		update_user_meta( $user_id, self::REVIEW_NOTICE_DISMISSED_META, time() );
		delete_user_meta( $user_id, self::REVIEW_NOTICE_REMIND_LATER_META );

		wp_send_json_success( array( 'message' => __( 'Notice dismissed.', 'trustlens' ) ) );
	}

	/**
	 * Snooze review notice for later.
	 *
	 * @since 1.1.1
	 */
	public function ajax_remind_later_review_notice(): void {
		check_ajax_referer( 'wstl_review_notice', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) && ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'trustlens' ) ) );
		}

		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid user.', 'trustlens' ) ) );
		}

		update_user_meta( $user_id, self::REVIEW_NOTICE_REMIND_LATER_META, time() + ( 14 * DAY_IN_SECONDS ) );

		wp_send_json_success( array( 'message' => __( 'Reminder set.', 'trustlens' ) ) );
	}

	/**
	 * Convert an array to a CSV row string.
	 *
	 * @since 1.1.0
	 * @param array $fields Array of field values.
	 * @return string CSV-formatted row.
	 */
	private function array_to_csv_row( array $fields ): string {
		$escaped = array();
		foreach ( $fields as $field ) {
			$field = (string) $field;
			// Escape quotes and wrap in quotes if contains comma, quote, or newline.
			if ( strpos( $field, ',' ) !== false || strpos( $field, '"' ) !== false || strpos( $field, "\n" ) !== false ) {
				$field = '"' . str_replace( '"', '""', $field ) . '"';
			}
			$escaped[] = $field;
		}
		return implode( ',', $escaped );
	}
}
