<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * TriggerNinja Platform Manager
 * 
 * Handles platform enablement and configuration interface
 * Designed to scale from 2 to 200+ platforms elegantly
 */
class TriggerNinja_Platform_Manager
{
    /**
     * Plugin instance
     * 
     * @var TriggerNinja_Plugin
     */
    private $plugin;
    
    /**
     * Available platforms registry
     * 
     * @var array
     */
    private $available_platforms = array();
    
    /**
     * Constructor
     * 
     * @param TriggerNinja_Plugin $plugin
     */
    public function __construct( $plugin )
    {
        $this->plugin = $plugin;
        
        // Hook into WordPress admin
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_action( 'ninja_forms_settings_custom_field', array( $this, 'render_platform_manager' ), 10, 2 );
        add_action( 'wp_ajax_triggerninja_toggle_platform', array( $this, 'ajax_toggle_platform' ) );
        add_action( 'wp_ajax_triggerninja_configure_platform', array( $this, 'ajax_configure_platform' ) );
        add_action( 'wp_ajax_triggerninja_save_platform_config', array( $this, 'ajax_save_platform_config' ) );
        add_action( 'wp_ajax_triggerninja_authorize_aweber', array( $this, 'ajax_authorize_aweber' ) );
        
        // Initialize platform registry
        $this->init_platform_registry();
    }
    
    /**
     * Initialize platform registry with available platforms
     */
    private function init_platform_registry()
    {
        // Register core platforms
        $this->register_available_platform( 'acelle', array(
            'name'        => __( 'Acelle Mail', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Unlimited email marketing.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'acelle.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Acelle'
        ));

        $this->register_available_platform( 'aweber', array(
            'name'        => __( 'AWeber', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email Marketing & More for Small Businesses.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'aweber.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Aweber'
        ));
        
        
        $this->register_available_platform( 'brevo', array(
            'name'        => __( 'Brevo', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'All-in-one marketing platform (formerly Sendinblue).', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'brevo.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Brevo'
        ));

        $this->register_available_platform( 'campaignmonitor', array(
            'name'        => __( 'Campaign Monitor', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing platform & services.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'campaignmonitor.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_CampaignMonitor'
        ));

        $this->register_available_platform( 'campayn', array(
            'name'        => __( 'Campayn', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Digital and Email Marketing Platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'campayn.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Campayn'
        ));

        $this->register_available_platform( 'companyhub', array(
            'name'        => __( 'CompanyHub', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Sales CRM for growing teams.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'companyhub.svg',
            'category'    => 'crm',
            'class'       => 'TriggerNinja_CompanyHub'
        ));

        $this->register_available_platform( 'constantcontact', array(
            'name'        => __( 'Constant Contact', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing platform with powerful automation.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'constantcontact.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_ConstantContact'
        ));

        $this->register_available_platform( 'directiq', array(
            'name'        => __( 'DirectIQ', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'directiq.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_DirectIQ'
        ));

        $this->register_available_platform( 'elasticemail', array(
            'name'        => __( 'Elastic Email', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'elasticemail.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_ElasticEmail'
        ));

        $this->register_available_platform( 'emailoctopus', array(
            'name'        => __( 'EmailOctopus', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing made easy.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'emailoctopus.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_EmailOctopus'
        ));

        $this->register_available_platform( 'engagebay', array(
            'name'        => __( 'EngageBay', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'All-in-one marketing, sales, and service platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'engagebay.svg',
            'category'    => 'crm',
            'class'       => 'TriggerNinja_EngageBay'
        ));

        $this->register_available_platform( 'flodesk', array(
            'name'        => __( 'Flodesk', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing that’s different by design.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'flodesk.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Flodesk'
        ));

        $this->register_available_platform( 'getresponse', array(
            'name'        => __( 'GetResponse', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing & automation platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'getresponse.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_GetResponse'
        ));

        $this->register_available_platform( 'gohighlevel', array(
            'name'        => __( 'GoHighLevel', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'All-in-one sales and marketing platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'gohighlevel.svg',
            'category'    => 'crm',
            'class'       => 'TriggerNinja_GoHighLevel'
        ));

        $this->register_available_platform( 'kit', array(
            'name'        => __( 'Kit', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing that automates your growth.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'kit.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Kit'
        ));
        
        $this->register_available_platform( 'klaviyo', array(
            'name'        => __( 'Klaviyo', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email Marketing Platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'klaviyo.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Klaviyo'
        ));
        
        $this->register_available_platform( 'lemlist', array(
            'name'        => __( 'Lemlist', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Prospecting tool to automate multi-channel outreach.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'lemlist.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Lemlist'
        ));

        $this->register_available_platform( 'loops', array(
            'name'        => __( 'Loops', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing platform for modern SaaS businesses.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'loops.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Loops'
        ));

        $this->register_available_platform( 'mailchimp', array(
            'name'        => __( 'Mailchimp', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Small Business Marketing Tools.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'mailchimp.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Mailchimp'
        ));

        $this->register_available_platform( 'mailcoach', array(
            'name'        => __( 'Mailcoach', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing platform for Laravel and PHP.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'mailcoach.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Mailcoach'
        ));
        
        $this->register_available_platform( 'mailerlite', array(
            'name'        => __( 'MailerLite', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Create Email Marketing Your Audience Will Love.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'mailerlite.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_MailerLite'
        ));
        
        $this->register_available_platform( 'mailjet', array(
            'name'        => __( 'Mailjet', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing and SMS platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'mailjet.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Mailjet'
        ));

        $this->register_available_platform( 'mailrelay', array(
            'name'        => __( 'MailRelay', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'mailrelay.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_MailRelay'
        ));

        $this->register_available_platform( 'mailwizz', array(
            'name'        => __( 'MailWizz', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing application.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'mailwizz.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_MailWizz'
        ));

        $this->register_available_platform( 'rapidmail', array(
            'name'        => __( 'Rapidmail', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'The simply good email marketing tool.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'rapidmail.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_Rapidmail'
        ));

        $this->register_available_platform( 'saleshandy', array(
            'name'        => __( 'SalesHandy', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Cold email outreach platform.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'saleshandy.svg',
            'category'    => 'crm',
            'class'       => 'TriggerNinja_SalesHandy'
        ));

        $this->register_available_platform( 'sendfox', array(
            'name'        => __( 'SendFox', 'triggerninja-for-ninja-forms' ),
            'description' => __( 'Email marketing for content creators.', 'triggerninja-for-ninja-forms' ),
            'icon'        => 'sendfox.svg',
            'category'    => 'email_marketing',
            'class'       => 'TriggerNinja_SendFox'
        ));
        
        // Hook for additional platform registration
        do_action( 'triggerninja_register_available_platforms', $this );
    }
    
    /**
     * Register an available platform
     * 
     * @param string $id Platform ID
     * @param array $config Platform configuration
     */
    public function register_available_platform( $id, $config )
    {
        $this->available_platforms[ $id ] = array_merge( array(
            'name'        => '',
            'description' => '',
            'icon'        => 'default-icon.svg',
            'category'    => 'other',
            'class'       => ''
        ), $config );
    }
    
    /**
     * Enqueue scripts and styles for the platform manager
     */
    public function enqueue_scripts( $hook )
    {
        // Only load on Ninja Forms settings page
        if( strpos( $hook, 'ninja-forms' ) === false ) return;
        
        wp_enqueue_script( 
            'triggerninja-platform-manager', 
            $this->plugin->url( 'assets/js/platform-manager.js' ), 
            array( 'jquery' ), 
            $this->plugin->version(), 
            true 
        );

        wp_enqueue_script( 
            'triggerninja-platform-connections',
            $this->plugin->url( 'assets/js/platform-connections.js' ),
            array( 'jquery', 'triggerninja-platform-manager' ),
            $this->plugin->version(),
            true
        );
        
        wp_enqueue_style( 
            'triggerninja-platform-manager', 
            $this->plugin->url( 'assets/css/platform-manager.css' ), 
            array(), 
            $this->plugin->version() 
        );
        
        // Localize script for AJAX
        wp_localize_script( 'triggerninja-platform-manager', 'triggerNinjaAjax', array(
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'triggerninja_platform_manager' ),
            'strings' => array(
                'enabling'  => __( 'Enabling...', 'triggerninja-for-ninja-forms' ),
                'disabling' => __( 'Disabling...', 'triggerninja-for-ninja-forms' ),
                'configuring' => __( 'Loading configuration...', 'triggerninja-for-ninja-forms' ),
                'error'     => __( 'An error occurred. Please try again.', 'triggerninja-for-ninja-forms' )
            ),
            'connections' => array(
                'aweber' => array(
                    'missingCode'     => __( 'Please enter the authorization code from AWeber.', 'triggerninja-for-ninja-forms' ),
                    'missingVerifier' => __( 'Missing PKCE code verifier. Please refresh the page and try again.', 'triggerninja-for-ninja-forms' ),
                    'missingNonce'    => __( 'Security data is missing. Please refresh the page and try again.', 'triggerninja-for-ninja-forms' ),
                    'connecting'      => __( 'Connecting...', 'triggerninja-for-ninja-forms' ),
                    'connectSuccess'  => __( 'AWeber connected successfully!', 'triggerninja-for-ninja-forms' ),
                    'connectError'    => __( 'Error connecting AWeber. Please try again.', 'triggerninja-for-ninja-forms' ),
                    'connectErrorPrefix' => __( 'Error connecting AWeber: ', 'triggerninja-for-ninja-forms' ),
                    'disconnectConfirm' => __( 'Are you sure you want to disconnect AWeber?', 'triggerninja-for-ninja-forms' ),
                    'disconnecting'   => __( 'Disconnecting...', 'triggerninja-for-ninja-forms' ),
                    'disconnectSuccess' => __( 'AWeber disconnected successfully.', 'triggerninja-for-ninja-forms' ),
                    'disconnectError' => __( 'Error disconnecting AWeber. Please try again.', 'triggerninja-for-ninja-forms' )
                ),
                'constantcontact' => array(
                    'disconnectConfirm' => __( 'Are you sure you want to disconnect Constant Contact?', 'triggerninja-for-ninja-forms' ),
                    'disconnectSuccess' => __( 'Constant Contact disconnected successfully.', 'triggerninja-for-ninja-forms' ),
                    'disconnectError'   => __( 'Error disconnecting Constant Contact.', 'triggerninja-for-ninja-forms' )
                )
            )
        ));
    }
    
    /**
     * Render the main platform management interface
     * 
     * @param array $field
     * @param string $field_id
     */
    public function render_platform_manager( $field, $field_id )
    {
        if( $field['type'] !== 'triggerninja_platform_manager' ) return;
        
        $enabled_platforms = $this->get_enabled_platforms();
        $available_platforms = $this->get_available_platforms_by_category();
        
        ?>
        <div class="triggerninja-platform-manager">
            <!-- Header Section -->
            <div class="platform-manager-header">
                <h3><?php esc_html_e( 'Platform Management', 'triggerninja-for-ninja-forms' ); ?></h3>
                <p class="description">
                    <?php esc_html_e( 'Enable the email marketing platforms you want to use, then configure their settings. Only enabled platforms will appear in your Ninja Forms.', 'triggerninja-for-ninja-forms' ); ?>
                </p>
            </div>
            
            <!-- Enabled Platforms Section -->
            <div class="enabled-platforms-section" <?php echo empty( $enabled_platforms ) ? 'style="display: none;"' : ''; ?>>
                <h4><?php esc_html_e( 'Enabled Platforms', 'triggerninja-for-ninja-forms' ); ?></h4>
                <div class="platform-grid enabled-grid">
                    <?php foreach( $enabled_platforms as $platform_id => $platform_config ) : ?>
                        <?php $this->render_platform_card( $platform_id, $platform_config, true ); ?>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Search and Filter -->
            <div class="platform-browser">
                <div class="platform-browser-header">
                    <h4><?php esc_html_e( 'Available Platforms', 'triggerninja-for-ninja-forms' ); ?></h4>
                    <div class="platform-search-filter">
                        <input type="text" 
                               id="platform-search" 
                               placeholder="<?php esc_attr_e( 'Search platforms...', 'triggerninja-for-ninja-forms' ); ?>" 
                               class="platform-search-input" />
                        <select id="platform-category-filter" class="platform-category-filter">
                            <option value=""><?php esc_html_e( 'All Categories', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="email_marketing"><?php esc_html_e( 'Email Marketing', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="crm"><?php esc_html_e( 'CRM', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="automation"><?php esc_html_e( 'Marketing Automation', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="other"><?php esc_html_e( 'Other', 'triggerninja-for-ninja-forms' ); ?></option>
                        </select>
                    </div>
                </div>
                
                <!-- Available Platforms Grid -->
                <div class="available-platforms-grid">
                    <?php foreach( $available_platforms as $category => $platforms ) : ?>
                        <?php if( !empty( $platforms ) ) : ?>
                            <div class="platform-category" data-category="<?php echo esc_attr( $category ); ?>">
                                <?php if( count( $available_platforms ) > 1 ) : ?>
                                    <h5 class="category-title"><?php echo esc_html( $this->get_category_label( $category ) ); ?></h5>
                                <?php endif; ?>
                                <div class="platform-grid">
                                    <?php foreach( $platforms as $platform_id => $platform_config ) : ?>
                                        <?php if( !$this->is_platform_enabled( $platform_id ) ) : ?>
                                            <?php $this->render_platform_card( $platform_id, $platform_config, false ); ?>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Platform Configuration Modal -->
        <div id="platform-config-modal" class="platform-config-modal" style="display: none;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3 class="modal-title"></h3>
                    <button type="button" class="modal-close">&times;</button>
                </div>
                <div class="modal-body"></div>
                <div class="modal-footer">
                    <button type="button" class="button button-secondary modal-cancel"><?php esc_html_e( 'Cancel', 'triggerninja-for-ninja-forms' ); ?></button>
                    <button type="button" class="button button-primary modal-save"><?php esc_html_e( 'Save Settings', 'triggerninja-for-ninja-forms' ); ?></button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render a platform card
     * 
     * @param string $platform_id
     * @param array $platform_config
     * @param bool $enabled
     */
    private function render_platform_card( $platform_id, $platform_config, $enabled = false )
    {
        ?>
        <div class="platform-card <?php echo $enabled ? 'enabled' : 'available'; ?>" 
             data-platform="<?php echo esc_attr( $platform_id ); ?>"
             data-category="<?php echo esc_attr( $platform_config['category'] ?? 'other' ); ?>">
            
            <div class="platform-icon">
                <img src="<?php echo esc_url( $this->plugin->url( 'assets/images/platforms/' . $platform_config['icon'] ) ); ?>" 
                     alt="<?php echo esc_attr( $platform_config['name'] ); ?>"
                     onerror="this.src='<?php echo esc_url( $this->plugin->url( 'assets/images/platforms/default-icon.svg' ) ); ?>'" />
            </div>
            
            <div class="platform-info">
                <h5 class="platform-name"><?php echo esc_html( $platform_config['name'] ); ?></h5>
                <p class="platform-description"><?php echo esc_html( $platform_config['description'] ); ?></p>
            </div>
            
            <div class="platform-actions">
                <?php if( $enabled ) : ?>
                    <button type="button" 
                            class="button button-primary platform-configure" 
                            data-platform="<?php echo esc_attr( $platform_id ); ?>">
                        <?php esc_html_e( 'Configure', 'triggerninja-for-ninja-forms' ); ?>
                    </button>
                    <button type="button" 
                            class="button button-secondary platform-disable" 
                            data-platform="<?php echo esc_attr( $platform_id ); ?>">
                        <?php esc_html_e( 'Disable', 'triggerninja-for-ninja-forms' ); ?>
                    </button>
                <?php else : ?>
                    <button type="button" 
                            class="button button-primary platform-enable" 
                            data-platform="<?php echo esc_attr( $platform_id ); ?>">
                        <?php esc_html_e( 'Enable', 'triggerninja-for-ninja-forms' ); ?>
                    </button>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get enabled platforms
     * 
     * @return array
     */
    public function get_enabled_platforms()
    {
        $enabled = get_option( 'triggerninja_enabled_platforms', array() );
        $result = array();
        
        foreach( $enabled as $platform_id ) {
            if( isset( $this->available_platforms[ $platform_id ] ) ) {
                $result[ $platform_id ] = $this->available_platforms[ $platform_id ];
            }
        }
        
        return $result;
    }
    
    /**
     * Get available platforms organized by category
     * 
     * @return array
     */
    private function get_available_platforms_by_category()
    {
        $by_category = array();
        
        foreach( $this->available_platforms as $platform_id => $config ) {
            $category = $config['category'] ?? 'other';
            $by_category[ $category ][ $platform_id ] = $config;
        }
        
        // Sort platforms alphabetically within each category
        foreach( $by_category as $category => $platforms ) {
            uasort( $by_category[ $category ], function( $a, $b ) {
                return strcmp( $a['name'], $b['name'] );
            });
        }
        
        return $by_category;
    }
    
    /**
     * Check if a platform is enabled
     * 
     * @param string $platform_id
     * @return bool
     */
    private function is_platform_enabled( $platform_id )
    {
        $enabled = get_option( 'triggerninja_enabled_platforms', array() );
        return in_array( $platform_id, $enabled );
    }
    
    /**
     * Get category label
     * 
     * @param string $category
     * @return string
     */
    private function get_category_label( $category )
    {
        $labels = array(
            'email_marketing' => __( 'Email Marketing', 'triggerninja-for-ninja-forms' ),
            'crm'            => __( 'CRM', 'triggerninja-for-ninja-forms' ),
            'automation'     => __( 'Marketing Automation', 'triggerninja-for-ninja-forms' ),
            'other'          => __( 'Other', 'triggerninja-for-ninja-forms' )
        );
        
        return $labels[ $category ] ?? ucfirst( $category );
    }
    
    /**
     * AJAX handler to toggle platform enabled/disabled
     */
    public function ajax_toggle_platform()
    {
        check_ajax_referer( 'triggerninja_platform_manager', 'nonce' );
        
        if( !current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $platform_id = sanitize_text_field( wp_unslash( $_POST['platform_id'] ?? '' ) );
        $action = sanitize_text_field( wp_unslash( $_POST['action_type'] ?? '' ) ); // 'enable' or 'disable'
        
        if( !$platform_id || !in_array( $action, array( 'enable', 'disable' ) ) ) {
            wp_send_json_error( __( 'Invalid parameters.', 'triggerninja-for-ninja-forms' ) );
        }
        
        if( !isset( $this->available_platforms[ $platform_id ] ) ) {
            wp_send_json_error( __( 'Platform not found.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $enabled_platforms = get_option( 'triggerninja_enabled_platforms', array() );
        
        if( $action === 'enable' ) {
            if( !in_array( $platform_id, $enabled_platforms ) ) {
                $enabled_platforms[] = $platform_id;
                update_option( 'triggerninja_enabled_platforms', $enabled_platforms );
            }
        } else {
            $enabled_platforms = array_diff( $enabled_platforms, array( $platform_id ) );
            update_option( 'triggerninja_enabled_platforms', array_values( $enabled_platforms ) );
        }
        
        wp_send_json_success( array(
            'message' => $action === 'enable' ? __( 'Platform enabled successfully.', 'triggerninja-for-ninja-forms' ) : __( 'Platform disabled successfully.', 'triggerninja-for-ninja-forms' ),
            'enabled_platforms' => get_option( 'triggerninja_enabled_platforms', array() )
        ) );
    }
    
    /**
     * AJAX handler to get platform configuration form
     */
    public function ajax_configure_platform()
    {
        check_ajax_referer( 'triggerninja_platform_manager', 'nonce' );
        
        if( !current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $platform_id = sanitize_text_field( wp_unslash( $_POST['platform_id'] ?? '' ) );
        
        if( !$platform_id || !$this->is_platform_enabled( $platform_id ) ) {
            wp_send_json_error( __( 'Platform not found or not enabled.', 'triggerninja-for-ninja-forms' ) );
        }
        
        // Get the platform instance to retrieve settings fields
        $platform = $this->plugin->get_platform( 'triggerninja_' . $platform_id );
        
        if( !$platform ) {
            wp_send_json_error( __( 'Platform instance not found.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $settings_fields = $platform->get_settings_fields();
        
        ob_start();
        foreach( $settings_fields as $field_id => $field_config ) {
            echo '<div class="platform-field">';
            $this->render_field( $field_id, $field_config );
            echo '</div>';
        }
        $form_html = ob_get_clean();
        
        wp_send_json_success( array(
            /* translators: %s is a platform name. */
            'title' => sprintf( __( 'Configure %s', 'triggerninja-for-ninja-forms' ), $platform->get_name() ),
            'form_html' => $form_html
        ) );
    }
    
    /**
     * Render individual field (simplified version for modal)
     * 
     * @param string $field_id
     * @param array $field_config
     */
    private function render_field( $field_id, $field_config )
    {
        $value = get_option( $field_id, '' );
        $type = isset( $field_config['type'] ) ? $field_config['type'] : 'textbox';
        $label = isset( $field_config['label'] ) ? $field_config['label'] : '';
        $desc = isset( $field_config['desc'] ) ? $field_config['desc'] : '';
        
        // Handle hidden fields without wrapper
        if( $type === 'hidden' ) {
            $field_value = isset( $field_config['value'] ) ? $field_config['value'] : $value;
            ?>
            <input type="hidden" 
                   id="<?php echo esc_attr( $field_id ); ?>" 
                   name="<?php echo esc_attr( $field_id ); ?>" 
                   value="<?php echo esc_attr( $field_value ); ?>" />
            <?php
            return;
        }
        
        ?>
        <div class="field-wrapper field-<?php echo esc_attr( $type ); ?>">
            <?php if( $label ) : ?>
                <label for="<?php echo esc_attr( $field_id ); ?>">
                    <?php echo esc_html( $label ); ?>
                </label>
            <?php endif; ?>
            
            <?php
            switch( $type ) {
                case 'textbox':
                case 'text':
                    ?>
                    <input type="text" 
                           id="<?php echo esc_attr( $field_id ); ?>" 
                           name="<?php echo esc_attr( $field_id ); ?>" 
                           value="<?php echo esc_attr( $value ); ?>" 
                           class="regular-text" />
                    <?php
                    break;
                    
                case 'password':
                    ?>
                    <input type="password" 
                           id="<?php echo esc_attr( $field_id ); ?>" 
                           name="<?php echo esc_attr( $field_id ); ?>" 
                           value="<?php echo esc_attr( $value ); ?>" 
                           class="regular-text" />
                    <?php
                    break;
                    
                case 'textarea':
                    ?>
                    <textarea id="<?php echo esc_attr( $field_id ); ?>" 
                              name="<?php echo esc_attr( $field_id ); ?>" 
                              rows="4" 
                              class="large-text"><?php echo esc_textarea( $value ); ?></textarea>
                    <?php
                    break;
                    
                case 'select':
                    $options = isset( $field_config['options'] ) ? $field_config['options'] : array();
                    ?>
                    <select id="<?php echo esc_attr( $field_id ); ?>" 
                            name="<?php echo esc_attr( $field_id ); ?>">
                        <?php foreach( $options as $option_value => $option_label ) : ?>
                            <option value="<?php echo esc_attr( $option_value ); ?>" 
                                    <?php selected( $value, $option_value ); ?>>
                                <?php echo esc_html( $option_label ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <?php
                    break;
                    
                case 'checkbox':
                    ?>
                    <label>
                        <input type="checkbox" 
                               id="<?php echo esc_attr( $field_id ); ?>" 
                               name="<?php echo esc_attr( $field_id ); ?>" 
                               value="1" 
                               <?php checked( $value, '1' ); ?> />
                        <?php echo esc_html( $label ); ?>
                    </label>
                    <?php
                    break;
                    
                case 'html':
                    $content = isset( $field_config['content'] ) ? $field_config['content'] : '';
                    echo wp_kses_post( $content );
                    break;
                    
                case 'button':
                    $button_text = isset( $field_config['button_text'] ) ? $field_config['button_text'] : __( 'Button', 'triggerninja-for-ninja-forms' );
                    $button_class = isset( $field_config['button_class'] ) ? $field_config['button_class'] : 'button';
                    ?>
                    <button type="button" 
                            id="<?php echo esc_attr( $field_id ); ?>" 
                            class="<?php echo esc_attr( $button_class ); ?>"
                            data-field-id="<?php echo esc_attr( $field_id ); ?>">
                        <?php echo esc_html( $button_text ); ?>
                    </button>
                    <?php
                    break;
                    
            }
            ?>
            
            <?php if( $desc ) : ?>
                <p class="description"><?php echo wp_kses_post( $desc ); ?></p>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Get the settings structure for Ninja Forms
     * 
     * @return array
     */
    public function get_settings_structure()
    {
        return array(
            'triggerninja_platform_manager' => array(
                'id'    => 'triggerninja_platform_manager',
                'type'  => 'triggerninja_platform_manager',
                'label' => __( 'Platform Management', 'triggerninja-for-ninja-forms' ),
                'desc'  => __( 'Enable and configure your email marketing platform integrations.', 'triggerninja-for-ninja-forms' ),
            ),
        );
    }
    
    /**
     * Render platform manager as a standalone page
     * 
     * This is called by the admin page class
     */
    public function render_platform_manager_page()
    {
        $enabled_platforms = $this->get_enabled_platforms();
        $available_platforms = $this->get_available_platforms_by_category();
        
        ?>
        <div class="triggerninja-platform-manager">
            <!-- Enabled Platforms Section -->
            <div class="enabled-platforms-section" <?php echo empty( $enabled_platforms ) ? 'style="display: none;"' : ''; ?>>
                <h4><?php esc_html_e( 'Enabled Platforms', 'triggerninja-for-ninja-forms' ); ?></h4>
                <div class="platform-grid enabled-grid">
                    <?php foreach( $enabled_platforms as $platform_id => $platform_config ) : ?>
                        <?php $this->render_platform_card( $platform_id, $platform_config, true ); ?>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Search and Filter -->
            <div class="platform-browser">
                <div class="platform-browser-header">
                    <h4><?php esc_html_e( 'Available Platforms', 'triggerninja-for-ninja-forms' ); ?></h4>
                    <div class="platform-search-filter">
                        <input type="text" 
                               id="platform-search" 
                               placeholder="<?php esc_attr_e( 'Search platforms...', 'triggerninja-for-ninja-forms' ); ?>" 
                               class="platform-search-input" />
                        <select id="platform-category-filter" class="platform-category-filter">
                            <option value=""><?php esc_html_e( 'All Categories', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="email_marketing"><?php esc_html_e( 'Email Marketing', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="crm"><?php esc_html_e( 'CRM', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="automation"><?php esc_html_e( 'Marketing Automation', 'triggerninja-for-ninja-forms' ); ?></option>
                            <option value="other"><?php esc_html_e( 'Other', 'triggerninja-for-ninja-forms' ); ?></option>
                        </select>
                    </div>
                </div>
                
                <!-- Available Platforms Grid -->
                <div class="available-platforms-grid">
                    <?php foreach( $available_platforms as $category => $platforms ) : ?>
                        <?php if( !empty( $platforms ) ) : ?>
                            <div class="platform-category" data-category="<?php echo esc_attr( $category ); ?>">
                                <?php if( count( $available_platforms ) > 1 ) : ?>
                                    <h5 class="category-title"><?php echo esc_html( $this->get_category_label( $category ) ); ?></h5>
                                <?php endif; ?>
                                <div class="platform-grid">
                                    <?php foreach( $platforms as $platform_id => $platform_config ) : ?>
                                        <?php if( !$this->is_platform_enabled( $platform_id ) ) : ?>
                                            <?php $this->render_platform_card( $platform_id, $platform_config, false ); ?>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Platform Configuration Modal -->
        <div id="platform-config-modal" class="platform-config-modal" style="display: none;">
            <div class="modal-content">
                <div class="modal-header">
                    <h3 class="modal-title"></h3>
                    <button type="button" class="modal-close">&times;</button>
                </div>
                <div class="modal-body"></div>
                <div class="modal-footer">
                    <button type="button" class="button button-secondary modal-cancel"><?php esc_html_e( 'Cancel', 'triggerninja-for-ninja-forms' ); ?></button>
                    <button type="button" class="button button-primary modal-save"><?php esc_html_e( 'Save Settings', 'triggerninja-for-ninja-forms' ); ?></button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * AJAX handler to save platform configuration
     */
    public function ajax_save_platform_config()
    {
                check_ajax_referer( 'triggerninja_platform_manager', 'nonce' );
        
                if( !current_user_can( 'manage_options' ) ) {
                    wp_die( esc_html__( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) );
                }        
        $platform_id = sanitize_text_field( wp_unslash( $_POST['platform_id'] ?? '' ) );
        
        if( !$platform_id || !$this->is_platform_enabled( $platform_id ) ) {
            wp_send_json_error( __( 'Platform not found or not enabled.', 'triggerninja-for-ninja-forms' ) );
        }
        
        // Get the platform instance to retrieve settings fields structure
        $platform = $this->plugin->get_platform( 'triggerninja_' . $platform_id );
        
        if( !$platform ) {
            wp_send_json_error( __( 'Platform instance not found.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $settings_fields = $platform->get_settings_fields();
        
        // Validate and save each field
        $saved_fields = array();
        foreach( $settings_fields as $field_id => $field_config ) {
            if( isset( $_POST[ $field_id ] ) ) {
                $value = sanitize_text_field( wp_unslash( $_POST[ $field_id ] ) );

                if ( ! is_scalar($value) ) {
                    $value = '';
                }
                
                // Sanitize based on field type
                $field_type = $field_config['type'] ?? 'textbox';
                switch( $field_type ) {
                    case 'password':
                    case 'textbox':
                    case 'text':
                        $value = sanitize_text_field( $value );
                        break;
                    case 'textarea':
                        $value = sanitize_textarea_field( $value );
                        break;
                    case 'email':
                        $value = sanitize_email( $value );
                        break;
                    case 'checkbox':
                        $value = $value ? '1' : '0';
                        break;
                    case 'select':
                        // Validate against allowed options
                        $options = array_keys( $field_config['options'] ?? array() );
                        $value = in_array( $value, $options ) ? $value : '';
                        break;
                    default:
                        $value = sanitize_text_field( $value );
                }
                
                // Save the field
                update_option( $field_id, $value );
                $saved_fields[ $field_id ] = $value;
            }
        }
        
        // Reload platform settings to ensure they're fresh
        // if( method_exists( $platform, 'reload_settings' ) ) {
        //     $platform->reload_settings();
        // }
        
        /**
         * Action hook after platform configuration is saved
         * 
         * @param string $platform_id Platform ID
         * @param array $saved_fields Saved field values
         * @param TriggerNinja_Platform_Interface $platform Platform instance
         */
        do_action( 'triggerninja_platform_config_saved', $platform_id, $saved_fields, $platform );
        
        wp_send_json_success( array(
            /* translators: %s is a platform name. */
            'message' => sprintf( __( '%s configuration saved successfully!', 'triggerninja-for-ninja-forms' ), $platform->get_name() ),
            'saved_fields' => array_keys( $saved_fields )
        ) );
    }
    
    /**
     * AJAX handler for AWeber authorization
     */
     public function ajax_authorize_aweber()
    {
        check_ajax_referer( 'triggerninja_aweber_auth', 'nonce' );
        
        if( !current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $auth_code = sanitize_text_field( wp_unslash( $_POST['auth_code'] ?? '' ) );
        $code_verifier = sanitize_text_field( wp_unslash( $_POST['code_verifier'] ?? '' ) );
        
        if( empty( $auth_code ) ) {
            wp_send_json_error( __( 'Authorization code is required.', 'triggerninja-for-ninja-forms' ) );
        }
        
        if( empty( $code_verifier ) ) {
            wp_send_json_error( __( 'Code verifier is missing.', 'triggerninja-for-ninja-forms' ) );
        }
        
        // Get AWeber platform instance
        $platform = $this->plugin->get_platform( 'triggerninja_aweber' );
        
        if( !$platform ) {
            wp_send_json_error( __( 'AWeber platform not found.', 'triggerninja-for-ninja-forms' ) );
        }
        
        // Exchange authorization code for tokens
        $api = new TriggerNinja_Aweber_API();
        $token_response = $api->exchange_code_for_tokens( $auth_code, $code_verifier );
        
        if( $token_response->status >= 200 && $token_response->status < 300 && $token_response->details ) {
            $tokens = $token_response->details;
            
            // Save tokens
            update_option( 'triggerninja_aweber_access_token', $tokens->access_token );
            update_option( 'triggerninja_aweber_refresh_token', $tokens->refresh_token ?? '' );
            update_option( 'triggerninja_aweber_token_expires', time() + ( $tokens->expires_in ?? 3600 ) );
            
            // Save the authorization code
            update_option( 'triggerninja_aweber_auth_code', $auth_code );
            
            // Reload platform settings
            // if( method_exists( $platform, 'reload_settings' ) ) {
            //     $platform->reload_settings();
            // }
            
            wp_send_json_success( array(
                'message' => __( 'AWeber authorization successful! You are now connected.', 'triggerninja-for-ninja-forms' ),
                'status' => 'connected',
                'status_html' => wp_kses_post( '<span style="color: green;">' . __( 'Connected', 'triggerninja-for-ninja-forms' ) . '</span>' )
            ) );
        } else {
            $error_message = $token_response->error ? $token_response->error : __( 'Failed to authorize with AWeber', 'triggerninja-for-ninja-forms' );
            wp_send_json_error( array(
                'message' => $error_message,
                'status' => 'not_connected'
            ) );
        }
    }
    
    /**
     * Get available platforms registry
     * 
     * @return array
     */
    public function get_available_platforms()
    {
        return $this->available_platforms;
    }
}
