<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * TriggerNinja HTTP Client
 * 
 * Centralized HTTP request handler with consistent headers, logging, and response handling
 */
class TriggerNinja_Http_Client
{
    /**
     * Plugin version for User-Agent
     */
    const PLUGIN_VERSION = '1.0.0';
    
    /**
     * Default headers for all requests
     * 
     * @var array
     */
    private static $default_headers = array(
        'Content-Type' => 'application/json',
        'Accept' => 'application/json'
    );
    
    /**
     * Make an HTTP request with consistent headers and logging
     * 
     * @param string $method HTTP method (GET, POST, PUT, DELETE)
     * @param string $url Full URL for the request
     * @param array $data Request data
     * @param array $headers Additional headers
     * @param array $options Additional wp_remote_request options
     * @param int $form_id Form ID for logging purposes
     * @return TriggerNinja_Response
     */
    public static function request( $method, $url, $data = array(), $headers = array(), $options = array(), $form_id = 0, $platform_id = null )
    {
        // Prepare default arguments
        $args = array_merge( array(
            'method' => strtoupper( $method ),
            'headers' => self::get_default_headers(),
            'timeout' => 30
        ), $options );
        
        // Merge additional headers
        $args['headers'] = array_merge( $args['headers'], $headers );
        
        // Add body for POST/PUT requests (only if not already provided in options)
        if( in_array( $args['method'], array( 'POST', 'PUT', 'PATCH' ) ) && ! isset( $args['body'] ) ) {
            if ( isset( $args['headers']['Content-Type'] ) && stripos( $args['headers']['Content-Type'], 'application/json' ) !== false ) {
                $args['body'] = json_encode( $data );
            } else {
                $args['body'] = $data;
            }
        } elseif( $args['method'] === 'GET' && !empty( $data ) ) {
            // For GET requests, add data as query parameters
            $url = add_query_arg( $data, $url );
        }
        
        // Make the request
        $response = wp_remote_request( $url, $args );
        
        // Log request and response if logging is enabled
        self::maybe_log_request( $method, $url, $data, $headers, $response, $form_id, $platform_id );
        
        return TriggerNinja_Response::createFromResponse( $response );
    }
    
    /**
     * GET request wrapper
     * 
     * @param string $url Full URL
     * @param array $data Query parameters
     * @param array $headers Additional headers
     * @param array $options Additional options
     * @param int $form_id Form ID for logging purposes
     * @return TriggerNinja_Response
     */
    public static function get( $url, $data = array(), $headers = array(), $options = array(), $form_id = 0, $platform_id = null )
    {
        return self::request( 'GET', $url, $data, $headers, $options, $form_id, $platform_id );
    }
    
    /**
     * POST request wrapper
     * 
     * @param string $url Full URL
     * @param array $data Request body data
     * @param array $headers Additional headers
     * @param array $options Additional options
     * @param int $form_id Form ID for logging purposes
     * @return TriggerNinja_Response
     */
    public static function post( $url, $data = array(), $headers = array(), $options = array(), $form_id = 0, $platform_id = null )
    {
        return self::request( 'POST', $url, $data, $headers, $options, $form_id, $platform_id );
    }
    
    /**
     * PUT request wrapper
     * 
     * @param string $url Full URL
     * @param array $data Request body data
     * @param array $headers Additional headers
     * @param array $options Additional options
     * @param int $form_id Form ID for logging purposes
     * @return TriggerNinja_Response
     */
    public static function put( $url, $data = array(), $headers = array(), $options = array(), $form_id = 0, $platform_id = null )
    {
        return self::request( 'PUT', $url, $data, $headers, $options, $form_id, $platform_id );
    }
    
    /**
     * Get default headers including User-Agent
     * 
     * @return array
     */
    private static function get_default_headers()
    {
        $headers = self::$default_headers;
        $headers['User-Agent'] = sprintf( 'TriggerNinja/%s WordPress/%s', 
            self::PLUGIN_VERSION, 
            get_bloginfo( 'version' ) 
        );
        
        return $headers;
    }
    
    /**
     * Maybe log request and response to database
     * 
     * @param string $method HTTP method
     * @param string $url Request URL
     * @param array $data Request data
     * @param array $headers Request headers
     * @param mixed $response WordPress HTTP response
     * @param int $form_id Form ID
     */
    private static function maybe_log_request( $method, $url, $data, $headers, $response, $form_id = 0, $platform_id = null )
    {
        // Only log if logging is enabled
        if( ! TriggerNinja_Logger::is_logging_enabled() ) {
            return;
        }
        
        // Only log requests that have a valid form_id (actual form submissions)
        // Skip logging administrative requests like list fetching during configuration
        if( empty( $form_id ) || $form_id <= 0 ) {
            return;
        }
        
        $platform = $platform_id;
        
        // Get response data
        $status_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $response_data = json_decode( $response_body, true ) ?: $response_body;
        
        // Determine status and error message
        $status = ( $status_code >= 200 && $status_code < 300 ) ? 'success' : 'error';
        $error_message = '';
        
        if( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            $status = 'error';
            $status_code = 0;
        } elseif( $status === 'error' ) {
            // Try to extract error message from response
            if( is_array( $response_data ) && isset( $response_data['message'] ) ) {
                $error_message = $response_data['message'];
            } elseif( is_array( $response_data ) && isset( $response_data['error'] ) ) {
                $error_message = is_string( $response_data['error'] ) ? $response_data['error'] : 'API Error';
            } else {
                $error_message = 'HTTP ' . $status_code . ' Error';
            }
        }
        
        // Remove sensitive data from request for logging
        $safe_data = self::sanitize_request_data( $data, $headers );
        
        // Log the request
        TriggerNinja_Logger::log_request(
            $platform,
            $method,
            $url,
            $safe_data,
            $response_data,
            $status_code,
            $status,
            $error_message,
            $form_id
        );
    }
    

    
    /**
     * Sanitize request data by removing sensitive information
     * 
     * @param array $data Request data
     * @param array $headers Request headers
     * @return array Sanitized data
     */
    private static function sanitize_request_data( $data, $headers )
    {
        $safe_data = $data;
        
        // Remove sensitive headers (but keep structure for debugging)
        $safe_headers = $headers;
        foreach( $safe_headers as $key => $value ) {
            if( stripos( $key, 'authorization' ) !== false ||
                stripos( $key, 'api-key' ) !== false ||
                stripos( $key, 'api_key' ) !== false ||
                stripos( $key, 'token' ) !== false ) {
                $safe_headers[ $key ] = str_repeat( '*', min( 8, strlen( $value ) ) );
            }
        }
        
        return array(
            'data' => $safe_data,
            'headers' => $safe_headers
        );
    }
}

/**
 * TriggerNinja Response
 * 
 * Handles API response parsing and provides easy access to response data
 */
class TriggerNinja_Response
{
    private $status;
    private $details;
    private $error;

    public function __construct( $status, $details = array(), $error = null )
    {
        $this->status  = $status;
        $this->details = $details;
        $this->error   = $error;
    }

    public function __get( $property )
    {
        if( ! isset( $this->$property ) ) return null;
        return $this->$property;
    }

    public static function createFromResponse( $response )
    {
        $status = wp_remote_retrieve_response_code( $response );

        if( is_wp_error( $response ) ){
            // Return a "null" object without data.
            return new self( $status, array(), $response->get_error_message() );
        }

        $body = json_decode( wp_remote_retrieve_body( $response ) );

        // Handle error responses
        if( $status >= 400 ) {
            $error_message = isset( $body->error->message ) ? $body->error->message : 'Unknown error';
            return new self( $status, array(), $error_message );
        }

        if( isset( $body->data ) ){
            // Paged data is stored nested.
            return new self( $status, $body->data );
        }

        return new self( $status, $body );
    }
}
