<?php if ( ! defined( 'ABSPATH' ) ) exit;

class TriggerNinja_Plugin
{
    private $version;
    private $url;
    private $dir;
    private $platforms = array();
    private $platform_manager;
    private $admin_page;
    
    /**
     * Plugin edition (free/pro)
     * 
     * @var string
     */
    private $edition = 'free';
    
    /**
     * Minimum WordPress version
     * 
     * @var string
     */
    const MIN_WP_VERSION = '5.6';
    
    /**
     * Minimum PHP version
     * 
     * @var string
     */
    const MIN_PHP_VERSION = '7.4';
    
    /**
     * Minimum Ninja Forms version
     * 
     * @var string
     */
    const MIN_NF_VERSION = '3.0';
    
    /**
     * Free version field limit
     * 
     * @var array
     */
    const FREE_FIELDS = array( 'email_address', 'first_name', 'last_name' );

    public function __construct( $version, $file )
    {
        $this->version = $version;
        $this->url = plugin_dir_url( $file );
        $this->dir = plugin_dir_path( $file );
    }

    public function init()
    {
        // Check compatibility first
        if( ! $this->check_compatibility() ) {
            return; // Don't initialize if compatibility check fails
        }
        
        add_action( 'admin_init',     array( $this, 'setup_license'   ) );

        add_action( 'ninja_forms_loaded', array( $this, 'setup_submission_metabox' ) );

        add_filter( 'ninja_forms_plugin_settings',        array( $this, 'plugin_settings'        ), 10, 1 );
        add_filter( 'ninja_forms_plugin_settings_groups', array( $this, 'plugin_settings_groups' ), 10, 1 );

        // Initialize platform manager
        $this->platform_manager = new TriggerNinja_Platform_Manager( $this );
        
        // Initialize admin page
        if( is_admin() ) {
            $this->admin_page = new TriggerNinja_Admin_Page( $this );
        }
        
        // Initialize platforms first
        $this->init_platforms();
        
        // Register actions after platforms are initialized
        add_filter( 'ninja_forms_register_actions', array( $this, 'register_actions' ) );
        
        // Setup AJAX handlers
        $this->setup_ajax_handlers();
        
        /**
         * Hook for when TriggerNinja is fully initialized
         * 
         * @param TriggerNinja_Plugin $plugin Plugin instance
         */
        do_action( 'triggerninja_initialized', $this );
    }

    /*
    |--------------------------------------------------------------------------
    | Platform Management
    |--------------------------------------------------------------------------
    */

    /**
     * Initialize enabled platforms only
     */
    public function init_platforms()
    {
        // Get enabled platforms from settings
        $enabled_platforms = get_option( 'triggerninja_enabled_platforms', array() );
        
        // Register only enabled platforms
        foreach( $enabled_platforms as $platform_id ) {
            $this->register_enabled_platform( $platform_id );
        }

        // Hook for third-party platform registration
        do_action( 'triggerninja_register_platforms', $this );
    }
    
    /**
     * Register an enabled platform instance
     * 
     * @param string $platform_id
     */
    private function register_enabled_platform( $platform_id )
    {
        $class_name = $this->get_platform_class_name( $platform_id );
        
        if( class_exists( $class_name ) ) {
            $platform_instance = new $class_name();
            $this->register_platform( 'triggerninja_' . $platform_id, $platform_instance );
        }
    }
    
    /**
     * Get platform class name from platform ID
     * 
     * @param string $platform_id
     * @return string
     */
    private function get_platform_class_name( $platform_id )
    {
        // Convert platform_id to class name (e.g., 'emailoctopus' -> 'TriggerNinja_EmailOctopus')
        $class_suffix = str_replace( '_', '', ucwords( $platform_id, '_' ) );
        return 'TriggerNinja_' . $class_suffix;
    }

    /**
     * Register a new platform
     * 
     * @param string $id
     * @param TriggerNinja_Platform_Interface $platform
     */
    public function register_platform( $id, TriggerNinja_Platform_Interface $platform )
    {
        $this->platforms[ $id ] = $platform;
    }

    /**
     * Get registered platform
     * 
     * @param string $id
     * @return TriggerNinja_Platform_Interface|null
     */
    public function get_platform( $id )
    {
        return isset( $this->platforms[ $id ] ) ? $this->platforms[ $id ] : null;
    }

    /**
     * Get all registered platforms
     * 
     * @return array
     */
    public function get_platforms()
    {
        return $this->platforms;
    }
    
    /**
     * Get platform manager instance
     * 
     * @return TriggerNinja_Platform_Manager
     */
    public function get_platform_manager()
    {
        return $this->platform_manager;
    }

    /*
    |--------------------------------------------------------------------------
    | Hooks and Filters
    |--------------------------------------------------------------------------
    */

    public function setup_license()
    {
        if ( ! class_exists( 'NF_Extension_Updater' ) ) return;
        new NF_Extension_Updater( 'TriggerNinja - Actions for Ninja Forms', $this->version(), 'TriggerNinja Team', $this->dir(), 'triggerninja-for-ninja-forms' );
    }

    public function setup_submission_metabox()
    {
        if( ! is_admin() ) return;
        require_once $this->dir( 'includes/metabox.php' );
        new TriggerNinja_Metabox();
    }

    public function register_actions( $actions )
    {
        // Register separate actions for each platform
        foreach( $this->platforms as $platform_id => $platform ) {
            $action_class = $this->get_platform_action_class( $platform_id );
            if( class_exists( $action_class ) ) {
                $actions[ $platform_id ] = new $action_class();
            }
        }
        
        return $actions;
    }

    public function plugin_settings( $settings )
    {
        // Platform management is now handled via dedicated admin page
        // No longer adding settings to Ninja Forms settings page
        return $settings;
    }

    public function plugin_settings_groups( $groups )
    {
        // Platform management is now handled via dedicated admin page
        // No longer adding settings group to Ninja Forms settings page
        return $groups;
    }

    /*
    |--------------------------------------------------------------------------
    | Version & Licensing Methods
    |--------------------------------------------------------------------------
    */
    
    /**
     * Check if pro features are available
     * 
     * @return bool
     */
    public function is_pro()
    {
        /**
         * Filter to enable pro features
         * 
         * @param bool $is_pro Default is false
         * @return bool
         */
        return apply_filters( 'triggerninja_is_pro', false );
    }
    
    /**
     * Get edition (free/pro)
     * 
     * @return string
     */
    public function get_edition()
    {
        return $this->is_pro() ? 'pro' : 'free';
    }
    
    /**
     * Check version compatibility
     * 
     * @return bool
     */
    public function check_compatibility()
    {
        $compatible = true;
        
        // Check WordPress version
        if( version_compare( get_bloginfo( 'version' ), self::MIN_WP_VERSION, '<' ) ) {
            add_action( 'admin_notices', array( $this, 'wp_version_notice' ) );
            $compatible = false;
        }
        
        // Check PHP version
        if( version_compare( PHP_VERSION, self::MIN_PHP_VERSION, '<' ) ) {
            add_action( 'admin_notices', array( $this, 'php_version_notice' ) );
            $compatible = false;
        }
        
        // Check Ninja Forms version
        if( version_compare( get_option( 'ninja_forms_version', '0.0.0' ), self::MIN_NF_VERSION, '<' ) ) {
            add_action( 'admin_notices', array( $this, 'nf_version_notice' ) );
            $compatible = false;
        }
        
        return $compatible;
    }
    
    /**
     * WordPress version notice
     */
    public function wp_version_notice()
    {
        echo '<div class="notice notice-error"><p>';
        /* translators: %1$s is the required WordPress version, %2$s is the current WordPress version. */
        echo esc_html( sprintf( __( 'TriggerNinja requires WordPress %1$s or higher. You are running %2$s.', 'triggerninja-for-ninja-forms' ), self::MIN_WP_VERSION, get_bloginfo( 'version' ) ) );
        echo '</p></div>';
    }
    
    /**
     * PHP version notice
     */
    public function php_version_notice()
    {
        echo '<div class="notice notice-error"><p>';
        /* translators: %1$s is the required PHP version, %2$s is the current PHP version. */
        echo esc_html( sprintf( __( 'TriggerNinja requires PHP %1$s or higher. You are running %2$s.', 'triggerninja-for-ninja-forms' ), self::MIN_PHP_VERSION, PHP_VERSION ) );
        echo '</p></div>';
    }
    
    /**
     * Ninja Forms version notice
     */
    public function nf_version_notice()
    {
        echo '<div class="notice notice-error"><p>';
        /* translators: %s is the required Ninja Forms version. */
        echo esc_html( sprintf( __( 'TriggerNinja requires Ninja Forms %s or higher. Please update Ninja Forms.', 'triggerninja-for-ninja-forms' ), self::MIN_NF_VERSION ) );
        echo '</p></div>';
    }

    /*
    |--------------------------------------------------------------------------
    | AWeber OAuth2 AJAX Handlers
    |--------------------------------------------------------------------------
    */
    
    /**
     * Setup AWeber OAuth2 AJAX handlers
     */
    private function setup_ajax_handlers()
    {
        add_action( 'wp_ajax_triggerninja_aweber_generate_auth_url', array( $this, 'ajax_aweber_generate_auth_url' ) );
        add_action( 'wp_ajax_triggerninja_aweber_exchange_token', array( $this, 'ajax_aweber_exchange_token' ) );
        add_action( 'wp_ajax_triggerninja_aweber_disconnect', array( $this, 'ajax_aweber_disconnect' ) );
        // Constant Contact OAuth handlers
        add_action( 'wp_ajax_triggerninja_constantcontact_generate_auth_url', array( $this, 'ajax_constantcontact_generate_auth_url' ) );
        add_action( 'wp_ajax_triggerninja_constantcontact_exchange_token', array( $this, 'ajax_constantcontact_exchange_token' ) );
        add_action( 'wp_ajax_triggerninja_constantcontact_disconnect', array( $this, 'ajax_constantcontact_disconnect' ) );
    }
    
    /**
     * Generate AWeber authorization URL with PKCE
     */
         public function ajax_aweber_generate_auth_url()
        {
            // Verify nonce and permissions
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
            if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) ), 'triggerninja_aweber_auth' ) ) {
                wp_send_json_error( array( 'message' => __( 'Security check failed.', 'triggerninja-for-ninja-forms' ) ) );
            }        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) ) );
        }
        
        try {
            // Generate PKCE codes
            $pkce = TriggerNinja_Aweber_API::generate_pkce();
            $nonce = wp_create_nonce( 'triggerninja_aweber_auth' );
            
            // Get authorization URL
            $auth_url = TriggerNinja_Aweber_API::get_authorization_url( $pkce['code_challenge'], $nonce );
            
            wp_send_json_success( array(
                'auth_url' => $auth_url,
                'pkce_data' => array(
                    'code_verifier' => $pkce['code_verifier'],
                    'code_challenge' => $pkce['code_challenge']
                )
            ) );
            
        } catch ( Exception $e ) {
            wp_send_json_error( array( 'message' => $e->getMessage() ) );
        }
    }
    
    /**
     * Exchange authorization code for tokens
     */
         public function ajax_aweber_exchange_token()
        {
            // Verify nonce and permissions
            if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) ), 'triggerninja_aweber_auth' ) ) {
                wp_send_json_error( array( 'message' => __( 'Security check failed.', 'triggerninja-for-ninja-forms' ) ) );
            }        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) ) );
        }
        
        $auth_code = sanitize_text_field( wp_unslash( $_POST['auth_code'] ?? '' ) );
        $pkce_data = isset($_POST['pkce_data']) && is_array($_POST['pkce_data']) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['pkce_data'] ) ) : array();
        $code_verifier = $pkce_data['code_verifier'] ?? '';
        
        if ( empty( $auth_code ) || empty( $code_verifier ) ) {
            wp_send_json_error( array( 'message' => __( 'Authorization code and PKCE verifier are required.', 'triggerninja-for-ninja-forms' ) ) );
        }
        
        try {
            // Exchange code for tokens
            $response = TriggerNinja_Aweber_API::exchange_code_for_tokens( $auth_code, $code_verifier );
            
            if ( $response->status >= 200 && $response->status < 300 && $response->details ) {
                $tokens = array();
                
                if ( is_object( $response->details ) ) {
                    $tokens['access_token'] = $response->details->access_token ?? '';
                    $tokens['refresh_token'] = $response->details->refresh_token ?? '';
                } elseif ( is_array( $response->details ) ) {
                    $tokens['access_token'] = $response->details['access_token'] ?? '';
                    $tokens['refresh_token'] = $response->details['refresh_token'] ?? '';
                }
                
                if ( !empty( $tokens['access_token'] ) ) {
                    // Save tokens using AWeber platform
                    $aweber_platform = $this->get_platform( 'triggerninja_aweber' );
                    if ( $aweber_platform && method_exists( $aweber_platform, 'save_tokens' ) ) {
                        $aweber_platform->save_tokens( $tokens );
                    }
                    
                    wp_send_json_success( array( 'message' => __( 'AWeber connected successfully!', 'triggerninja-for-ninja-forms' ) ) );
                } else {
                    wp_send_json_error( array( 'message' => __( 'No access token received from AWeber.', 'triggerninja-for-ninja-forms' ) ) );
                }
            } else {
                $error_message = $response->error ?: __( 'Failed to exchange authorization code for tokens.', 'triggerninja-for-ninja-forms' );
                wp_send_json_error( array( 'message' => $error_message ) );
            }
            
        } catch ( Exception $e ) {
            wp_send_json_error( array( 'message' => $e->getMessage() ) );
        }
    }
    
    /**
     * Disconnect AWeber account
     */
         public function ajax_aweber_disconnect()
        {
            // Verify nonce and permissions
            if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) ), 'triggerninja_aweber_auth' ) ) {
                wp_send_json_error( array( 'message' => __( 'Security check failed.', 'triggerninja-for-ninja-forms' ) ) );
            }        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) ) );
        }
        
        try {
            // Clear tokens using AWeber platform
            $aweber_platform = $this->get_platform( 'triggerninja_aweber' );
            if ( $aweber_platform && method_exists( $aweber_platform, 'clear_tokens' ) ) {
                $aweber_platform->clear_tokens();
            }
            
            wp_send_json_success( array( 'message' => __( 'AWeber disconnected successfully.', 'triggerninja-for-ninja-forms' ) ) );
            
        } catch ( Exception $e ) {
            wp_send_json_error( array( 'message' => $e->getMessage() ) );
        }
    }

    /**
     * Exchange authorization code for tokens
     */
    public function ajax_constantcontact_exchange_token()
    {
                if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['state'] ?? '' ) ), 'triggerninja_constantcontact_auth' ) ) {
            wp_die( 'Security check failed', 'Security check failed', array( 'response' => 403 ) );
        }

                $code = sanitize_text_field( wp_unslash( $_REQUEST['code'] ?? '' ) );
        if ( empty( $code ) ) {
            wp_die( 'No authorization code provided', 'Authorization error', array( 'response' => 400 ) );
        }

        $pkce = get_option( 'triggerninja_constantcontact_pkce' );
        $code_verifier = $pkce['code_verifier'] ?? '';

        if ( empty( $code_verifier ) ) {
            wp_die( 'No code verifier found', 'Authorization error', array( 'response' => 400 ) );
        }

        $redirect_uri = admin_url( 'admin-ajax.php?action=triggerninja_constantcontact_exchange_token' );

        $response = TriggerNinja_ConstantContact_API::exchange_code_for_tokens( $code, $code_verifier, $redirect_uri );

        if ( $response->status >= 200 && $response->status < 300 && ! empty( $response->details->access_token ) ) {
            $platform = $this->get_platform( 'triggerninja_constantcontact' );
            if ( $platform ) {
                $tokens = array(
                    'access_token'  => $response->details->access_token,
                    'refresh_token' => $response->details->refresh_token,
                    'expires_in'    => $response->details->expires_in,
                );
                $platform->save_tokens( $tokens );
            }

            // Redirect back to the settings page
            wp_safe_redirect( admin_url( 'admin.php?page=triggerninja' ) );
            exit;
        } else {
            wp_die( 'Failed to get access token', 'Authorization error', array( 'response' => 400 ) );
        }
    }

    /**
     * Disconnect Constant Contact account
     */
    public function ajax_constantcontact_disconnect()
    {
                if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) ), 'triggerninja_constantcontact_auth' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'triggerninja-for-ninja-forms' ) ) );
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) ) );
        }

        try {
            $platform = $this->get_platform( 'triggerninja_constantcontact' );
            if ( $platform && method_exists( $platform, 'clear_tokens' ) ) {
                $platform->clear_tokens();
            }

            wp_send_json_success( array( 'message' => __( 'Constant Contact disconnected successfully.', 'triggerninja-for-ninja-forms' ) ) );

        } catch ( Exception $e ) {
            wp_send_json_error( array( 'message' => $e->getMessage() ) );
        }
    }
    
    /*
    |--------------------------------------------------------------------------
    | Helper Methods
    |--------------------------------------------------------------------------
    */
    
    /**
     * Get the action class name for a platform
     * 
     * @param string $platform_id
     * @return string
     */
    private function get_platform_action_class( $platform_id )
    {
        // Handle prefixed platform IDs (triggerninja_aweber -> Aweber)
        $clean_id = str_replace( 'triggerninja_', '', $platform_id );
        // Convert to proper class name format (aweber -> Aweber, emailoctopus -> EmailOctopus)
        $class_suffix = str_replace( '_', '', ucwords( $clean_id, '_' ) );
        return 'TriggerNinja_' . $class_suffix . '_Action';
    }

    /*
    |--------------------------------------------------------------------------
    | Plugin Methods
    |--------------------------------------------------------------------------
    */

    public function version()
    {
        return $this->version;
    }

    public function url( $url = '' )
    {
        return trailingslashit( $this->url ) . $url;
    }

    public function dir( $path = '' )
    {
        return trailingslashit( $this->dir ) . $path;
    }

    public function template( $file, $args = array() )
    {
        $path = $this->dir( 'templates/' . $file );
        if( ! file_exists(  $path ) ) return '';
        extract( $args );

        ob_start();
        include $path;
        return ob_get_clean();
    }

}
