<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * MailerLite Platform Integration
 */
class TriggerNinja_MailerLite extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_mailerlite';
        $this->name = __( 'TN MailerLite', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_token = $this->get_setting( 'triggerninja_mailerlite_api_token' );
        if( $api_token ) {
            $this->api = new TriggerNinja_MailerLite_API( $api_token );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        $default_fields = array(
            'triggerninja_mailerlite_api_token' => array(
                'id'    => 'triggerninja_mailerlite_api_token',
                'type'  => 'textbox',
                'label' => __( 'MailerLite API Token', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'mailerlite',
                'value' => $this->get_setting( 'triggerninja_mailerlite_api_token' ),
                'placeholder' => __( 'Enter your MailerLite API Token', 'triggerninja-for-ninja-forms' ),
                /* translators: %s is a URL to the MailerLite API settings page. */
                'desc' => sprintf( __( 'Get your API Token from <a href="%s" target="_blank">MailerLite Integrations > API</a>', 'triggerninja-for-ninja-forms' ), 'https://app.mailerlite.com/integrations/api' )
            )
        );
        
        /**
         * Filter to modify MailerLite settings fields
         */
        return apply_filters( 'triggerninja_mailerlite_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $default_fields = array(
            'group' => array(
                'name' => 'group',
                'type' => 'select',
                'label' => __( 'MailerLite Group', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_groups(),
                'value' => ''
            )
        );
        
        /**
         * Filter to modify MailerLite action fields
         */
        $fields = apply_filters( 'triggerninja_mailerlite_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'group' ] ) ) {
            return $this->add_debug( $data, __( 'No MailerLite group specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API token
        $api_token = $this->get_setting( 'triggerninja_mailerlite_api_token' );
        if( empty( $api_token ) ) {
            return $this->add_debug( $data, __( 'MailerLite API token not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance with form_id for logging
        $api = new TriggerNinja_MailerLite_API( $api_token, $form_id );

        // Prepare subscriber data
        $email = trim( $action_settings[ 'email_address' ] );
        
        // Collect subscriber fields - filter out empty values
        $fields = array();
        
        if( isset( $action_settings[ 'first_name' ] ) && ! empty( trim( $action_settings[ 'first_name' ] ) ) ) {
            $fields['first_name'] = trim( $action_settings[ 'first_name' ] );
        }
        
        if( isset( $action_settings[ 'last_name' ] ) && ! empty( trim( $action_settings[ 'last_name' ] ) ) ) {
            $fields['last_name'] = trim( $action_settings[ 'last_name' ] );
        }

        /**
         * Filter to process additional fields
         */
        $fields = apply_filters( 'triggerninja_process_fields', $fields, $action_settings, $form_id, $data );
        $fields = apply_filters( "triggerninja_{$this->id}_process_fields", $fields, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );

        // Subscribe to group
        $response = $api->subscribe_to_group( 
            $email, 
            $fields,
            $action_settings[ 'group' ]
        );

        // Check for successful subscription
        if( $response->status >= 200 && $response->status < 300 ) {
            $response_details = $response->details;
            $subscriber_id = is_object( $response_details ) && isset( $response_details->id ) ? $response_details->id : 'subscribed';
            
            $response_data = array(
                'subscriber_id' => $subscriber_id,
                'email' => $email,
                'fields' => $fields,
                'group_id' => $action_settings[ 'group' ],
                'status' => 'subscribed'
            );

            /**
             * Filter response data before saving
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            $data = $this->add_debug( $data, __( 'Successfully subscribed to MailerLite group', 'triggerninja-for-ninja-forms' ), 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
            
        } else {
            $error_message = $response->error ? $response->error : __( 'Failed to subscribe to MailerLite group', 'triggerninja-for-ninja-forms' );
            /* translators: %s is an error message. */
            $data = $this->add_debug( $data, sprintf( __( 'MailerLite error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available groups from MailerLite
     * 
     * @return array
     */
    public function get_groups()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API token first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_groups();
        
        // The TriggerNinja_Response class already extracts the 'data' array from the JSON response
        // So $response->details contains the array directly, not an object with a 'data' property
        if( ! $response->details || ! is_array( $response->details ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading groups', 'triggerninja-for-ninja-forms' ) ) );
        }

        $groups = array();
        foreach( $response->details as $group ) {
            if( is_object( $group ) && isset( $group->id ) && isset( $group->name ) ) {
                $groups[] = array(
                    'value' => $group->id,
                    'label' => $group->name
                );
            }
        }
        
        array_unshift( $groups, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify MailerLite groups
         */
        return apply_filters( 'triggerninja_mailerlite_groups', $groups, $this );
    }
    
    /**
     * Check if platform is connected (has valid API token)
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_mailerlite_api_token' ) ) && $this->api !== null;
    }
    
    /**
     * Test API connection
     * 
     * @return bool
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return false;
        }
        
        $response = $this->api->test_connection();
        return $response->status >= 200 && $response->status < 300;
    }
    
    /**
     * Get available lists from MailerLite (alias for get_groups)
     * Required by TriggerNinja_Platform_Interface
     * 
     * @return array
     */
    public function get_lists()
    {
        return $this->get_groups();
    }
    
    /**
     * Get platform-specific metabox data for submissions
     * Required by TriggerNinja_Platform_Interface
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'subscriber_id' ] ) ){
            $label = __( 'Subscriber ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'subscriber_id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'group_id' ] ) ){
            $label = __( 'Group ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'group_id' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }
        
        if( isset( $submission_data[ 'fields' ] ) && ! empty( $submission_data[ 'fields' ] ) ){
            $label = __( 'Subscriber Fields', 'triggerninja-for-ninja-forms' );
            $field_info = array();
            foreach( $submission_data[ 'fields' ] as $key => $value ) {
                if( ! empty( $value ) ) {
                    $field_info[] = "{$key}: {$value}";
                }
            }
            $data[ $label ] = implode( ', ', $field_info );
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}