<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * MailerLite API Integration
 */
class TriggerNinja_MailerLite_API
{
    const URL = 'https://connect.mailerlite.com/api/';
    
    /**
     * API Token
     * 
     * @var string
     */
    private $api_token;
    
    /**
     * Form ID for logging
     * 
     * @var int
     */
    private $form_id;
    
    /**
     * Constructor
     * 
     * @param string $api_token MailerLite API token
     * @param int $form_id Form ID for logging
     */
    public function __construct( $api_token = '', $form_id = 0 )
    {
        $this->api_token = $api_token;
        $this->form_id = $form_id;
    }
    
    /**
     * Make API request to MailerLite using centralized HTTP client
     * 
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array $data Request data
     * @return TriggerNinja_Response
     */
    private function request( $endpoint, $method = 'GET', $data = array() )
    {
        $url = self::URL . $endpoint;
        
        $headers = array(
            'User-Agent' => 'TriggerNinja',
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'Authorization' => 'Bearer ' . $this->api_token
        );
        
        $options = array(
            'timeout' => 30
        );
        
        $response = TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $this->form_id, 'mailerlite' );
        
        // Handle MailerLite specific error format
        if ( $response->status >= 400 && $response->details ) {
            $error_message = 'Unknown error occurred';
            
            // Try to extract MailerLite error message
            if ( is_object( $response->details ) && isset( $response->details->message ) ) {
                $error_message = $response->details->message;
            } elseif ( is_array( $response->details ) && isset( $response->details['message'] ) ) {
                $error_message = $response->details['message'];
            } elseif ( $response->error ) {
                $error_message = $response->error;
            }
            
            return new TriggerNinja_Response( $response->status, array(), $error_message );
        }
        
        return $response;
    }
    
    /**
     * Get all groups
     * 
     * @return TriggerNinja_Response
     */
    public function get_groups()
    {
        return $this->request( 'groups' );
    }
    
    /**
     * Subscribe subscriber to a group
     * 
     * @param string $email Subscriber email address
     * @param array $fields Subscriber fields (first_name, last_name, etc.)
     * @param string $group_id Group ID (optional)
     * @param string $status Subscriber status (optional)
     * @param string $ip_address IP address (optional)
     * @return TriggerNinja_Response
     */
    public function subscribe_to_group( $email, $fields = array(), $group_id = '', $status = '', $ip_address = '' )
    {
        if ( empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'Email is required' );
        }
        
        $subscriber_data = array(
            'email' => $email
        );
        
        // Add fields if provided
        if ( ! empty( $fields ) ) {
            $subscriber_data['fields'] = $fields;
        }
        
        // Add group if specified
        if ( ! empty( $group_id ) ) {
            $subscriber_data['groups'] = array( $group_id );
        }
        
        // Add optional status
        if ( ! empty( $status ) ) {
            $subscriber_data['status'] = $status;
        }
        
        // Add optional IP address
        if ( ! empty( $ip_address ) ) {
            $subscriber_data['ip_address'] = $ip_address;
        }
        
        return $this->request( 'subscribers', 'POST', $subscriber_data );
    }
    
    /**
     * Get subscriber by email
     * 
     * @param string $email Subscriber email address
     * @return TriggerNinja_Response
     */
    public function get_subscriber( $email )
    {
        if ( empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'Email is required' );
        }
        
        return $this->request( "subscribers/{$email}" );
    }
    
    /**
     * Get available fields
     * 
     * @return TriggerNinja_Response
     */
    public function get_fields()
    {
        return $this->request( 'fields' );
    }
    
    /**
     * Test API connection by getting groups
     * 
     * @return TriggerNinja_Response
     */
    public function test_connection()
    {
        return $this->get_groups();
    }
    
    /**
     * Get account info for connection testing
     * 
     * @return TriggerNinja_Response
     */
    public function get_account_info()
    {
        return $this->get_groups();
    }
}