<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Mailchimp Platform Integration
 */
class TriggerNinja_Mailchimp extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_mailchimp';
        $this->name = __( 'TN Mailchimp', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_key = $this->get_setting( 'triggerninja_mailchimp_api_key' );
        if( $api_key ) {
            $this->api = new TriggerNinja_Mailchimp_API( $api_key );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        $default_fields = array(
            'triggerninja_mailchimp_api_key' => array(
                'id'    => 'triggerninja_mailchimp_api_key',
                'type'  => 'textbox',
                'label' => __( 'Mailchimp API Key', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'mailchimp',
                'value' => $this->get_setting( 'triggerninja_mailchimp_api_key' ),
                'placeholder' => __( 'Enter your Mailchimp API Key', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf(
                    /* translators: %s is a URL to the Mailchimp API Keys help page. */
                    __( 'Go to Account > Extras > API Keys to get your API key from <a href="%s" target="_blank">Mailchimp</a>', 'triggerninja-for-ninja-forms' ), 
                    'https://mailchimp.com/help/about-api-keys/' 
                )
            )
        );
        
        /**
         * Filter to modify Mailchimp settings fields
         */
        return apply_filters( 'triggerninja_mailchimp_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $default_fields = array(
            'list' => array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'Mailchimp Audience', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_lists(),
                'value' => ''
            ),
            'double_opt_in' => array(
                'name' => 'double_opt_in',
                'type' => 'textbox',
                'label' => __( 'Double Opt-in', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'value' => 'false',
                'placeholder' => __( 'Enter true or false', 'triggerninja-for-ninja-forms' ),
                'desc' => __( 'Enter "true" to send confirmation email before subscribing, or "false" to subscribe immediately.', 'triggerninja-for-ninja-forms' )
            )
        );
        
        /**
         * Filter to modify Mailchimp action fields
         */
        $fields = apply_filters( 'triggerninja_mailchimp_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'list' ] ) ) {
            return $this->add_debug( $data, __( 'No Mailchimp audience specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API key (with extensibility hook for advanced users)
        $api_key = $this->get_setting( 'triggerninja_mailchimp_api_key' );
        
        /**
         * Filter to modify Mailchimp API key for advanced setups
         */
        $api_key = apply_filters( 'triggerninja_mailchimp_api_key', $api_key, $action_settings, $this );
        
        if( empty( $api_key ) ) {
            return $this->add_debug( $data, __( 'Mailchimp API key not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance
        $api = new TriggerNinja_Mailchimp_API( $api_key, $form_id );

        // Prepare profile data
        $email = trim( $action_settings[ 'email_address' ] );
        $first_name = isset( $action_settings[ 'first_name' ] ) ? trim( $action_settings[ 'first_name' ] ) : '';
        $last_name = isset( $action_settings[ 'last_name' ] ) ? trim( $action_settings[ 'last_name' ] ) : '';

        // Prepare merge fields for Mailchimp
        $merge_fields = array();
        
        if( $first_name ) {
            $merge_fields['FNAME'] = $first_name;
        }
        
        if( $last_name ) {
            $merge_fields['LNAME'] = $last_name;
        }

        /**
         * Filter to process additional fields
         */
        $merge_fields = apply_filters( 'triggerninja_process_fields', $merge_fields, $action_settings, $form_id, $data );
        $merge_fields = apply_filters( "triggerninja_{$this->id}_process_fields", $merge_fields, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );

        // Get double opt-in setting from action settings
        $double_opt_in_value = isset( $action_settings[ 'double_opt_in' ] ) ? trim( strtolower( $action_settings[ 'double_opt_in' ] ) ) : 'false';
        
        // Validate double opt-in value
        if( ! in_array( strtolower( $action_settings[ 'double_opt_in' ] ), array( 'true', 'false' ) ) ) {
            /* translators: %s is the invalid value. */
            return $this->add_debug( $data, sprintf( __( 'Invalid Double Opt-in value "%s". Please enter "true" or "false".', 'triggerninja-for-ninja-forms' ), $action_settings[ 'double_opt_in' ] ), 'error' );
        }
        
        $double_opt_in = ( $double_opt_in_value === 'true' || $double_opt_in_value === '1' );
        
        // Use the Mailchimp API to subscribe
        $subscription_response = $api->subscribe_to_list( 
            $action_settings[ 'list' ], 
            $email, 
            $merge_fields,
            $double_opt_in
        );

        // Check for successful subscription
        if( $subscription_response->status >= 200 && $subscription_response->status < 300 ) {
            $response_details = $subscription_response->details;
            $contact_id = is_object( $response_details ) && isset( $response_details->id ) ? $response_details->id : 'subscribed';
            
            $response_data = array(
                'contact_id' => $contact_id,
                'email' => $email,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'list_id' => $action_settings[ 'list' ],
                'status' => $double_opt_in ? 'pending_confirmation' : 'subscribed',
                'double_opt_in' => $double_opt_in,
                'merge_fields' => $merge_fields
            );

            /**
             * Filter response data before saving
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            $success_message = $double_opt_in ? 
                __( 'Successfully sent confirmation email via Mailchimp', 'triggerninja-for-ninja-forms' ) :
                __( 'Successfully subscribed to Mailchimp audience', 'triggerninja-for-ninja-forms' );
            $data = $this->add_debug( $data, $success_message, 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            
        } else {
            $error_message = $subscription_response->error ? $subscription_response->error : __( 'Failed to subscribe to Mailchimp audience', 'triggerninja-for-ninja-forms' );
            /* translators: %s is the error message returned by the API. */
            $data = $this->add_debug( $data, sprintf( __( 'Mailchimp error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available lists from Mailchimp
     * 
     * @return array
     */
    public function get_lists()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API key first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_lists();
        
        if( ! $response->details || ! isset( $response->details->lists ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading audiences', 'triggerninja-for-ninja-forms' ) ) );
        }

        $lists = array_map( array( $this, 'format_list_options' ), $response->details->lists );
        array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify Mailchimp lists
         */
        return apply_filters( 'triggerninja_mailchimp_lists', $lists, $this );
    }
    
    /**
     * Format list options for dropdown
     * 
     * @param object $list
     * @return array
     */
    private function format_list_options( $list )
    {
        return array(
            'value' => $list->id,
            'label' => $list->name
        );
    }

    /**
     * Validate Mailchimp connection
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_mailchimp_api_key' ) ) && $this->api !== null;
    }

    /**
     * Test connection to Mailchimp
     * 
     * @return array
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return array(
                'success' => false,
                'message' => __( 'API key not configured', 'triggerninja-for-ninja-forms' )
            );
        }

        try {
            $response = $this->api->get_account_info();
            
            if( $response->status >= 200 && $response->status < 300 && $response->details ) {
                $lists_count = is_object( $response->details ) && isset( $response->details->lists ) ? count( $response->details->lists ) : 0;
                return array(
                    'success' => true,
                    'message' => __( 'Successfully connected to Mailchimp', 'triggerninja-for-ninja-forms' ),
                    'data' => array(
                        'lists_count' => $lists_count
                    )
                );
            }

            return array(
                'success' => false,
                'message' => __( 'Invalid response from Mailchimp API', 'triggerninja-for-ninja-forms' )
            );

        } catch( Exception $e ) {
            return array(
                'success' => false,
                /* translators: %s is the error message returned by the exception. */
                'message' => sprintf( __( 'Connection failed: %s', 'triggerninja-for-ninja-forms' ), $e->getMessage() )
            );
        }
    }

    /**
     * Get platform-specific metabox data for submissions
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'contact_id' ] ) ){
            $label = __( 'Contact ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'contact_id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'first_name' ] ) && ! empty( $submission_data[ 'first_name' ] ) ){
            $label = __( 'First Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'first_name' ];
        }

        if( isset( $submission_data[ 'last_name' ] ) && ! empty( $submission_data[ 'last_name' ] ) ){
            $label = __( 'Last Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'last_name' ];
        }

        if( isset( $submission_data[ 'list_id' ] ) ){
            $label = __( 'Audience ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'list_id' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }
        
        if( isset( $submission_data[ 'double_opt_in' ] ) ){
            $label = __( 'Double Opt-in', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'double_opt_in' ] ? __( 'Enabled', 'triggerninja-for-ninja-forms' ) : __( 'Disabled', 'triggerninja-for-ninja-forms' );
        }

        if( isset( $submission_data[ 'merge_fields' ] ) && ! empty( $submission_data[ 'merge_fields' ] ) ){
            $label = __( 'Merge Fields', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = implode( ', ', array_map( function( $key, $value ) {
                return "{$key}: {$value}";
            }, array_keys( $submission_data[ 'merge_fields' ] ), $submission_data[ 'merge_fields' ] ) );
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}