<?php if ( ! defined( 'ABSPATH' ) ) exit;

class TriggerNinja_Mailchimp_API
{
    private $api_key;
    private $base_url;
    private $form_id;

    public function __construct( $api_key = '', $form_id = 0 )
    {
        $this->api_key = $api_key;
        $this->form_id = $form_id;
        
        // Extract datacenter from API key (key format: xxxxx-us1)
        if( $api_key && strpos( $api_key, '-' ) !== false ) {
            $parts = explode( '-', $api_key );
            $datacenter = end( $parts );
            $this->base_url = "https://{$datacenter}.api.mailchimp.com/3.0/";
        }
    }

    public function request( $endpoint, $method = 'GET', $data = array() )
    {
        if( ! $this->base_url ) {
            return new TriggerNinja_Response( 400, array(), 'Invalid API key format' );
        }
        
        $url = trailingslashit( $this->base_url ) . $endpoint;
        
        $headers = array(
            'Authorization' => 'api_key ' . $this->api_key
        );
        
        return TriggerNinja_Http_Client::request( $method, $url, $data, $headers, array(), $this->form_id, 'mailchimp' );
    }

    /**
     * Get lists (audiences) from Mailchimp
     * 
     * @param int $count Number of lists to retrieve
     * @return TriggerNinja_Response
     */
    public function get_lists( $count = 1000 )
    {
        return $this->request( "lists?count={$count}" );
    }

    /**
     * Subscribe contact to a list
     * 
     * @param string $list_id List ID
     * @param string $email Email address
     * @param array $merge_fields Merge fields (FNAME, LNAME, etc.)
     * @param bool $double_opt_in Whether to send double opt-in email
     * @return TriggerNinja_Response
     */
    public function subscribe_to_list( $list_id, $email, $merge_fields = array(), $double_opt_in = false )
    {
        $subscriber_data = array(
            'email_address' => trim( $email ),
            'status' => $double_opt_in ? 'pending' : 'subscribed'
        );
        
        if( !empty( $merge_fields ) ) {
            $subscriber_data['merge_fields'] = $merge_fields;
        }

        return $this->request( "lists/{$list_id}/members", 'POST', $subscriber_data );
    }

    /**
     * Unsubscribe contact from a list
     * 
     * @param string $list_id List ID
     * @param string $email Email address
     * @return TriggerNinja_Response
     */
    public function unsubscribe_from_list( $list_id, $email )
    {
        // First, search for the member
        $search_response = $this->request( "search-members?query=" . urlencode( $email ) );
        
        if( $search_response->status >= 200 && $search_response->status < 300 && $search_response->details ) {
            $search_data = $search_response->details;
            
            if( isset( $search_data->exact_matches ) && 
                isset( $search_data->exact_matches->members ) && 
                count( $search_data->exact_matches->members ) > 0 ) {
                
                $member_id = $search_data->exact_matches->members[0]->id;
                return $this->request( "lists/{$list_id}/members/{$member_id}", 'DELETE' );
            }
        }
        
        return new TriggerNinja_Response( 404, array(), 'Member not found' );
    }

    /**
     * Get account info for connection testing
     * 
     * @return TriggerNinja_Response
     */
    public function get_account_info()
    {
        return $this->request( 'lists?count=1' );
    }
}