<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Lemlist Platform Integration
 */
class TriggerNinja_Lemlist extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_lemlist';
        $this->name = __( 'TN Lemlist', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_key = $this->get_setting( 'triggerninja_lemlist_api_key' );
        if( $api_key ) {
            $this->api = new TriggerNinja_Lemlist_API( $api_key );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        $default_fields = array(
            'triggerninja_lemlist_api_key' => array(
                'id'    => 'triggerninja_lemlist_api_key',
                'type'  => 'textbox',
                'label' => __( 'Lemlist API Key', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'lemlist',
                'value' => $this->get_setting( 'triggerninja_lemlist_api_key' ),
                'placeholder' => __( 'Enter your Lemlist API Key', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf(
                    /* translators: %s is a URL to the Lemlist integrations page. */
                    __( 'Get your API Key from <a href="%s" target="_blank">Lemlist Settings > Integrations</a>', 'triggerninja-for-ninja-forms' ), 
                    'https://app.lemlist.com/' 
                )
            )
        );
        
        /**
         * Filter to modify Lemlist settings fields
         */
        return apply_filters( 'triggerninja_lemlist_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $default_fields = array(
            'campaign' => array(
                'name' => 'campaign',
                'type' => 'select',
                'label' => __( 'Lemlist Campaign', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_campaigns(),
                'value' => ''
            )
        );
        
        /**
         * Filter to modify Lemlist action fields
         */
        $fields = apply_filters( 'triggerninja_lemlist_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Get common field structure for forms (basic fields only)
     * 
     * @return array
     */
    protected function get_common_fields()
    {
        $base_fields = array(
            'email_address' => array(
                'name' => 'email_address',
                'type' => 'textbox',
                'label' => __( 'Email Address', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            ),
            'first_name' => array(
                'name' => 'first_name',
                'type' => 'textbox',
                'label' => __( 'First Name', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            ),
            'last_name' => array(
                'name' => 'last_name',
                'type' => 'textbox',
                'label' => __( 'Last Name', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            )
        );

        /**
         * Filter to add additional fields
         * 
         * @param array $base_fields The base fields
         * @param string $platform_id The platform ID
         * @return array Extended fields array
         */
        $extended_fields = apply_filters( 'triggerninja_platform_fields', $base_fields, $this->id );
        $extended_fields = apply_filters( "triggerninja_{$this->id}_fields", $extended_fields, $this->id );

        return $extended_fields;
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'campaign' ] ) ) {
            return $this->add_debug( $data, __( 'No Lemlist campaign specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API key
        $api_key = $this->get_setting( 'triggerninja_lemlist_api_key' );
        if( empty( $api_key ) ) {
            return $this->add_debug( $data, __( 'Lemlist API key not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance with form_id for logging
        $api = new TriggerNinja_Lemlist_API( $api_key, $form_id );

        // Prepare lead data
        $email = trim( $action_settings[ 'email_address' ] );
        
        // Collect basic lead data - filter out empty values
        $lead_data = array_filter( array(
            'firstName' => isset( $action_settings[ 'first_name' ] ) ? trim( $action_settings[ 'first_name' ] ) : '',
            'lastName' => isset( $action_settings[ 'last_name' ] ) ? trim( $action_settings[ 'last_name' ] ) : ''
        ) );

        /**
         * Filter to process additional fields
         */
        $lead_data = apply_filters( 'triggerninja_process_fields', $lead_data, $action_settings, $form_id, $data );
        $lead_data = apply_filters( "triggerninja_{$this->id}_process_fields", $lead_data, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', $lead_data, $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", $lead_data, $action_settings, $form_id, $data );

        // Add lead to campaign
        $response = $api->add_lead_to_campaign( 
            $action_settings[ 'campaign' ], 
            $email, 
            $lead_data 
        );

        // Check for successful submission
        if( $response->status >= 200 && $response->status < 300 ) {
            $response_details = $response->details;
            $lead_id = is_object( $response_details ) && isset( $response_details->_id ) ? $response_details->_id : 'added';
            
            $response_data = array(
                'lead_id' => $lead_id,
                'email' => $email,
                'campaign_id' => $action_settings[ 'campaign' ],
                'status' => 'added',
                'lead_data' => $lead_data
            );

            /**
             * Filter response data before saving
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, $lead_data, $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, $lead_data, $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            $data = $this->add_debug( $data, __( 'Successfully added lead to Lemlist campaign', 'triggerninja-for-ninja-forms' ), 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, $lead_data, $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, $lead_data, $action_settings, $form_id, $data );
            
        } else {
            $error_message = $response->error ? $response->error : __( 'Failed to add lead to Lemlist campaign', 'triggerninja-for-ninja-forms' );
            /* translators: %s is an error message. */
            $data = $this->add_debug( $data, sprintf( __( 'Lemlist error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, $lead_data, $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, $lead_data, $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available campaigns from Lemlist
     * 
     * @return array
     */
    public function get_campaigns()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API key first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_campaigns();
        
        if( ! $response->details || ! isset( $response->details->campaigns ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading campaigns', 'triggerninja-for-ninja-forms' ) ) );
        }

        $campaigns = array();
        foreach( $response->details->campaigns as $campaign ) {
            $campaigns[] = array(
                'value' => $campaign->_id,
                'label' => $campaign->name
            );
        }
        
        array_unshift( $campaigns, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify Lemlist campaigns
         */
        return apply_filters( 'triggerninja_lemlist_campaigns', $campaigns, $this );
    }
    
    /**
     * Check if platform is connected (has valid API key)
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_lemlist_api_key' ) ) && $this->api !== null;
    }
    
    /**
     * Test API connection
     * 
     * @return bool
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return false;
        }
        
        $response = $this->api->test_connection();
        return $response->status >= 200 && $response->status < 300;
    }
    
    /**
     * Get available lists from Lemlist (alias for get_campaigns)
     * Required by TriggerNinja_Platform_Interface
     * 
     * @return array
     */
    public function get_lists()
    {
        return $this->get_campaigns();
    }
    
    /**
     * Get platform-specific metabox data for submissions
     * Required by TriggerNinja_Platform_Interface
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'lead_id' ] ) ){
            $label = __( 'Lead ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'lead_id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'campaign_id' ] ) ){
            $label = __( 'Campaign ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'campaign_id' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }
        
        if( isset( $submission_data[ 'lead_data' ] ) && ! empty( $submission_data[ 'lead_data' ] ) ){
            $label = __( 'Lead Data', 'triggerninja-for-ninja-forms' );
            $lead_info = array();
            foreach( $submission_data[ 'lead_data' ] as $key => $value ) {
                if( ! empty( $value ) ) {
                    $lead_info[] = "{$key}: {$value}";
                }
            }
            $data[ $label ] = implode( ', ', $lead_info );
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}
