<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Lemlist API Integration
 */
class TriggerNinja_Lemlist_API
{
    const URL = 'https://api.lemlist.com/api/';
    
    /**
     * API Key
     * 
     * @var string
     */
    private $api_key;
    
    /**
     * Form ID for logging
     * 
     * @var int
     */
    private $form_id;
    
    /**
     * Constructor
     * 
     * @param string $api_key Lemlist API key
     * @param int $form_id Form ID for logging
     */
    public function __construct( $api_key = '', $form_id = 0 )
    {
        $this->api_key = $api_key;
        $this->form_id = $form_id;
    }
    
    /**
     * Make API request to Lemlist using centralized HTTP client
     * 
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array $data Request data
     * @return TriggerNinja_Response
     */
    private function request( $endpoint, $method = 'GET', $data = array() )
    {
        $url = self::URL . $endpoint;
        
        $headers = array(
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'Authorization' => 'Basic ' . base64_encode( ':' . $this->api_key )
        );
        
        $options = array(
            'timeout' => 30
        );
        
        $response = TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $this->form_id, 'lemlist' );
        
        // Handle Lemlist specific error format
        if ( $response->status >= 400 && $response->details ) {
            $error_message = 'Unknown error occurred';
            
            // Try to extract Lemlist error message
            if ( is_object( $response->details ) && isset( $response->details->message ) ) {
                $error_message = $response->details->message;
            } elseif ( is_array( $response->details ) && isset( $response->details['message'] ) ) {
                $error_message = $response->details['message'];
            } elseif ( $response->error ) {
                $error_message = $response->error;
            }
            
            return new TriggerNinja_Response( $response->status, array(), $error_message );
        }
        
        return $response;
    }
    
    /**
     * Get all campaigns
     * 
     * @param int $limit Number of campaigns to retrieve
     * @return TriggerNinja_Response
     */
    public function get_campaigns( $limit = 100 )
    {
        return $this->request( "campaigns?version=v2&limit={$limit}" );
    }
    
    /**
     * Add lead to campaign
     * 
     * @param string $campaign_id Campaign ID
     * @param string $email Lead email address
     * @param array $lead_data Lead data
     * @return TriggerNinja_Response
     */
    public function add_lead_to_campaign( $campaign_id, $email, $lead_data = array() )
    {
        if ( empty( $campaign_id ) || empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'Campaign ID and email are required' );
        }
        
        $endpoint = "campaigns/{$campaign_id}/leads/{$email}";
        
        // Check if lead already exists
        if ( $this->find_lead_by_email( $email, $campaign_id ) ) {
            // Update existing lead
            return $this->request( $endpoint, 'PATCH', $lead_data );
        } else {
            // Create new lead
            return $this->request( $endpoint, 'POST', $lead_data );
        }
    }
    
    /**
     * Find lead by email in campaign
     * 
     * @param string $email Lead email address
     * @param string $campaign_id Campaign ID
     * @return bool True if lead exists, false otherwise
     */
    public function find_lead_by_email( $email, $campaign_id )
    {
        if ( empty( $email ) || empty( $campaign_id ) ) {
            return false;
        }
        
        $endpoint = "leads/{$email}?version=v2&campaignId={$campaign_id}";
        $response = $this->request( $endpoint );
        
        if ( $response->status < 200 || $response->status >= 300 ) {
            return false;
        }
        
        // Check if response contains lead data
        if ( $response->details && is_array( $response->details ) && count( $response->details ) > 0 ) {
            $lead = $response->details[0];
            if ( is_object( $lead ) && isset( $lead->variables->email ) && $lead->variables->email === $email ) {
                return true;
            } elseif ( is_array( $lead ) && isset( $lead['variables']['email'] ) && $lead['variables']['email'] === $email ) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get lead details by email
     * 
     * @param string $email Lead email address
     * @param string $campaign_id Campaign ID (optional)
     * @return TriggerNinja_Response
     */
    public function get_lead( $email, $campaign_id = '' )
    {
        if ( empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'Email is required' );
        }
        
        $endpoint = "leads/{$email}?version=v2";
        if ( ! empty( $campaign_id ) ) {
            $endpoint .= "&campaignId={$campaign_id}";
        }
        
        return $this->request( $endpoint );
    }
    
    /**
     * Test API connection by getting campaigns
     * 
     * @return TriggerNinja_Response
     */
    public function test_connection()
    {
        return $this->get_campaigns( 1 );
    }
    
    /**
     * Get account info for connection testing
     * 
     * @return TriggerNinja_Response
     */
    public function get_account_info()
    {
        return $this->get_campaigns( 1 );
    }
}