<?php if ( ! defined( 'ABSPATH' ) || ! class_exists( 'TriggerNinja_Platform_Abstract' )) exit;

/**
 * Klaviyo Platform Integration
 * 
 * Provides integration with Klaviyo email marketing platform
 */
class TriggerNinja_Klaviyo extends TriggerNinja_Platform_Abstract
{
    /**
     * Platform ID
     */
    protected $id = 'triggerninja_klaviyo';

    /**
     * Platform display name
     */
    protected $name = 'TN Klaviyo';

    /**
     * Klaviyo API instance
     */
    protected $api;

    /**
     * Initialize the platform
     */
    protected function init_api()
    {
        $api_key = $this->get_setting( 'triggerninja_klaviyo_api_key' );
        
        if( ! empty( $api_key ) ) {
            $this->api = new TriggerNinja_Klaviyo_API( $api_key );
        }
    }

    /**
     * Get platform settings fields for Ninja Forms
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        return array(
            'triggerninja_klaviyo_api_key' => array(
                'id'    => 'triggerninja_klaviyo_api_key',
                'type'  => 'textbox',
                'label' => __( 'Klaviyo Private API Key', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'klaviyo',
                'value' => $this->get_setting( 'triggerninja_klaviyo_api_key' ),
                'placeholder' => __( 'Enter your Klaviyo Private API Key', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf(
                    /* translators: %s is a URL to the Klaviyo API settings page. */
                    __( 'Get your Private API Key from <a href="%s" target="_blank">Klaviyo Account Settings</a>', 'triggerninja-for-ninja-forms' ), 
                    'https://www.klaviyo.com/account#api-keys-tab' 
                )
            )
        );
    }

    /**
     * Get action fields for Ninja Forms actions
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $fields = array(
            'list' => array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'Klaviyo List', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_lists(),
                'value' => ''
            )
        );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'list' ] ) ) {
            return $this->add_debug( $data, __( 'No Klaviyo list specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API key
        $api_key = $this->get_setting( 'triggerninja_klaviyo_api_key' );
        if( empty( $api_key ) ) {
            return $this->add_debug( $data, __( 'Klaviyo API key not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance with form_id for logging
        $api = new TriggerNinja_Klaviyo_API( $api_key, $form_id );

        // Prepare profile data
        $profile_data = array(
            'email' => trim( $action_settings[ 'email_address' ] )
        );

        // Add optional fields
        if( ! empty( $action_settings[ 'first_name' ] ) ) {
            $profile_data['first_name'] = trim( $action_settings[ 'first_name' ] );
        }

        if( ! empty( $action_settings[ 'last_name' ] ) ) {
            $profile_data['last_name'] = trim( $action_settings[ 'last_name' ] );
        }

        /**
         * Filter to process additional fields
         * 
         * @param array $profile_data Current profile data
         * @param array $action_settings Form action settings
         * @param int $form_id Form ID
         * @param array $data Form submission data
         * @return array Extended profile data
         */
        $profile_data = apply_filters( 'triggerninja_process_fields', $profile_data, $action_settings, $form_id, $data );
        $profile_data = apply_filters( "triggerninja_{$this->id}_process_fields", $profile_data, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', $profile_data, $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", $profile_data, $action_settings, $form_id, $data );

        // Create structured data for Klaviyo API
        $subscriber_data = array(
            'data' => array(
                'type' => 'profile',
                'attributes' => $profile_data
            )
        );

        /**
         * Filter final API payload before sending
         */
        $subscriber_data = apply_filters( 'triggerninja_api_payload', $subscriber_data, 'profile-import', $action_settings );
        $subscriber_data = apply_filters( "triggerninja_{$this->id}_api_payload", $subscriber_data, 'profile-import', $action_settings );

        // Create or update profile first
        $profile_response = $api->create_or_update_profile( $subscriber_data );

        // Check for successful profile creation - TriggerNinja_Response strips the 'data' wrapper
        if( $profile_response->status >= 200 && $profile_response->status < 300 && $profile_response->details && isset( $profile_response->details->id ) ) {
            // Profile created/updated successfully, now subscribe to list
            $subscription_response = $api->subscribe_to_list( 
                $action_settings[ 'list' ], 
                $profile_data['email'], 
                'TriggerNinja' 
            );

            if( $subscription_response->status >= 200 && $subscription_response->status < 300 && $subscription_response->details ) {
                $response_data = array(
                    'profile_id' => $profile_response->details->id,
                    'email' => $profile_data['email'],
                    'list_id' => $action_settings[ 'list' ],
                    'status' => 'subscribed'
                );

                /**
                 * Filter response data before saving
                 */
                $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, $subscriber_data, $action_settings );
                $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, $subscriber_data, $action_settings );
                
                $data = $this->save_submission_data( $data, $response_data );
                $data = $this->add_debug( $data, __( 'Successfully subscribed to Klaviyo list', 'triggerninja-for-ninja-forms' ), 'success' );
                
                /**
                 * Hook after successful API request
                 */
                do_action( 'triggerninja_after_api_success', $response_data, $subscriber_data, $action_settings, $form_id, $data );
                do_action( "triggerninja_{$this->id}_after_api_success", $response_data, $subscriber_data, $action_settings, $form_id, $data );
                
            } else {
                $error_message = $subscription_response->error ? $subscription_response->error : __( 'Failed to subscribe to list', 'triggerninja-for-ninja-forms' );
                /* translators: %s is the error message returned by the API. */
                $data = $this->add_debug( $data, sprintf( __( 'Klaviyo subscription error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
                
                /**
                 * Hook after API error
                 */
                do_action( 'triggerninja_after_api_error', $error_message, $subscriber_data, $action_settings, $form_id, $data );
                do_action( "triggerninja_{$this->id}_after_api_error", $error_message, $subscriber_data, $action_settings, $form_id, $data );
            }
        } else {
            $error_message = $profile_response->error ? $profile_response->error : __( 'Failed to create profile', 'triggerninja-for-ninja-forms' );
            /* translators: %s is the error message returned by the API. */
            $data = $this->add_debug( $data, sprintf( __( 'Klaviyo profile error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, $subscriber_data, $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, $subscriber_data, $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available lists from Klaviyo
     * 
     * @return array
     */
    public function get_lists()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API key first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $lists = $this->api->get_all_lists();
        
        if( empty( $lists ) ) {
            return array( array( 'value' => '', 'label' => __( 'No lists found or API error', 'triggerninja-for-ninja-forms' ) ) );
        }

        $formatted_lists = array();
        foreach( $lists as $list_id => $list_name ) {
            $formatted_lists[] = array(
                'value' => $list_id,
                'label' => $list_name
            );
        }

        array_unshift( $formatted_lists, array( 'value' => '', 'label' => '-' ) );
        return $formatted_lists;
    }

    /**
     * Validate Klaviyo connection
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_klaviyo_api_key' ) ) && $this->api !== null;
    }

    /**
     * Test connection to Klaviyo
     * 
     * @return array
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return array(
                'success' => false,
                'message' => __( 'API key not configured', 'triggerninja-for-ninja-forms' )
            );
        }

        try {
            $response = $this->api->get_account_info();
            
            if( $response->details && isset( $response->details->data ) ) {
                $account_data = $response->details->data[0] ?? null;
                if( $account_data ) {
                    return array(
                        'success' => true,
                        'message' => __( 'Successfully connected to Klaviyo', 'triggerninja-for-ninja-forms' ),
                        'data' => array(
                            'account_name' => $account_data->attributes->contact_information->organization_name ?? 'Klaviyo Account',
                            'account_id' => $account_data->id ?? ''
                        )
                    );
                }
            }

            return array(
                'success' => false,
                'message' => __( 'Invalid response from Klaviyo API', 'triggerninja-for-ninja-forms' )
            );

        } catch( Exception $e ) {
            return array(
                'success' => false,
                /* translators: %s is the error message returned by the exception. */
                'message' => sprintf( __( 'Connection failed: %s', 'triggerninja-for-ninja-forms' ), $e->getMessage() )
            );
        }
    }

    /**
     * Get platform-specific metabox data for submissions
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'profile_id' ] ) ){
            $label = __( 'Profile ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'profile_id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'list_id' ] ) ){
            $label = __( 'List ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'list_id' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}