<?php if ( ! defined( 'ABSPATH' ) ) exit;

class TriggerNinja_Klaviyo_API
{
    const URL = 'https://a.klaviyo.com/api/';
    private $api_key;
    private $form_id;

    public function __construct( $api_key = '', $form_id = 0 )
    {
        $this->api_key = $api_key;
        $this->form_id = $form_id;
    }

    public function request( $method, $endpoint, $data = array(), $options = array() )
    {
        $url = trailingslashit( self::URL ) . $endpoint;
        
        $headers = array(
            'revision'      => '2024-02-15',
            'Authorization' => 'Klaviyo-API-Key ' . $this->api_key
        );
        
        return TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $this->form_id, 'klaviyo' );
    }
    
    /**
     * Get raw response body as array (preserves full JSON structure including links)
     * 
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param array $data Request data
     * @return array|null Parsed JSON response or null on error
     */
    public function request_raw( $method, $endpoint, $data = array() )
    {
        $url = trailingslashit( self::URL ) . $endpoint;
        
        $headers = array(
            'revision'      => '2024-02-15',
            'Authorization' => 'Klaviyo-API-Key ' . $this->api_key
        );
        
        $response = TriggerNinja_Http_Client::request( $method, $url, $data, $headers, array(), $this->form_id, 'klaviyo' );
        
        // Return raw parsed JSON for this method
        if( $response->status < 200 || $response->status >= 300 ) {
            return null;
        }
        
        // Convert response details back to array format
        return json_decode( json_encode( $response->details ), true );
    }

    /**
     * Get lists with pagination support
     * 
     * @param string $endpoint_url Full URL or endpoint for lists (for pagination)
     * @return TriggerNinja_Response
     */
    public function get_lists( $endpoint_url = 'lists' )
    {
        // If it's a full URL from pagination, extract just the endpoint part
        if( strpos( $endpoint_url, 'https://a.klaviyo.com/api/' ) === 0 ) {
            $endpoint_url = str_replace( 'https://a.klaviyo.com/api/', '', $endpoint_url );
        }
        
        return $this->request( 'GET', $endpoint_url );
    }

    /**
     * Create or update a profile
     * 
     * @param array $profile_data Profile information
     * @return TriggerNinja_Response
     */
    public function create_or_update_profile( $profile_data )
    {
        return $this->request( 'POST', 'profile-import/', $profile_data );
    }

    /**
     * Subscribe email to a list
     * 
     * @param string $list_id List ID
     * @param string $email Email address
     * @param string $source Source of subscription
     * @return TriggerNinja_Response
     */
    public function subscribe_to_list( $list_id, $email, $source = 'TriggerNinja' )
    {
        $subscription_data = array(
            'data' => array(
                'type' => 'profile-subscription-bulk-create-job',
                'attributes' => array(
                    'custom_source' => $source,
                    'profiles' => array(
                        'data' => array(
                            array(
                                'type' => 'profile',
                                'attributes' => array(
                                    'email' => $email,
                                    'subscriptions' => array(
                                        'email' => array(
                                            'marketing' => array(
                                                'consent' => 'SUBSCRIBED',
                                            )
                                        )
                                    )
                                )
                            )
                        )
                    )
                ),
                'relationships' => array(
                    'list' => array(
                        'data' => array(
                            'id' => $list_id,
                            'type' => 'list'
                        )
                    )
                )
            )
        );

        return $this->request( 'POST', 'profile-subscription-bulk-create-jobs/', $subscription_data );
    }

    /**
     * Get all lists (handles pagination internally)
     * 
     * @return array All lists
     */
    public function get_all_lists()
    {
        $all_lists = array();
        $next_url = '';
        $request_count = 0;

        do {
            $request_count++;
            
            // Construct endpoint - use 'lists' for first request, or the next URL for subsequent requests
            $endpoint = empty( $next_url ) ? 'lists' : str_replace( 'https://a.klaviyo.com/api/', '', $next_url );
            
            // Use raw request to get full JSON structure including links
            $body = $this->request_raw( 'GET', $endpoint );
            
            if( $body && isset( $body['data'] ) && ! empty( $body['data'] ) ) {
                foreach( $body['data'] as $list ) {
                    $all_lists[ $list['id'] ] = $list['attributes']['name'];
                }
                
                // Check for pagination - look for links->next
                if( isset( $body['links']['next'] ) && ! empty( $body['links']['next'] ) && $body['links']['next'] !== null ) {
                    $next_url = $body['links']['next'];
                } else {
                    $next_url = '';
                }
            } else {
                $next_url = '';
            }
        } while ( ! empty( $next_url ) && $request_count < 10 ); // Safety limit

        return $all_lists;
    }

    /**
     * Get account info for connection testing
     * 
     * @return TriggerNinja_Response
     */
    public function get_account_info()
    {
        return $this->request( 'GET', 'accounts/' );
    }
}