<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Kit Platform Integration
 */
class TriggerNinja_Kit extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_kit';
        $this->name = __( 'TN Kit', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_key = $this->get_setting( 'triggerninja_kit_api_key' );
        if( $api_key ) {
            $this->api = new TriggerNinja_Kit_API( $api_key );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        $default_fields = array(
            'triggerninja_kit_api_key' => array(
                'id'    => 'triggerninja_kit_api_key',
                'type'  => 'textbox',
                'label' => __( 'Kit API Key', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'kit',
                'value' => $this->get_setting( 'triggerninja_kit_api_key' ),
                'placeholder' => __( 'Enter your Kit API Key', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf(
                    /* translators: %s is a URL to the Kit Developer Settings page. */
                    __( 'Go to Account Settings > Developer. Create and copy V4 API Key from <a href="%s" target="_blank">Kit Developer Settings</a>', 'triggerninja-for-ninja-forms' ), 
                    'https://app.kit.com/account_settings/developer_settings' 
                )
            )
        );
        
        /**
         * Filter to modify Kit settings fields
         */
        return apply_filters( 'triggerninja_kit_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $default_fields = array(
            'subscription_type' => array(
                'name' => 'subscription_type',
                'type' => 'select',
                'label' => __( 'Subscription Type', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => array(
                    array( 'value' => 'sequence', 'label' => __( 'Subscribe to Sequence', 'triggerninja-for-ninja-forms' ) ),
                    array( 'value' => 'form', 'label' => __( 'Subscribe to Form', 'triggerninja-for-ninja-forms' ) )
                ),
                'value' => 'sequence'
            ),
            'sequence' => array(
                'name' => 'sequence',
                'type' => 'select',
                'label' => __( 'Kit Sequence', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_sequences(),
                'value' => '',
                'deps' => array(
                    'subscription_type' => 'sequence'
                )
            ),
            'form' => array(
                'name' => 'form',
                'type' => 'select',
                'label' => __( 'Kit Form', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_forms(),
                'value' => '',
                'deps' => array(
                    'subscription_type' => 'form'
                )
            )
        );
        
        /**
         * Filter to modify Kit action fields
         */
        $fields = apply_filters( 'triggerninja_kit_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Get common field structure for forms (Kit specific - excludes last_name)
     * 
     * Kit only supports first_name, not last_name
     * 
     * @return array
     */
    protected function get_common_fields()
    {
        $base_fields = array(
            'email_address' => array(
                'name' => 'email_address',
                'type' => 'textbox',
                'label' => __( 'Email Address', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            ),
            'first_name' => array(
                'name' => 'first_name',
                'type' => 'textbox',
                'label' => __( 'First Name', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            )
            // Note: last_name intentionally excluded - Kit doesn't support it
        );

        /**
         * Filter to add additional fields
         * 
         * @param array $base_fields The base fields (email, first_name only for Kit)
         * @param string $platform_id The platform ID
         * @return array Extended fields array
         */
        $extended_fields = apply_filters( 'triggerninja_platform_fields', $base_fields, $this->id );
        $extended_fields = apply_filters( "triggerninja_{$this->id}_fields", $extended_fields, $this->id );

        return $extended_fields;
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'subscription_type' ] ) ) {
            return $this->add_debug( $data, __( 'No Kit subscription type specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        $subscription_type = $action_settings[ 'subscription_type' ];
        
        if( $subscription_type === 'sequence' && empty( $action_settings[ 'sequence' ] ) ) {
            return $this->add_debug( $data, __( 'No Kit sequence specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        if( $subscription_type === 'form' && empty( $action_settings[ 'form' ] ) ) {
            return $this->add_debug( $data, __( 'No Kit form specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API key (with extensibility hook for advanced users)
        $api_key = $this->get_setting( 'triggerninja_kit_api_key' );
        
        /**
         * Filter to modify Kit API key for advanced setups
         */
        $api_key = apply_filters( 'triggerninja_kit_api_key', $api_key, $action_settings, $this );
        
        if( empty( $api_key ) ) {
            return $this->add_debug( $data, __( 'Kit API key not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance
        $api = new TriggerNinja_Kit_API( $api_key, $form_id );

        // Prepare profile data
        $email = trim( $action_settings[ 'email_address' ] );
        $first_name = isset( $action_settings[ 'first_name' ] ) ? trim( $action_settings[ 'first_name' ] ) : '';

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', compact( 'email', 'first_name' ), $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", compact( 'email', 'first_name' ), $action_settings, $form_id, $data );

        // First, create subscriber
        $subscriber_response = $api->create_subscriber( $email, $first_name, 'active' );

        // Then subscribe to sequence or form
        if( $subscription_type === 'sequence' ) {
            $subscription_response = $api->subscribe_to_sequence( $action_settings[ 'sequence' ], $email );
            $target_id = $action_settings[ 'sequence' ];
            $target_type = 'sequence';
        } else {
            $subscription_response = $api->subscribe_to_form( $action_settings[ 'form' ], $email );
            $target_id = $action_settings[ 'form' ];
            $target_type = 'form';
        }

        // Check for successful subscription (we check the subscription response as it's the main action)
        if( $subscription_response->status >= 200 && $subscription_response->status < 300 ) {
            $response_data = array(
                'email' => $email,
                'first_name' => $first_name,
                'subscription_type' => $target_type,
                'target_id' => $target_id,
                'status' => 'subscribed',
                'subscriber_created' => ( $subscriber_response->status >= 200 && $subscriber_response->status < 300 )
            );

            /**
             * Filter response data before saving
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, compact( 'email', 'first_name' ), $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, compact( 'email', 'first_name' ), $action_settings );
            $data = $this->save_submission_data( $data, $response_data );
            /* translators: %s is the target type (sequence or form). */
            $data = $this->add_debug( $data, sprintf( __( 'Successfully subscribed to Kit %s', 'triggerninja-for-ninja-forms' ), $target_type ), 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, compact( 'email', 'first_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, compact( 'email', 'first_name' ), $action_settings, $form_id, $data );
            
        } else {
            $error_message = $subscription_response->error ? $subscription_response->error : __( 'Failed to subscribe to Kit', 'triggerninja-for-ninja-forms' );
            /* translators: %s is the error message returned by the API. */
            $data = $this->add_debug( $data, sprintf( __( 'Kit error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, compact( 'email', 'first_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, compact( 'email', 'first_name' ), $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available sequences from Kit
     * 
     * @return array
     */
    public function get_sequences()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API key first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_sequences();
        
        if( ! $response->details || ! isset( $response->details->sequences ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading sequences', 'triggerninja-for-ninja-forms' ) ) );
        }

        $sequences = array_map( array( $this, 'format_option' ), $response->details->sequences );
        array_unshift( $sequences, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify Kit sequences
         */
        return apply_filters( 'triggerninja_kit_sequences', $sequences, $this );
    }

    /**
     * Get available forms from Kit
     * 
     * @return array
     */
    public function get_forms()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API key first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_forms();
        
        if( ! $response->details || ! isset( $response->details->forms ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading forms', 'triggerninja-for-ninja-forms' ) ) );
        }

        $forms = array_map( array( $this, 'format_option' ), $response->details->forms );
        array_unshift( $forms, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify Kit forms
         */
        return apply_filters( 'triggerninja_kit_forms', $forms, $this );
    }

    /**
     * Get available lists from Kit (required by interface)
     * 
     * Note: Kit uses sequences and forms instead of traditional lists.
     * This method returns sequences for compatibility with the interface.
     * 
     * @return array
     */
    public function get_lists()
    {
        // For Kit, we return sequences as "lists" since that's the closest concept
        return $this->get_sequences();
    }
    
    /**
     * Format option for dropdown
     * 
     * @param object $item
     * @return array
     */
    private function format_option( $item )
    {
        return array(
            'value' => $item->id,
            'label' => $item->name
        );
    }

    /**
     * Validate Kit connection
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_kit_api_key' ) ) && $this->api !== null;
    }

    /**
     * Test connection to Kit
     * 
     * @return array
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return array(
                'success' => false,
                'message' => __( 'API key not configured', 'triggerninja-for-ninja-forms' )
            );
        }

        try {
            $response = $this->api->get_account_info();
            
            if( $response->status >= 200 && $response->status < 300 && $response->details ) {
                return array(
                    'success' => true,
                    'message' => __( 'Successfully connected to Kit', 'triggerninja-for-ninja-forms' ),
                    'data' => array(
                        'account_id' => is_object( $response->details ) && isset( $response->details->id ) ? $response->details->id : 'connected'
                    )
                );
            }

            return array(
                'success' => false,
                'message' => __( 'Invalid response from Kit API', 'triggerninja-for-ninja-forms' )
            );

        } catch( Exception $e ) {
            return array(
                'success' => false,
                /* translators: %s is the error message returned by the exception. */
                'message' => sprintf( __( 'Connection failed: %s', 'triggerninja-for-ninja-forms' ), $e->getMessage() )
            );
        }
    }

    /**
     * Get platform-specific metabox data for submissions
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'first_name' ] ) && ! empty( $submission_data[ 'first_name' ] ) ){
            $label = __( 'First Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'first_name' ];
        }

        if( isset( $submission_data[ 'subscription_type' ] ) ){
            $label = __( 'Subscription Type', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = ucfirst( $submission_data[ 'subscription_type' ] );
        }

        if( isset( $submission_data[ 'target_id' ] ) ){
            $label = __( 'Target ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'target_id' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }
        
        if( isset( $submission_data[ 'subscriber_created' ] ) ){
            $label = __( 'Subscriber Created', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'subscriber_created' ] ? __( 'Yes', 'triggerninja-for-ninja-forms' ) : __( 'No', 'triggerninja-for-ninja-forms' );
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}