<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * GetResponse Platform Integration
 */
class TriggerNinja_GetResponse extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_getresponse';
        $this->name = __( 'TN GetResponse', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_key = $this->get_setting( 'triggerninja_getresponse_api_key' );
        if( $api_key ) {
            $this->api = new TriggerNinja_GetResponse_API( $api_key );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        $default_fields = array(
            'triggerninja_getresponse_api_key' => array(
                'id'    => 'triggerninja_getresponse_api_key',
                'type'  => 'textbox',
                'label' => __( 'GetResponse API Key', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'getresponse',
                'value' => $this->get_setting( 'triggerninja_getresponse_api_key' ),
                'placeholder' => __( 'Enter your GetResponse API Key', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf(
                    /* translators: %s is a URL to the GetResponse API settings page. */
                    __( 'Get your API Key from <a href="%s" target="_blank">GetResponse API Settings</a>', 'triggerninja-for-ninja-forms' ), 
                    'https://app.getresponse.com/api' 
                )
            )
        );
        
        /**
         * Filter to modify GetResponse settings fields
         */
        return apply_filters( 'triggerninja_getresponse_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $default_fields = array(
            'campaign' => array(
                'name' => 'campaign',
                'type' => 'select',
                'label' => __( 'GetResponse Campaign', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_campaigns(),
                'value' => ''
            )
        );
        
        /**
         * Filter to modify GetResponse action fields
         */
        $fields = apply_filters( 'triggerninja_getresponse_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'campaign' ] ) ) {
            return $this->add_debug( $data, __( 'No GetResponse campaign specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API key
        $api_key = $this->get_setting( 'triggerninja_getresponse_api_key' );
        if( empty( $api_key ) ) {
            return $this->add_debug( $data, __( 'GetResponse API key not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance with form_id for logging
        $api = new TriggerNinja_GetResponse_API( $api_key, $form_id );

        // Prepare contact data
        $email = trim( $action_settings[ 'email_address' ] );
        $first_name = isset( $action_settings[ 'first_name' ] ) ? trim( $action_settings[ 'first_name' ] ) : '';
        $last_name = isset( $action_settings[ 'last_name' ] ) ? trim( $action_settings[ 'last_name' ] ) : '';
        
        // Combine first and last name for GetResponse 'name' field
        $name_parts = array_filter( array( $first_name, $last_name ) );
        $full_name = implode( ' ', $name_parts );

        /**
         * Filter to process additional fields
         */
        $contact_data = apply_filters( 'triggerninja_process_fields', array(
            'email' => $email,
            'name' => $full_name,
            'first_name' => $first_name,
            'last_name' => $last_name
        ), $action_settings, $form_id, $data );
        $contact_data = apply_filters( "triggerninja_{$this->id}_process_fields", $contact_data, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', $contact_data, $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", $contact_data, $action_settings, $form_id, $data );

        // Subscribe to campaign
        $response = $api->subscribe_to_campaign( 
            $action_settings[ 'campaign' ], 
            $email, 
            $full_name
        );

        // Check for successful subscription
        if( $response->status >= 200 && $response->status < 300 ) {
            $response_details = $response->details;
            $contact_id = is_object( $response_details ) && isset( $response_details->contactId ) ? $response_details->contactId : 'subscribed';
            
            $response_data = array(
                'contact_id' => $contact_id,
                'email' => $email,
                'name' => $full_name,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'campaign_id' => $action_settings[ 'campaign' ],
                'status' => 'subscribed'
            );

            /**
             * Filter response data before saving
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, $contact_data, $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, $contact_data, $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            $data = $this->add_debug( $data, __( 'Successfully subscribed to GetResponse campaign', 'triggerninja-for-ninja-forms' ), 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, $contact_data, $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, $contact_data, $action_settings, $form_id, $data );
            
        } else {
            $error_message = $response->error ? $response->error : __( 'Failed to subscribe to GetResponse campaign', 'triggerninja-for-ninja-forms' );
            /* translators: %s is an error message. */
            $data = $this->add_debug( $data, sprintf( __( 'GetResponse error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, $contact_data, $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, $contact_data, $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available campaigns from GetResponse
     * 
     * @return array
     */
    public function get_campaigns()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API key first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_campaigns();
        
        if( ! $response->details || ! is_array( $response->details ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading campaigns', 'triggerninja-for-ninja-forms' ) ) );
        }

        $campaigns = array();
        foreach( $response->details as $campaign ) {
            $campaigns[] = array(
                'value' => $campaign->campaignId,
                'label' => $campaign->name
            );
        }
        
        array_unshift( $campaigns, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify GetResponse campaigns
         */
        return apply_filters( 'triggerninja_getresponse_campaigns', $campaigns, $this );
    }
    
    /**
     * Check if platform is connected (has valid API key)
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_getresponse_api_key' ) ) && $this->api !== null;
    }
    
    /**
     * Test API connection
     * 
     * @return bool
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return false;
        }
        
        $response = $this->api->test_connection();
        return $response->status >= 200 && $response->status < 300;
    }
    
    /**
     * Get available lists from GetResponse (alias for get_campaigns)
     * Required by TriggerNinja_Platform_Interface
     * 
     * @return array
     */
    public function get_lists()
    {
        return $this->get_campaigns();
    }
    
    /**
     * Get platform-specific metabox data for submissions
     * Required by TriggerNinja_Platform_Interface
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'contact_id' ] ) ){
            $label = __( 'Contact ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'contact_id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'name' ] ) ){
            $label = __( 'Full Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'name' ];
        }

        if( isset( $submission_data[ 'campaign_id' ] ) ){
            $label = __( 'Campaign ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'campaign_id' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}