<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * GetResponse API Integration
 */
class TriggerNinja_GetResponse_API
{
    const URL = 'https://api.getresponse.com/v3/';
    
    /**
     * API Key
     * 
     * @var string
     */
    private $api_key;
    
    /**
     * Form ID for logging
     * 
     * @var int
     */
    private $form_id;
    
    /**
     * Constructor
     * 
     * @param string $api_key GetResponse API key
     * @param int $form_id Form ID for logging
     */
    public function __construct( $api_key = '', $form_id = 0 )
    {
        $this->api_key = $api_key;
        $this->form_id = $form_id;
    }
    
    /**
     * Make API request to GetResponse using centralized HTTP client
     * 
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array $data Request data
     * @return TriggerNinja_Response
     */
    private function request( $endpoint, $method = 'GET', $data = array() )
    {
        $url = self::URL . $endpoint;
        
        $headers = array(
            'Content-Type' => 'application/json',
            'X-Auth-Token' => 'api-key ' . $this->api_key
        );
        
        $options = array(
            'timeout' => 30
        );
        
        $response = TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $this->form_id, 'getresponse' );
        
        // Handle GetResponse specific error format
        if ( $response->status >= 400 && $response->details ) {
            $error_message = 'Unknown error occurred';
            
            // Try to extract GetResponse error message
            if ( is_object( $response->details ) && isset( $response->details->message ) ) {
                $error_message = $response->details->message;
            } elseif ( is_array( $response->details ) && isset( $response->details['message'] ) ) {
                $error_message = $response->details['message'];
            } elseif ( $response->error ) {
                $error_message = $response->error;
            }
            
            return new TriggerNinja_Response( $response->status, array(), $error_message );
        }
        
        return $response;
    }
    
    /**
     * Get all campaigns (lists)
     * 
     * @return TriggerNinja_Response
     */
    public function get_campaigns()
    {
        return $this->request( 'campaigns' );
    }
    
    /**
     * Subscribe contact to a campaign
     * 
     * @param string $campaign_id Campaign ID
     * @param string $email Contact email address
     * @param string $name Contact name (optional)
     * @param string $ip_address IP address (optional)
     * @return TriggerNinja_Response
     */
    public function subscribe_to_campaign( $campaign_id, $email, $name = '', $ip_address = '' )
    {
        if ( empty( $campaign_id ) || empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'Campaign ID and email are required' );
        }
        
        $contact_data = array(
            'email' => $email,
            'campaign' => array(
                'campaignId' => $campaign_id
            ),
            'dayOfCycle' => 0
        );
        
        // Add optional name
        if ( ! empty( $name ) ) {
            $contact_data['name'] = $name;
        }
        
        // Add optional IP address
        if ( ! empty( $ip_address ) ) {
            $contact_data['ipAddress'] = $ip_address;
        }
        
        return $this->request( 'contacts', 'POST', $contact_data );
    }
    
    /**
     * Get contact by email
     * 
     * @param string $email Contact email address
     * @return TriggerNinja_Response
     */
    public function get_contact( $email )
    {
        if ( empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'Email is required' );
        }
        
        return $this->request( "contacts?query[email]=" . urlencode( $email ) );
    }
    
    /**
     * Test API connection by getting campaigns
     * 
     * @return TriggerNinja_Response
     */
    public function test_connection()
    {
        return $this->get_campaigns();
    }
    
    /**
     * Get account info for connection testing
     * 
     * @return TriggerNinja_Response
     */
    public function get_account_info()
    {
        return $this->get_campaigns();
    }
}