<?php if ( ! defined( 'ABSPATH' ) ) exit;

require_once plugin_dir_path( __FILE__ ) . 'elasticemail-api.php';

class TriggerNinja_ElasticEmail extends TriggerNinja_Platform_Abstract
{
    protected $id = 'elasticemail';
    protected $name = 'TN Elastic Email';

    public function __construct()
    {
        parent::__construct();
        add_action( 'triggerninja_platform_config_saved', array( $this, 'platform_config_saved' ), 10, 3 );
    }

    protected function init_api()
    {
        $this->settings['api_key'] = get_option( 'triggerninja_elasticemail_api_key', '' );
        $this->settings['public_account_id'] = get_option( 'triggerninja_elasticemail_public_accountid', '' );

        $this->api = new stdClass();
        $this->api->api_key = $this->settings['api_key'];
        $this->api->public_account_id = $this->settings['public_account_id'];
    }

    public function get_settings_fields()
    {
        return array(
            'triggerninja_elasticemail_api_key' => array(
                'name' => 'triggerninja_elasticemail_api_key',
                'type' => 'text',
                'label' => __( 'API Key', 'triggerninja-for-ninja-forms' ),
                'default' => ''
            )
        );
    }

    public function get_action_fields()
    {
        $fields = array(
            'list' => array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'List', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_lists()
            )
        );
        return array_merge($fields, $this->get_common_fields());
    }

    public function is_connected()
    {
        return ! empty( $this->api->api_key ) && ! empty( $this->api->public_account_id );
    }

    public function get_lists()
    {
        if ( ! $this->is_connected() ) {
            return array(
                array(
                    'label' => __( 'Please connect to Elastic Email first', 'triggerninja-for-ninja-forms' ),
                    'value' => ''
                )
            );
        }

        $res = TriggerNinja_ElasticEmail_API::api_get_lists( $this->api->api_key );
        if ( $res->status >= 200 && $res->status < 300 ) {
            $lists = array();
            if ( $res->details && is_array( $res->details ) ) {
                foreach ( $res->details as $list ) {
                    $lists[] = array(
                        'label' => $list->listname,
                        'value' => $list->publiclistid
                    );
                }
            }

            array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );
            
            return $lists;
        }
        return array(
            array(
                'label' => __( 'Error fetching lists', 'triggerninja-for-ninja-forms' ),
                'value' => ''
            )
        );
    }

    public function process_submission( $action_settings, $form_id, $data )
    {
        if ( ! $this->is_connected() ) {
            return $this->add_debug( $data, 'Elastic Email not connected', 'error' );
        }

        if ( empty( $action_settings['list'] ) ) {
            return $this->add_debug( $data, 'No Elastic Email list specified', 'error' );
        }

        $email = $action_settings['email_address'] ?? '';
        $first_name = $action_settings['first_name'] ?? '';
        $last_name = $action_settings['last_name'] ?? '';

        if ( empty( $email ) ) {
            return $this->add_debug( $data, 'No email provided', 'error' );
        }

        $contact_data = array(
            'publiclistid'    => $action_settings['list'],
            'firstName'       => $first_name,
            'lastName'        => $last_name,
            'email'           => $email,
            'publicAccountID' => $this->api->public_account_id
        );

        $res = TriggerNinja_ElasticEmail_API::api_create_or_update_contact( $this->api->api_key, $contact_data, $form_id );

        if ( $res->status >= 200 && $res->status < 300 ) {
            $data = $this->save_submission_data( $data, $res->details );
            return $this->add_debug( $data, 'Contact sent to Elastic Email', 'success' );
        } else {
            return $this->add_debug( $data, 'Elastic Email API error: ' . $res->error, 'error' );
        }
    }

    public function get_submission_metabox_data( $submission_data )
    {
        return isset( $submission_data['extra'][ $this->get_id() ] ) ? $submission_data['extra'][ $this->get_id() ] : array();
    }

    public function platform_config_saved( $platform_id, $saved_fields, $platform )
    {
        if ( $platform_id !== 'elasticemail' ) {
            return;
        }

        if ( ! empty( $saved_fields['triggerninja_elasticemail_api_key'] ) ) {
            $res = TriggerNinja_ElasticEmail_API::api_get_account( $saved_fields['triggerninja_elasticemail_api_key'] );
            if ( $res->status >= 200 && $res->status < 300 ) {
                if ( ! empty( $res->details->publicaccountid ) ) {
                    update_option( 'triggerninja_elasticemail_public_accountid', $res->details->publicaccountid );
                }
            }
        }
    }
}