<?php if ( ! defined( 'ABSPATH' ) ) exit;

require_once plugin_dir_path( __FILE__ ) . 'constantcontact-api.php';

class TriggerNinja_ConstantContact extends TriggerNinja_Platform_Abstract
{
    protected $id = 'constantcontact';
    protected $name = 'TN Constant Contact';

    protected function init_api()
    {
        $tokens = $this->get_tokens();

        // Check if tokens need to be refreshed
        if ( ! empty( $tokens['refresh_token'] ) && time() > $tokens['expires'] ) {
            $response = TriggerNinja_ConstantContact_API::refresh_access_token( $tokens['refresh_token'] );

            if ( $response->status === 200 && ! empty( $response->details->access_token ) ) {
                $new_tokens = array(
                    'access_token'  => $response->details->access_token,
                    'refresh_token' => $response->details->refresh_token,
                    'expires_in'    => $response->details->expires_in,
                );
                $this->save_tokens( $new_tokens );
                $tokens = $this->get_tokens(); // Get the updated tokens
            } else {
                // Refresh failed, clear tokens to force re-authentication
                $this->clear_tokens();
                return; // Stop initialization
            }
        }

        if ( $tokens && ! empty( $tokens['access_token'] ) ) {
            $this->api = new TriggerNinja_ConstantContact_API(
                $tokens['access_token'],
                $tokens['refresh_token'] ?? ''
            );
        }
    }

    /**
     * Platform settings fields
     */
    public function get_settings_fields()
    {
        if (!function_exists('wp_create_nonce')) {
            return array();
        }

        if ($this->is_connected()) {
            $disconnect_nonce = wp_create_nonce( 'triggerninja_constantcontact_auth' );
            return array(
                'main_config' => array(
                    'id' => 'triggerninja_constantcontact_main_config',
                    'type' => 'html',
                    'content' => '<p style="color: green;">' . __('Connected', 'triggerninja-for-ninja-forms') . '</p>' .
                        '<button type="button" class="button button-secondary" id="constantcontact-disconnect">' . __( 'Disconnect', 'triggerninja-for-ninja-forms' ) . '</button>' .
                        '<input type="hidden" id="constantcontact-disconnect-nonce" value="' . esc_attr( $disconnect_nonce ) . '" />'
                )
            );
        }

        $redirect_uri = admin_url( 'admin-ajax.php?action=triggerninja_constantcontact_exchange_token' );

        $html = '<div style="margin-bottom: 15px;">';
        $html .= '<p style="margin-bottom: 10px;">' . __( 'Please copy this redirect URI and paste it into your Constant Contact application settings:', 'triggerninja-for-ninja-forms' ) . '</p>';
        $html .= '<input type="text" readonly value="' . esc_attr( $redirect_uri ) . '" style="width: 100%;" />';
        $html .= '</div>';

        // Generate PKCE codes for the authorization URL
        $pkce = TriggerNinja_ConstantContact_API::generate_pkce();
        $nonce = wp_create_nonce( 'triggerninja_constantcontact_auth' );
        $auth_url = TriggerNinja_ConstantContact_API::get_authorization_url( $pkce['code_challenge'], $nonce, $redirect_uri );

        update_option( 'triggerninja_constantcontact_pkce', $pkce );

        $html .= '<div style="margin-bottom: 20px;">';
        $html .= '<p style="margin-bottom: 15px;">' . __( 'To connect Constant Contact, please click the button below:', 'triggerninja-for-ninja-forms' ) . '</p>';
        $html .= '<a href="' . esc_url( $auth_url ) . '" class="button button-primary">' . __( 'Authorize with Constant Contact', 'triggerninja-for-ninja-forms' ) . '</a>';
        $html .= '</div>';

        return array(
            'main_config' => array(
                'id' => 'triggerninja_constantcontact_main_config',
                'type' => 'html',
                'content' => $html
            )
        );
    }



    /**
     * Action fields used when adding the platform as an action to a form
     */
    public function get_action_fields()
    {
        $fields = array(
            'list' => array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'List', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_lists()
            )
        );
        return array_merge($fields, $this->get_common_fields());
    }

    public function is_connected()
    {
        return ! empty( $this->api );
    }

    public function save_tokens( $tokens )
    {
        if ( isset( $tokens['access_token'] ) ) {
            update_option( 'triggerninja_constantcontact_access_token', $tokens['access_token'] );
        }
        if ( isset( $tokens['refresh_token'] ) ) {
            update_option( 'triggerninja_constantcontact_refresh_token', $tokens['refresh_token'] );
        }
        if ( isset( $tokens['expires_in'] ) ) {
            // Add a small buffer (e.g., 60 seconds) to the expiration time
            $expires_time = time() + $tokens['expires_in'] - 60;
            update_option( 'triggerninja_constantcontact_token_expires', $expires_time );
        }

        // Reinitialize API with new tokens
        $this->init_api();
    }

    private function get_tokens()
    {
        $access_token = get_option( 'triggerninja_constantcontact_access_token', '' );
        $refresh_token = get_option( 'triggerninja_constantcontact_refresh_token', '' );
        $expires = get_option( 'triggerninja_constantcontact_token_expires', 0 );

        return array(
            'access_token' => $access_token,
            'refresh_token' => $refresh_token,
            'expires' => $expires
        );
    }

    public function clear_tokens()
    {
        delete_option( 'triggerninja_constantcontact_access_token' );
        delete_option( 'triggerninja_constantcontact_refresh_token' );
        delete_option( 'triggerninja_constantcontact_token_expires' );
        $this->api = null;
    }

    /**
     * Reload settings from database and reinitialize API
     */
    public function reload_settings()
    {
        $this->load_settings();
        $this->init_api();
    }

    public function get_lists()
    {
        if ( empty( $this->api ) ) {
            return new WP_Error( 'not_connected', 'Constant Contact not connected' );
        }

        $res = $this->api->get_lists();
        if ( $res->status >= 200 && $res->status < 300 ) {
            $lists = array();
            if ( $res->details->lists && is_array( $res->details->lists ) ){
                foreach ( $res->details->lists as $list ) {
                    $lists[] = array(
                        'label' => $list->name,
                        'value' => $list->list_id
                    );
                }
            }

            array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );

            return $lists;
        }
        
    }

    public function create_or_update_contact( $contact_data, $form_id = 0 )
    {
        if ( empty( $this->api ) ) {
            return new WP_Error( 'not_connected', 'Constant Contact not connected' );
        }

        $res = $this->api->create_or_update_contact( $contact_data, $form_id );
        if ( $res->status >= 200 && $res->status < 300 ) {
            return $res->details;
        }
        return new WP_Error( 'api_error', $res->body );
    }

    /**
     * Process form submission and send contact to Constant Contact
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        $email = $action_settings['email_address'] ?? '';
        $first = $action_settings['first_name'] ?? '';
        $last = $action_settings['last_name'] ?? '';
        $list_id = $action_settings['list'] ?? '';

        if ( empty( $email ) ) {
            return $this->add_debug( $data, 'No email provided', 'error', $form_id );
        }

        $contact = array(
            'email_address' => array( 'address' => $email ),
            'first_name' => $first,
            'last_name' => $last,
            'list_memberships' => isset( $list_id ) ? array( $list_id ) : array()
        );

        $res = $this->create_or_update_contact( $contact, $form_id );

        // Save some response data to submission
        $data = $this->save_submission_data( $data, $res );
    }

    /**
     * Provide submission meta box data
     */
    public function get_submission_metabox_data( $submission_data )
    {
        // Return any stored Constant Contact response data
        return isset( $submission_data['extra'][ $this->get_id() ] ) ? $submission_data['extra'][ $this->get_id() ] : array();
    }

}
