<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Constant Contact API helper
 * Implements PKCE flow similarly to AWeber implementation
 */
class TriggerNinja_ConstantContact_API
{
    const AUTH_BASE = 'https://authz.constantcontact.com/oauth2/default/v1';
    const TOKEN_URL = self::AUTH_BASE . '/token';
    const AUTHORIZE_URL = self::AUTH_BASE . '/authorize';
    const API_BASE = 'https://api.cc.email/v3';

    /**
     * Form ID for logging
     * 
     * @var int
     */
    private $form_id;
    
    /**
     * Access token
     * 
     * @var string
     */
    private $access_token;
    
    /**
     * Refresh token
     * 
     * @var string
     */
    private $refresh_token;

    // Client ID provided
    private static $client_id = '0682625b-4cb1-4b62-87d4-a224eb18f6c2';

    public function __construct( $access_token = '', $refresh_token = '', $form_id = 0)
    {
        $this->form_id = $form_id;
        $this->access_token = $access_token;
        $this->refresh_token = $refresh_token;
    }

    public static function get_client_id()
    {
        return self::$client_id;
    }

    public static function generate_pkce()
    {
        $verifier = bin2hex( random_bytes( 32 ) );
        $challenge = rtrim( strtr( base64_encode( hash( 'sha256', $verifier, true ) ), '+/', '-_' ), '=' );
        return array(
            'code_verifier' => $verifier,
            'code_challenge' => $challenge
        );
    }

    public static function get_authorization_url( $code_challenge, $nonce, $redirect_uri )
    {
        $params = array(
            'response_type' => 'code',
            'client_id' => self::$client_id,
            'redirect_uri' => $redirect_uri,
            'code_challenge' => $code_challenge,
            'code_challenge_method' => 'S256',
            'scope' => 'contact_data campaign_data offline_access',
            'state' => $nonce
        );

        return self::AUTHORIZE_URL . '?' . http_build_query( $params );
    }

    public static function exchange_code_for_tokens( $code, $code_verifier, $redirect_uri )
    {
        $body = array(
            'grant_type' => 'authorization_code',
            'client_id' => self::$client_id,
            'code' => $code,
            'redirect_uri' => $redirect_uri,
            'code_verifier' => $code_verifier
        );

        $response = wp_remote_post( self::TOKEN_URL, array(
            'headers' => array( 'Content-Type' => 'application/x-www-form-urlencoded' ),
            'body' => http_build_query( $body ),
            'timeout' => 20
        ) );

        return self::normalize_response( $response );
    }

    public static function refresh_access_token( $refresh_token )
    {
        $body = array(
            'grant_type' => 'refresh_token',
            'client_id' => self::$client_id,
            'refresh_token' => $refresh_token
        );

        $response = wp_remote_post( self::TOKEN_URL, array(
            'headers' => array( 'Content-Type' => 'application/x-www-form-urlencoded' ),
            'body' => http_build_query( $body ),
            'timeout' => 20
        ) );

        return self::normalize_response( $response );
    }

    private static function normalize_response( $wp_response )
    {
        if ( is_wp_error( $wp_response ) ) {
            return (object) array( 'status' => 500, 'error' => $wp_response->get_error_message() );
        }

        $status = wp_remote_retrieve_response_code( $wp_response );
        $body = wp_remote_retrieve_body( $wp_response );
        $decoded = json_decode( $body );

        return (object) array(
            'status' => $status,
            'body' => $body,
            'details' => $decoded
        );
    }

    private function request( $endpoint, $method = 'GET', $data = array(), $form_id = 0 )
    {
        if ( empty( $this->access_token ) ) {
            return new TriggerNinja_Response( 401, array(), 'No access token available' );
        }

        $url = self::API_BASE . $endpoint;

        $headers = array(
            'Accept' => 'application/json',
            'Content-Type' => 'application/json; charset=utf-8',
            'Authorization' => 'Bearer ' . $this->access_token
        );

        $options = array(
            'timeout' => 30
        );

        $response = TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $form_id, 'constantcontact' );

        return $response;
    }

    public function get_lists()
    {
        return $this->request( '/contact_lists', 'GET' );
    }

    public function create_or_update_contact( $contact_data, $form_id = 0 )
    {
        return $this->request( '/contacts/sign_up_form', 'POST', $contact_data, $form_id );
    }

    public function test_connection()
    {
        return $this->get_lists();
    }
}
