<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Campaign Monitor Platform Integration
 */
class TriggerNinja_CampaignMonitor extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_campaignmonitor';
        $this->name = __( 'TN Campaign Monitor', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_token = $this->get_setting( 'triggerninja_campaignmonitor_api_token' );
        if( $api_token ) {
            $this->api = new TriggerNinja_CampaignMonitor_API( $api_token );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        $default_fields = array(
            'triggerninja_campaignmonitor_api_token' => array(
                'id'    => 'triggerninja_campaignmonitor_api_token',
                'type'  => 'textbox',
                'label' => __( 'Campaign Monitor API Token', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'campaignmonitor',
                'value' => $this->get_setting( 'triggerninja_campaignmonitor_api_token' ),
                'placeholder' => __( 'Enter your Campaign Monitor API Token', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf(
                    /* translators: %s is a URL to the Campaign Monitor API Keys help page. */
                    __( 'Go to Account Settings > API Keys to get your API token from <a href="%s" target="_blank">Campaign Monitor</a>', 'triggerninja-for-ninja-forms' ), 
                    'https://help.campaignmonitor.com/api-keys' 
                )
            )
        );
        
        /**
         * Filter to modify Campaign Monitor settings fields
         */
        return apply_filters( 'triggerninja_campaignmonitor_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        // Pre-fetch all clients and their lists
        $clients_with_lists = $this->get_clients_with_lists();
        
        $default_fields = array(
            'client' => array(
                'name' => 'client',
                'type' => 'select',
                'label' => __( 'Campaign Monitor Client', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->format_clients_for_dropdown( $clients_with_lists ),
                'value' => '',
                'desc' => __( 'Select the Campaign Monitor client account.', 'triggerninja-for-ninja-forms' )
            )
        );
        
        // Add a list field for each client
        foreach( $clients_with_lists as $client_id => $client_data ) {
            if( empty( $client_id ) ) continue; // Skip empty client
            
            $default_fields["list_{$client_id}"] = array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'Campaign Monitor List', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $client_data['lists'],
                'value' => '',
                /* translators: %s is a client name. */
                'desc' => sprintf( __( 'Select a list from %s.', 'triggerninja-for-ninja-forms' ), $client_data['name'] ),
                'deps' => array(
                    'client' => $client_id
                )
            );
        }
        
        $default_fields['resubscribe'] = array(
            'name' => 'resubscribe',
            'type' => 'textbox',
            'label' => __( 'Resubscribe Existing', 'triggerninja-for-ninja-forms' ),
            'group' => 'primary',
            'width' => 'full',
            'value' => 'true',
            'placeholder' => __( 'Enter true or false', 'triggerninja-for-ninja-forms' ),
            'desc' => __( 'Enter "true" to resubscribe existing subscribers, or "false" to skip them.', 'triggerninja-for-ninja-forms' )
        );
        
        /**
         * Filter to modify Campaign Monitor action fields
         */
        $fields = apply_filters( 'triggerninja_campaignmonitor_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'client' ] ) ) {
            return $this->add_debug( $data, __( 'No Campaign Monitor client specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'list' ] ) ) {
            return $this->add_debug( $data, __( 'No Campaign Monitor list specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API token (with extensibility hook for advanced users)
        $api_token = $this->get_setting( 'triggerninja_campaignmonitor_api_token' );
        
        /**
         * Filter to modify Campaign Monitor API token for advanced setups
         */
        $api_token = apply_filters( 'triggerninja_campaignmonitor_api_token', $api_token, $action_settings, $this );
        
        if( empty( $api_token ) ) {
            return $this->add_debug( $data, __( 'Campaign Monitor API token not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance
        $api = new TriggerNinja_CampaignMonitor_API( $api_token, $form_id );

        // Prepare profile data
        $email = trim( $action_settings[ 'email_address' ] );
        $first_name = isset( $action_settings[ 'first_name' ] ) ? trim( $action_settings[ 'first_name' ] ) : '';
        $last_name = isset( $action_settings[ 'last_name' ] ) ? trim( $action_settings[ 'last_name' ] ) : '';
        
        // Build full name
        $name_parts = array_filter( array( $first_name, $last_name ) );
        $full_name = implode( ' ', $name_parts );

        // Prepare custom fields for Campaign Monitor
        $custom_fields = array();
        
        if( $first_name ) {
            $custom_fields['FirstName'] = $first_name;
        }
        
        if( $last_name ) {
            $custom_fields['LastName'] = $last_name;
        }

        /**
         * Filter to process additional fields
         */
        $custom_fields = apply_filters( 'triggerninja_process_fields', $custom_fields, $action_settings, $form_id, $data );
        $custom_fields = apply_filters( "triggerninja_{$this->id}_process_fields", $custom_fields, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );

        // Get resubscribe setting from action settings
        $resubscribe_value = isset( $action_settings[ 'resubscribe' ] ) ? trim( strtolower( $action_settings[ 'resubscribe' ] ) ) : 'true';
        
        // Validate resubscribe value
        if( ! in_array( strtolower( $action_settings[ 'resubscribe' ] ), array( 'true', 'false' ) ) ) {
            /* translators: %s is the invalid value. */
            return $this->add_debug( $data, sprintf( __( 'Invalid Resubscribe value "%s". Please enter "true" or "false".', 'triggerninja-for-ninja-forms' ), $action_settings[ 'resubscribe' ] ), 'error' );
        }
        
        $resubscribe = ( $resubscribe_value === 'true' || $resubscribe_value === '1' );
        
        // Use the Campaign Monitor API to subscribe
        $subscription_response = $api->subscribe_to_list( 
            $action_settings[ 'list' ], 
            $email, 
            $full_name,
            $custom_fields,
            $resubscribe
        );

        // Check for successful subscription
        if( $subscription_response->status >= 200 && $subscription_response->status < 300 ) {
            $response_details = $subscription_response->details;
            $contact_id = is_object( $response_details ) && isset( $response_details->EmailAddress ) ? $response_details->EmailAddress : 'subscribed';
            
            $response_data = array(
                'contact_id' => $contact_id,
                'email' => $email,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'full_name' => $full_name,
                'client_id' => $action_settings[ 'client' ],
                'list_id' => $action_settings[ 'list' ],
                'status' => 'subscribed',
                'resubscribe' => $resubscribe,
                'custom_fields' => $custom_fields
            );

            /**
             * Filter response data before saving
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            $data = $this->add_debug( $data, __( 'Successfully subscribed to Campaign Monitor list', 'triggerninja-for-ninja-forms' ), 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            
        } else {
            $error_message = $subscription_response->error ? $subscription_response->error : __( 'Failed to subscribe to Campaign Monitor list', 'triggerninja-for-ninja-forms' );
            /* translators: %s is an error message. */
            $data = $this->add_debug( $data, sprintf( __( 'Campaign Monitor error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get all clients with their lists pre-fetched
     * 
     * @return array
     */
    public function get_clients_with_lists()
    {
        if( ! $this->is_connected() ) {
            return array();
        }

        $clients_response = $this->api->get_clients();
        
        if( ! $clients_response->details ) {
            return array();
        }

        $clients_with_lists = array();
        
        foreach( $clients_response->details as $client ) {
            $client_id = $client->ClientID;
            $client_name = $client->Name;
            
            // Get lists for this client
            $lists_response = $this->api->get_lists( $client_id );
            $lists = array();
            
            if( $lists_response->details ) {
                $lists = array_map( array( $this, 'format_list_options' ), $lists_response->details );
            }
            
            // Always add a default "select" option at the beginning
            array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );
            
            $clients_with_lists[ $client_id ] = array(
                'name' => $client_name,
                'lists' => $lists
            );
        }
        
        return $clients_with_lists;
    }
    
    /**
     * Format clients for dropdown (just the client names)
     * 
     * @param array $clients_with_lists
     * @return array
     */
    public function format_clients_for_dropdown( $clients_with_lists )
    {
        $options = array(
            array( 'value' => '', 'label' => '-' )
        );
        
        foreach( $clients_with_lists as $client_id => $client_data ) {
            $options[] = array(
                'value' => $client_id,
                'label' => $client_data['name']
            );
        }
        
        return $options;
    }

    /**
     * Get available clients from Campaign Monitor
     * 
     * @return array
     */
    public function get_clients()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API token first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_clients();
        
        if( ! $response->details ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading clients', 'triggerninja-for-ninja-forms' ) ) );
        }

        $clients = array_map( array( $this, 'format_client_options' ), $response->details );
        array_unshift( $clients, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify Campaign Monitor clients
         */
        return apply_filters( 'triggerninja_campaignmonitor_clients', $clients, $this );
    }

    /**
     * Get available lists from Campaign Monitor for a specific client
     * 
     * @param string $client_id Client ID
     * @return array
     */
    public function get_lists( $client_id = '' )
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API token first', 'triggerninja-for-ninja-forms' ) ) );
        }

        if( empty( $client_id ) ) {
            return array( array( 'value' => '', 'label' => __( 'Select a client first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_lists( $client_id );
        
        if( ! $response->details ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading lists', 'triggerninja-for-ninja-forms' ) ) );
        }

        $lists = array_map( array( $this, 'format_list_options' ), $response->details );
        array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify Campaign Monitor lists
         */
        return apply_filters( 'triggerninja_campaignmonitor_lists', $lists, $this );
    }
    
    /**
     * Format client options for dropdown
     * 
     * @param object $client
     * @return array
     */
    private function format_client_options( $client )
    {
        return array(
            'value' => $client->ClientID,
            'label' => $client->Name
        );
    }
    
    /**
     * Format list options for dropdown
     * 
     * @param object $list
     * @return array
     */
    private function format_list_options( $list )
    {
        return array(
            'value' => $list->ListID,
            'label' => $list->Name
        );
    }

    /**
     * Validate Campaign Monitor connection
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_campaignmonitor_api_token' ) ) && $this->api !== null;
    }

    /**
     * Test connection to Campaign Monitor
     * 
     * @return array
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return array(
                'success' => false,
                'message' => __( 'API token not configured', 'triggerninja-for-ninja-forms' )
            );
        }

        try {
            $response = $this->api->test_connection();
            
            if( $response->status >= 200 && $response->status < 300 && $response->details ) {
                $clients_count = is_array( $response->details ) ? count( $response->details ) : 0;
                return array(
                    'success' => true,
                    'message' => __( 'Successfully connected to Campaign Monitor', 'triggerninja-for-ninja-forms' ),
                    'data' => array(
                        'clients_count' => $clients_count
                    )
                );
            }

            return array(
                'success' => false,
                'message' => __( 'Invalid response from Campaign Monitor API', 'triggerninja-for-ninja-forms' )
            );

        } catch( Exception $e ) {
            return array(
                'success' => false,
                /* translators: %s is an error message. */
                'message' => sprintf( __( 'Connection failed: %s', 'triggerninja-for-ninja-forms' ), $e->getMessage() )
            );
        }
    }

    /**
     * Get platform-specific metabox data for submissions
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'contact_id' ] ) ){
            $label = __( 'Contact ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'contact_id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'first_name' ] ) && ! empty( $submission_data[ 'first_name' ] ) ){
            $label = __( 'First Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'first_name' ];
        }

        if( isset( $submission_data[ 'last_name' ] ) && ! empty( $submission_data[ 'last_name' ] ) ){
            $label = __( 'Last Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'last_name' ];
        }

        if( isset( $submission_data[ 'full_name' ] ) && ! empty( $submission_data[ 'full_name' ] ) ){
            $label = __( 'Full Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'full_name' ];
        }

        if( isset( $submission_data[ 'client_id' ] ) ){
            $label = __( 'Client ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'client_id' ];
        }

        if( isset( $submission_data[ 'list_id' ] ) ){
            $label = __( 'List ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'list_id' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }
        
        if( isset( $submission_data[ 'resubscribe' ] ) ){
            $label = __( 'Resubscribe', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'resubscribe' ] ? __( 'Enabled', 'triggerninja-for-ninja-forms' ) : __( 'Disabled', 'triggerninja-for-ninja-forms' );
        }

        if( isset( $submission_data[ 'custom_fields' ] ) && ! empty( $submission_data[ 'custom_fields' ] ) ){
            $label = __( 'Custom Fields', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = implode( ', ', array_map( function( $key, $value ) {
                return "{$key}: {$value}";
            }, array_keys( $submission_data[ 'custom_fields' ] ), $submission_data[ 'custom_fields' ] ) );
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }

}
