<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Campaign Monitor API Integration
 */
class TriggerNinja_CampaignMonitor_API
{
    /**
     * API Token
     * 
     * @var string
     */
    private $api_token;
    
    /**
     * Form ID for logging
     * 
     * @var int
     */
    private $form_id;
    
    /**
     * API Base URL
     * 
     * @var string
     */
    private $base_url = 'https://api.createsend.com/api/v3.3/';
    
    /**
     * Constructor
     * 
     * @param string $api_token Campaign Monitor API token
     * @param int $form_id Form ID for logging
     */
    public function __construct( $api_token, $form_id = 0 )
    {
        $this->api_token = $api_token;
        $this->form_id = $form_id;
    }
    
    /**
     * Make API request to Campaign Monitor using centralized HTTP client
     * 
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array $data Request data
     * @return TriggerNinja_Response
     */
    private function request( $endpoint, $method = 'GET', $data = array() )
    {
        $url = $this->base_url . $endpoint;
        
        $headers = array(
            'Authorization' => 'Basic ' . base64_encode( $this->api_token . ':x' )
        );
        
        $options = array(
            'timeout' => 30
        );
        
        $response = TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $this->form_id, 'campaignmonitor' );
        
        // Handle Campaign Monitor specific error format
        if ( $response->status >= 400 && $response->details ) {
            $error_message = 'Unknown error occurred';
            
            // Try to extract Campaign Monitor error message
            if ( is_object( $response->details ) && isset( $response->details->Message ) ) {
                $error_message = $response->details->Message;
            } elseif ( is_array( $response->details ) && isset( $response->details['Message'] ) ) {
                $error_message = $response->details['Message'];
            } elseif ( $response->error ) {
                $error_message = $response->error;
            }
            
            return new TriggerNinja_Response( $response->status, array(), $error_message );
        }
        
        return $response;
    }
    
    /**
     * Get all clients
     * 
     * @return TriggerNinja_Response
     */
    public function get_clients()
    {
        return $this->request( 'clients.json' );
    }
    
    /**
     * Get lists for a specific client
     * 
     * @param string $client_id Client ID
     * @return TriggerNinja_Response
     */
    public function get_lists( $client_id )
    {
        if ( empty( $client_id ) ) {
            return new TriggerNinja_Response( 400, array(), 'Client ID is required' );
        }
        
        return $this->request( "clients/{$client_id}/lists.json" );
    }
    
    /**
     * Subscribe a contact to a list
     * 
     * @param string $list_id List ID
     * @param string $email Email address
     * @param string $name Full name (optional)
     * @param array $custom_fields Custom fields (optional)
     * @param bool $resubscribe Whether to resubscribe existing subscribers
     * @return TriggerNinja_Response
     */
    public function subscribe_to_list( $list_id, $email, $name = '', $custom_fields = array(), $resubscribe = true )
    {
        if ( empty( $list_id ) || empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'List ID and email are required' );
        }
        
        $data = array(
            'EmailAddress' => $email,
            'Name' => $name,
            'ConsentToTrack' => 'Yes',
            'Resubscribe' => $resubscribe
        );
        
        // Add custom fields if provided
        if ( ! empty( $custom_fields ) ) {
            $data['CustomFields'] = array();
            foreach ( $custom_fields as $key => $value ) {
                $data['CustomFields'][] = array(
                    'Key' => $key,
                    'Value' => $value
                );
            }
        }
        
        return $this->request( "subscribers/{$list_id}.json", 'POST', $data );
    }
    
    /**
     * Unsubscribe a contact from a list
     * 
     * @param string $list_id List ID
     * @param string $email Email address
     * @return TriggerNinja_Response
     */
    public function unsubscribe_from_list( $list_id, $email )
    {
        if ( empty( $list_id ) || empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'List ID and email are required' );
        }
        
        $data = array(
            'EmailAddress' => $email
        );
        
        return $this->request( "subscribers/{$list_id}/unsubscribe.json", 'POST', $data );
    }
    
    /**
     * Get subscriber details
     * 
     * @param string $list_id List ID
     * @param string $email Email address
     * @return TriggerNinja_Response
     */
    public function get_subscriber( $list_id, $email )
    {
        if ( empty( $list_id ) || empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'List ID and email are required' );
        }
        
        $email_encoded = urlencode( $email );
        return $this->request( "subscribers/{$list_id}.json?email={$email_encoded}" );
    }
    
    /**
     * Test API connection by getting clients
     * 
     * @return TriggerNinja_Response
     */
    public function test_connection()
    {
        return $this->get_clients();
    }
    
    /**
     * Get list details including custom fields
     * 
     * @param string $list_id List ID
     * @return TriggerNinja_Response
     */
    public function get_list_details( $list_id )
    {
        if ( empty( $list_id ) ) {
            return new TriggerNinja_Response( 400, array(), 'List ID is required' );
        }
        
        return $this->request( "lists/{$list_id}.json" );
    }
    
    /**
     * Get custom fields for a list
     * 
     * @param string $list_id List ID
     * @return TriggerNinja_Response
     */
    public function get_list_custom_fields( $list_id )
    {
        if ( empty( $list_id ) ) {
            return new TriggerNinja_Response( 400, array(), 'List ID is required' );
        }
        
        return $this->request( "lists/{$list_id}/customfields.json" );
    }
}