<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Brevo Platform Integration
 */
class TriggerNinja_Brevo extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_brevo';
        $this->name = __( 'TN Brevo', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_key = $this->get_setting( 'triggerninja_brevo_api_key' );
        if( $api_key ) {
            $this->api = new TriggerNinja_Brevo_API( $api_key );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        return array(
            'triggerninja_brevo_api_key' => array(
                'id'    => 'triggerninja_brevo_api_key',
                'type'  => 'textbox',
                'label' => __( 'Brevo API Key', 'triggerninja-for-ninja-forms' ),
                'desc'  => sprintf(
                    /* translators: %1$s is the opening <a> tag, %2$s is the closing </a> tag. */
                    __( 'Get your %1$sAPI Key%2$s from Brevo: Profile > SMTP & API > API Keys tab.', 'triggerninja-for-ninja-forms' ),
                    '<a href="https://account.brevo.com/advanced/api" target="_blank">', '</a>'
                ),
            ),
        );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $fields = array(
            'list' => array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'Brevo List', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_lists(),
                'value' => ''
            )
        );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'list' ] ) ) {
            return $this->add_debug( $data, __( 'No Brevo list specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API key
        $api_key = $this->get_setting( 'triggerninja_brevo_api_key' );
        if( empty( $api_key ) ) {
            return $this->add_debug( $data, __( 'Brevo API key not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance with form_id for logging
        $api = new TriggerNinja_Brevo_API( $api_key, $form_id );

        // Base fields
        $contact_data = array(
            'email' => trim( $action_settings[ 'email_address' ] ),
            'attributes' => array(),
            'listIds' => array( intval( $action_settings[ 'list' ] ) ),
            'updateEnabled' => true
        );

        // Add first name and last name if provided
        if( ! empty( $action_settings[ 'first_name' ] ) ) {
            $first_name = trim( $action_settings[ 'first_name' ] );
            // Brevo supports multiple language variations
            $contact_data['attributes']['FIRSTNAME'] = $first_name;
            $contact_data['attributes']['VORNAME'] = $first_name;  // German
            $contact_data['attributes']['NOMBRE'] = $first_name;   // Spanish
            $contact_data['attributes']['NOME'] = $first_name;     // Italian/Portuguese
            $contact_data['attributes']['PRENOM'] = $first_name;   // French
        }

        if( ! empty( $action_settings[ 'last_name' ] ) ) {
            $last_name = trim( $action_settings[ 'last_name' ] );
            // Brevo supports multiple language variations
            $contact_data['attributes']['LASTNAME'] = $last_name;
            $contact_data['attributes']['NACHNAME'] = $last_name;   // German
            $contact_data['attributes']['APELLIDOS'] = $last_name;  // Spanish
            $contact_data['attributes']['COGNOME'] = $last_name;    // Italian
            $contact_data['attributes']['SOBRENOME'] = $last_name;  // Portuguese
            $contact_data['attributes']['NOM'] = $last_name;       // French
        }

        /**
         * Filter to process additional fields
         * 
         * @param array $contact_data Current contact data
         * @param array $action_settings Form action settings
         * @param int $form_id Form ID
         * @param array $data Form submission data
         * @return array Extended contact data
         */
        $contact_data = apply_filters( 'triggerninja_process_fields', $contact_data, $action_settings, $form_id, $data );
        $contact_data = apply_filters( "triggerninja_{$this->id}_process_fields", $contact_data, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         * 
         * @param array $contact_data Contact data to be sent
         * @param array $action_settings Form action settings
         * @param int $form_id Form ID
         * @param array $data Form submission data
         */
        do_action( 'triggerninja_before_api_request', $contact_data, $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", $contact_data, $action_settings, $form_id, $data );

        /**
         * Filter final API payload before sending
         * 
         * @param array $contact_data Contact data
         * @param string $endpoint API endpoint
         * @param array $action_settings Form action settings
         * @return array Modified contact data
         */
        $contact_data = apply_filters( 'triggerninja_api_payload', $contact_data, 'contacts', $action_settings );
        $contact_data = apply_filters( "triggerninja_{$this->id}_api_payload", $contact_data, 'contacts', $action_settings );

        $response = $api->create_contact( $contact_data );

        if( $response->details ){
            $response_data = get_object_vars( $response->details );
            
            /**
             * Filter response data before saving
             * 
             * @param array $response_data API response data
             * @param array $contact_data Original contact data sent
             * @param array $action_settings Form action settings
             * @return array Modified response data
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, $contact_data, $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, $contact_data, $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            $data = $this->add_debug( $data, __( 'Successfully added to Brevo list', 'triggerninja-for-ninja-forms' ), 'success' );
            
            /**
             * Hook after successful API request
             * 
             * @param array $response_data API response data
             * @param array $contact_data Contact data sent
             * @param array $action_settings Form action settings
             * @param int $form_id Form ID
             * @param array $data Form submission data
             */
            do_action( 'triggerninja_after_api_success', $response_data, $contact_data, $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, $contact_data, $action_settings, $form_id, $data );
            
        } else {
            $error_message = $response->error ? $response->error : __( 'Unknown error occurred', 'triggerninja-for-ninja-forms' );
            /* translators: %s is an error message. */
            $data = $this->add_debug( $data, sprintf( __( 'Brevo error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             * 
             * @param string $error_message Error message
             * @param array $contact_data Contact data sent
             * @param array $action_settings Form action settings
             * @param int $form_id Form ID
             * @param array $data Form submission data
             */
            do_action( 'triggerninja_after_api_error', $error_message, $contact_data, $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, $contact_data, $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available lists from Brevo
     * 
     * @return array
     */
    public function get_lists()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API key first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $lists = $this->api->get_all_lists();
        
        if( empty( $lists ) ) {
            return array( array( 'value' => '', 'label' => __( 'No lists found or API error', 'triggerninja-for-ninja-forms' ) ) );
        }

        $formatted_lists = array();
        foreach( $lists as $list_id => $list_name ) {
            $formatted_lists[] = array(
                'value' => $list_id,
                'label' => $list_name
            );
        }

        array_unshift( $formatted_lists, array( 'value' => '', 'label' => '-' ) );
        return $formatted_lists;
    }

    /**
     * Validate Brevo connection
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_brevo_api_key' ) ) && $this->api !== null;
    }

    /**
     * Get platform-specific metabox data for submissions
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        // Base metabox data
        $data = array();

        if( isset( $submission_data[ 'id' ] ) ){
            $label = __( 'Contact ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'createdAt' ] ) ){
            $label = __( 'Created At', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'createdAt' ];
        }

        if( isset( $submission_data[ 'modifiedAt' ] ) ){
            $label = __( 'Modified At', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'modifiedAt' ];
        }

        if( isset( $submission_data[ 'error' ] ) ){
            $label = __( 'Error', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = is_object( $submission_data[ 'error' ] ) ? $submission_data[ 'error' ]->message : $submission_data[ 'error' ];
        }

        /**
         * Filter metabox data display
         * 
         * @param array $data Current metabox data
         * @param array $submission_data Full submission data
         * @param string $platform_id Platform ID
         * @return array Extended metabox data
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}