<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * AWeber Platform Integration with OAuth2
 */
class TriggerNinja_Aweber extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_aweber';
        $this->name = __( 'TN AWeber', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $tokens = $this->get_tokens();

        // Check if tokens need to be refreshed
        if ( ! empty( $tokens['refresh_token'] ) && time() > $tokens['expires'] ) {
            $response = TriggerNinja_Aweber_API::refresh_access_token( $tokens['refresh_token'] );

            if ( $response->status === 200 && ! empty( $response->details->access_token ) ) {
                $new_tokens = array(
                    'access_token'  => $response->details->access_token,
                    'refresh_token' => $response->details->refresh_token,
                    'expires_in'    => $response->details->expires_in,
                );
                $this->save_tokens( $new_tokens );
                $tokens = $this->get_tokens(); // Get the updated tokens
            } else {
                // Refresh failed, clear tokens to force re-authentication
                $this->clear_tokens();
                return; // Stop initialization
            }
        }

        if ( $tokens && ! empty( $tokens['access_token'] ) ) {
            $this->api = new TriggerNinja_Aweber_API(
                $tokens['access_token'],
                $tokens['refresh_token'] ?? ''
            );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        // Generate PKCE parameters if they don't exist
        $pkce_data = $this->get_setting( 'triggerninja_aweber_pkce' );
        if( empty( $pkce_data ) ) {
            $pkce = TriggerNinja_Aweber_API::generate_pkce();
            $pkce_data = array(
                'code_verifier' => $pkce['code_verifier'],
                'code_challenge' => $pkce['code_challenge'],
                'nonce' => '' // Will be generated when needed
            );
            update_option( 'triggerninja_aweber_pkce', $pkce_data );
            $this->settings['triggerninja_aweber_pkce'] = $pkce_data; // Update local cache
        }
        
        // Ensure nonce is created when WordPress functions are available
        if( empty( $pkce_data['nonce'] ) && function_exists( 'wp_create_nonce' ) ) {
            $pkce_data['nonce'] = wp_create_nonce( 'triggerninja_aweber_auth' );
            update_option( 'triggerninja_aweber_pkce', $pkce_data );
            $this->settings['triggerninja_aweber_pkce'] = $pkce_data; // Update local cache
        }
        
        $auth_url = TriggerNinja_Aweber_API::get_authorization_url( 
            $pkce_data['code_challenge'], 
            ! empty( $pkce_data['nonce'] ) ? $pkce_data['nonce'] : 'temp_nonce'
        );
        
        $default_fields = array(
            'triggerninja_aweber_status' => array(
                'id'    => 'triggerninja_aweber_status',
                'type'  => 'html',
                'label' => __( 'Connection Status', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'aweber',
                'content' => $this->is_connected() ? 
                    '<span style="color: green;">' . __( 'Connected', 'triggerninja-for-ninja-forms' ) . '</span>' :
                    '<span style="color: red;">' . __( 'Not Connected', 'triggerninja-for-ninja-forms' ) . '</span>'
            ),
            'triggerninja_aweber_code_verifier' => array(
                'id'    => 'triggerninja_aweber_code_verifier',
                'type'  => 'hidden',
                'value' => $pkce_data['code_verifier']
            ),
            'triggerninja_aweber_code_challenge' => array(
                'id'    => 'triggerninja_aweber_code_challenge',
                'type'  => 'hidden',
                'value' => $pkce_data['code_challenge']
            ),
            'triggerninja_aweber_nonce' => array(
                'id'    => 'triggerninja_aweber_nonce',
                'type'  => 'hidden',
                'value' => ! empty( $pkce_data['nonce'] ) ? $pkce_data['nonce'] : ''
            ),
            'triggerninja_aweber_auth_code' => array(
                'id'    => 'triggerninja_aweber_auth_code',
                'type'  => 'textbox',
                'label' => __( 'AWeber Authorization Code', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'aweber',
                'value' => $this->get_setting( 'triggerninja_aweber_auth_code' ),
                'placeholder' => __( 'Enter your AWeber authorization code', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf(
                    /* translators: %s is a URL to the AWeber authorization page. */
                    __( 'Get your authorization code from <a href="%s" target="_blank" rel="noopener noreferrer">AWeber Authorization Page</a>', 'triggerninja-for-ninja-forms' ), 
                    esc_url( $auth_url )
                )
            ),
            'triggerninja_aweber_authorize_btn' => array(
                'id'    => 'triggerninja_aweber_authorize_btn',
                'type'  => 'button',
                'label' => '',
                'width' => 'full',
                'group' => 'aweber',
                'button_text' => __( 'Authorize', 'triggerninja-for-ninja-forms' ),
                'button_class' => 'button button-primary aweber-authorize-btn',
                'desc' => __( 'Click to complete the authorization process with your authorization code.', 'triggerninja-for-ninja-forms' )
            )
        );
        
        /**
         * Filter to modify AWeber settings fields
         */
        return apply_filters( 'triggerninja_aweber_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        // Get accounts with all their lists for dependency setup
        $accounts_with_lists = $this->get_accounts_with_lists();
        
        $default_fields = array(
            'account' => array(
                'name' => 'account',
                'type' => 'select',
                'label' => __( 'AWeber Account', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_accounts(),
                'value' => ''
            )
        );
        
        // Add a list field for each account
        foreach( $accounts_with_lists as $account_id => $account_data ) {
            if( empty( $account_id ) ) continue; // Skip empty account
            
            $default_fields["list_{$account_id}"] = array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'AWeber List', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $account_data['lists'],
                'value' => '',
                /* translators: %s is an account name. */
                'desc' => sprintf( __( 'Select a list from account %s.', 'triggerninja-for-ninja-forms' ), $account_data['name'] ),
                'deps' => array(
                    'account' => $account_id
                )
            );
        }
        
        /**
         * Filter to modify AWeber action fields
         */
        $fields = apply_filters( 'triggerninja_aweber_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'account' ] ) ) {
            return $this->add_debug( $data, __( 'No AWeber account specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Find the list ID from the account-specific list field
        $account_id = $action_settings[ 'account' ];
        $list_field_name = "list";
        $list_id = isset( $action_settings[ $list_field_name ] ) ? $action_settings[ $list_field_name ] : '';
        
        if( empty( $list_id ) ) {
            return $this->add_debug( $data, __( 'No AWeber list specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get tokens
        $tokens = $this->get_tokens();
        if( empty( $tokens['access_token'] ) ) {
            return $this->add_debug( $data, __( 'AWeber not connected - please authorize in platform settings', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance with form_id for logging
        $api = new TriggerNinja_Aweber_API( $tokens['access_token'], $tokens['refresh_token'] ?? '', $form_id );

        // Prepare subscriber data
        $email = trim( $action_settings[ 'email_address' ] );
        
        // Collect subscriber fields - filter out empty values
        $fields = array();
        
        if( isset( $action_settings[ 'first_name' ] ) && ! empty( trim( $action_settings[ 'first_name' ] ) ) ) {
            $fields['first_name'] = trim( $action_settings[ 'first_name' ] );
        }
        
        if( isset( $action_settings[ 'last_name' ] ) && ! empty( trim( $action_settings[ 'last_name' ] ) ) ) {
            $fields['last_name'] = trim( $action_settings[ 'last_name' ] );
        }

        /**
         * Filter to process additional fields
         */
        $fields = apply_filters( 'triggerninja_process_fields', $fields, $action_settings, $form_id, $data );
        $fields = apply_filters( "triggerninja_{$this->id}_process_fields", $fields, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );

        // Check if contact exists first
        $check_response = $api->check_contact_exists( $email, $account_id, $list_id );
        $subscriber_id = null;
        
        if( $check_response->status >= 200 && $check_response->status < 300 && $check_response->details ) {
            if( isset( $check_response->details->entries ) && is_array( $check_response->details->entries ) && !empty( $check_response->details->entries ) ) {
                $subscriber_id = $check_response->details->entries[0]->id ?? null;
            }
        }

        // Create or update contact
        if( $subscriber_id ) {
            // Update existing contact
            $response = $api->update_contact( $email, $fields, $account_id, $list_id, $subscriber_id );
            $action = 'updated';
        } else {
            // Create new contact
            $response = $api->create_contact( $email, $fields, $account_id, $list_id );
            $action = 'created';
        }

        // Check for successful subscription
        if( $response->status >= 200 && $response->status < 300 ) {
            $response_details = $response->details;
            $final_subscriber_id = $subscriber_id ?: ( is_object( $response_details ) && isset( $response_details->id ) ? $response_details->id : 'subscribed' );
            
            $response_data = array(
                'subscriber_id' => $final_subscriber_id,
                'email' => $email,
                'fields' => $fields,
                'account_id' => $account_id,
                'list_id' => $list_id,
                'action' => $action,
                'status' => 'subscribed'
            );

            /**
             * Filter response data before saving
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            /* translators: %s is the action (e.g., "created", "updated"). */
            $data = $this->add_debug( $data, sprintf( __( 'Successfully %s AWeber subscriber', 'triggerninja-for-ninja-forms' ), $action ), 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
            
        } else {
            $error_message = $response->error ? $response->error : __( 'Failed to subscribe to AWeber list', 'triggerninja-for-ninja-forms' );
            /* translators: %s is an error message. */
            $data = $this->add_debug( $data, sprintf( __( 'AWeber error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, array( 'email' => $email, 'fields' => $fields ), $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available accounts from AWeber
     * 
     * @return array
     */
    public function get_accounts()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please connect AWeber first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_accounts();
        
        if( ! $response->details || ! isset( $response->details->entries ) || ! is_array( $response->details->entries ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading accounts', 'triggerninja-for-ninja-forms' ) ) );
        }

        $accounts = array();
        foreach( $response->details->entries as $account ) {
            if( isset( $account->id ) ) {
                $accounts[] = array(
                    'value' => $account->id,
                    'label' => $account->id
                );
            }
        }
        
        array_unshift( $accounts, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify AWeber accounts
         */
        return apply_filters( 'triggerninja_aweber_accounts', $accounts, $this );
    }
    
    /**
     * Get accounts with their lists for dependency setup
     * 
     * @return array
     */
    private function get_accounts_with_lists()
    {
        if( ! $this->is_connected() ) {
            return array();
        }

        $accounts_response = $this->api->get_accounts();
        
        if( ! $accounts_response->details || ! isset( $accounts_response->details->entries ) || ! is_array( $accounts_response->details->entries ) ) {
            return array();
        }

        $accounts_with_lists = array();
        
        foreach( $accounts_response->details->entries as $account ) {
            if( isset( $account->id ) ) {
                $account_id = $account->id;
                $account_name = $account_id; // AWeber accounts typically just use ID as name
                
                // Get lists for this account
                $lists_response = $this->api->get_lists( $account_id );
                $lists = array();
                
                if( $lists_response->status >= 200 && $lists_response->status < 300 && 
                    $lists_response->details && isset( $lists_response->details->entries ) && 
                    is_array( $lists_response->details->entries ) ) {
                    
                    foreach( $lists_response->details->entries as $list ) {
                        if( isset( $list->id ) && isset( $list->name ) ) {
                            $lists[] = array(
                                'value' => $list->id,
                                'label' => $list->name
                            );
                        }
                    }
                }
                
                // Add empty option at the beginning
                array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );
                
                $accounts_with_lists[ $account_id ] = array(
                    'name' => $account_name,
                    'lists' => $lists
                );
            }
        }
        
        /**
         * Filter to modify AWeber accounts with lists
         */
        return apply_filters( 'triggerninja_aweber_accounts_with_lists', $accounts_with_lists, $this );
    }
    
    /**
     * Get available lists for an account from AWeber
     * 
     * @param string $account_id Account ID
     * @return array
     */
    public function get_aweber_lists( $account_id = '' )
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please connect AWeber first', 'triggerninja-for-ninja-forms' ) ) );
        }

        if( empty( $account_id ) ) {
            return array( array( 'value' => '', 'label' => __( 'Select account first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_lists( $account_id );
        
        if( ! $response->details || ! isset( $response->details->entries ) || ! is_array( $response->details->entries ) ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading lists', 'triggerninja-for-ninja-forms' ) ) );
        }

        $lists = array();
        foreach( $response->details->entries as $list ) {
            if( isset( $list->id ) && isset( $list->name ) ) {
                $lists[] = array(
                    'value' => $list->id,
                    'label' => $list->name
                );
            }
        }
        
        array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify AWeber lists
         */
        return apply_filters( 'triggerninja_aweber_lists', $lists, $this );
    }
    
    /**
     * Check if platform is connected (has valid tokens)
     * 
     * @return bool
     */
    public function is_connected()
    {
        $tokens = $this->get_tokens();
        return !empty( $tokens['access_token'] ) && $this->api !== null;
    }
    
    /**
     * Test API connection
     * 
     * @return bool
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return false;
        }
        
        $response = $this->api->test_connection();
        return $response->status >= 200 && $response->status < 300;
    }
    
    /**
     * Get stored tokens
     * 
     * @return array
     */
    private function get_tokens()
    {
        $access_token = get_option( 'triggerninja_aweber_access_token', '' );
        $refresh_token = get_option( 'triggerninja_aweber_refresh_token', '' );
        $expires = get_option( 'triggerninja_aweber_token_expires', 0 );
        
        return array(
            'access_token' => $access_token,
            'refresh_token' => $refresh_token,
            'expires' => $expires
        );
    }
    
    /**
     * Save tokens
     * 
     * @param array $tokens
     */
    public function save_tokens( $tokens )
    {
        if ( isset( $tokens['access_token'] ) ) {
            update_option( 'triggerninja_aweber_access_token', $tokens['access_token'] );
        }
        if ( isset( $tokens['refresh_token'] ) ) {
            update_option( 'triggerninja_aweber_refresh_token', $tokens['refresh_token'] );
        }
        if ( isset( $tokens['expires_in'] ) ) {
            // Add a small buffer (e.g., 60 seconds) to the expiration time
            $expires_time = time() + $tokens['expires_in'] - 60;
            update_option( 'triggerninja_aweber_token_expires', $expires_time );
        }

        // Reinitialize API with new tokens
        $this->init_api();
    }
    
    /**
     * Clear stored tokens
     */
    public function clear_tokens()
    {
        delete_option( 'triggerninja_aweber_tokens' );
        $this->api = null;
    }
    
    /**
     * Get authorization interface
     * 
     * @return string
     */
    public function get_authorization_interface()
    {
        return $this->get_authorization_html_safe();
    }
    
    /**
     * Get authorization HTML for settings (safe version)
     * 
     * @return string
     */
    private function get_authorization_html_safe()
    {
        // During admin interface rendering, WordPress functions should be available
        // Only return placeholder during very early plugin initialization
        if ( ! function_exists( 'wp_create_nonce' ) || ! function_exists( 'admin_url' ) ) {
            // Return a simple interface without JavaScript for early init
            return $this->get_simple_authorization_html();
        }
        
        return $this->get_authorization_html();
    }
    
    /**
     * Get simple authorization HTML without WordPress functions
     * 
     * @return string
     */
    private function get_simple_authorization_html()
    {
        $is_connected = $this->is_connected();
        
        if( $is_connected ) {
            return '<p style="color: #46b450; font-weight: bold;">' . __( 'AWeber is connected and ready to use!', 'triggerninja-for-ninja-forms' ) . '</p>';
        } else {
            $html = '<div style="margin-bottom: 20px;">';
            $html .= '<p style="margin-bottom: 15px;">' . __( 'To connect AWeber, please follow these steps:', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '<div style="background: #f9f9f9; padding: 15px; border: 1px solid #ddd; border-radius: 4px; margin-bottom: 15px;">';
            $html .= '<p style="margin: 0 0 10px 0;"><strong>' . __( 'Step 1:', 'triggerninja-for-ninja-forms' ) . '</strong> ' . __( 'Click the link below to authorize TriggerNinja:', 'triggerninja-for-ninja-forms' ) . '</p>';
            
            // Generate auth URL without WordPress functions
            $client_id = TriggerNinja_Aweber_API::CLIENT_ID;
            $redirect_uri = TriggerNinja_Aweber_API::REDIRECT_URI;
            $scope = TriggerNinja_Aweber_API::SCOPE;
            $simple_auth_url = TriggerNinja_Aweber_API::AUTHORIZATION_URL . '?' . http_build_query(array(
                'response_type' => 'code',
                'client_id' => $client_id,
                'redirect_uri' => $redirect_uri,
                'scope' => $scope
            ));
            
            $html .= '<p style="margin: 0 0 15px 0;"><a href="' . esc_url( $simple_auth_url ) . '" target="_blank" class="button button-primary">' . __( 'Authorize with AWeber', 'triggerninja-for-ninja-forms' ) . '</a></p>';
            $html .= '<p style="margin: 0; font-size: 12px; color: #666;">' . __( 'This will open in a new window. After authorizing, copy the authorization code.', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '</div>';
            
            $html .= '<div style="margin-bottom: 15px;">';
            $html .= '<p style="margin: 0 0 8px 0;"><strong>' . __( 'Step 2:', 'triggerninja-for-ninja-forms' ) . '</strong> ' . __( 'Paste the authorization code here:', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '<input type="text" id="aweber-auth-code-simple" placeholder="' . esc_attr__( 'Enter authorization code', 'triggerninja-for-ninja-forms' ) . '" style="width: 300px; margin-right: 10px;" />';
            $html .= '<button type="button" class="button button-primary" disabled="disabled">' . __( 'Save & Connect', 'triggerninja-for-ninja-forms' ) . '</button>';
            $html .= '</div>';
            
            $html .= '<p class="description">' . __( 'Please refresh the page once WordPress has fully loaded to authorize AWeber.', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '</div>';
            
            return $html;
        }
    }
    
    /**
     * Get authorization HTML for settings
     * 
     * @return string
     */
    private function get_authorization_html()
    {
        $is_connected = $this->is_connected();
        $nonce = wp_create_nonce( 'triggerninja_aweber_auth' );

        if( $is_connected ) {
            $html = '<div style="margin-bottom: 15px;">';
            $html .= '<p style="color: #46b450; margin-bottom: 10px;">' . __( 'AWeber is connected and ready to use!', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '<button type="button" class="button button-secondary" id="aweber-disconnect">' . __( 'Disconnect AWeber', 'triggerninja-for-ninja-forms' ) . '</button>';
            $html .= '</div>';
        } else {
            // Generate PKCE codes for the authorization URL
            $pkce = TriggerNinja_Aweber_API::generate_pkce();
            $auth_url = TriggerNinja_Aweber_API::get_authorization_url( $pkce['code_challenge'], $nonce );
            
            $html = '<div style="margin-bottom: 20px;">';
            $html .= '<p style="margin-bottom: 15px;">' . __( 'To connect AWeber, please follow these steps:', 'triggerninja-for-ninja-forms' ) . '</p>';
            
            $html .= '<div style="background: #f9f9f9; padding: 15px; border: 1px solid #ddd; border-radius: 4px; margin-bottom: 15px;">';
            $html .= '<p style="margin: 0 0 10px 0;"><strong>' . __( 'Step 1:', 'triggerninja-for-ninja-forms' ) . '</strong> ' . __( 'Click the link below to authorize TriggerNinja:', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '<p style="margin: 0 0 15px 0;"><a href="' . esc_url( $auth_url ) . '" target="_blank" class="button button-primary">' . __( 'Authorize with AWeber', 'triggerninja-for-ninja-forms' ) . '</a></p>';
            $html .= '<p style="margin: 0; font-size: 12px; color: #666;">' . __( 'This will open in a new window. After authorizing, copy the authorization code.', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '</div>';
            
            $html .= '<div style="margin-bottom: 15px;">';
            $html .= '<p style="margin: 0 0 8px 0;"><strong>' . __( 'Step 2:', 'triggerninja-for-ninja-forms' ) . '</strong> ' . __( 'Paste the authorization code here:', 'triggerninja-for-ninja-forms' ) . '</p>';
            $html .= '<input type="text" id="aweber-auth-code" placeholder="' . esc_attr__( 'Enter authorization code', 'triggerninja-for-ninja-forms' ) . '" style="width: 300px; margin-right: 10px;" />';
            $html .= '<button type="button" class="button button-primary" id="aweber-save-code">' . __( 'Save & Connect', 'triggerninja-for-ninja-forms' ) . '</button>';
            $html .= '</div>';
            
            // Store PKCE data in hidden inputs for JavaScript to access
            $html .= '<input type="hidden" id="aweber-code-verifier" value="' . esc_attr( $pkce['code_verifier'] ) . '" />';
            $html .= '<input type="hidden" id="aweber-code-challenge" value="' . esc_attr( $pkce['code_challenge'] ) . '" />';
            $html .= '<input type="hidden" id="aweber-auth-nonce" value="' . esc_attr( $nonce ) . '" />';
            $html .= '</div>';
        }

        if ( $is_connected ) {
            $html .= '<input type="hidden" id="aweber-auth-nonce" value="' . esc_attr( $nonce ) . '" />';
        }

        return $html;
    }
    
    /**
     * Get available lists from AWeber (alias for get_aweber_lists)
     * Required by TriggerNinja_Platform_Interface
     * 
     * @return array
     */
    public function get_lists()
    {
        // For AWeber, we need an account ID to get lists
        // Return empty array - lists will be loaded dynamically based on account selection
        return array( array( 'value' => '', 'label' => __( 'Select account first', 'triggerninja-for-ninja-forms' ) ) );
    }
    
    /**
     * Get platform-specific metabox data for submissions
     * Required by TriggerNinja_Platform_Interface
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'subscriber_id' ] ) ){
            $label = __( 'Subscriber ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'subscriber_id' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'account_id' ] ) ){
            $label = __( 'Account ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'account_id' ];
        }

        if( isset( $submission_data[ 'list_id' ] ) ){
            $label = __( 'List ID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'list_id' ];
        }

        if( isset( $submission_data[ 'action' ] ) ){
            $label = __( 'Action', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = ucfirst( $submission_data[ 'action' ] );
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }
        
        if( isset( $submission_data[ 'fields' ] ) && ! empty( $submission_data[ 'fields' ] ) ){
            $label = __( 'Subscriber Fields', 'triggerninja-for-ninja-forms' );
            $field_info = array();
            foreach( $submission_data[ 'fields' ] as $key => $value ) {
                if( ! empty( $value ) ) {
                    $field_info[] = "{$key}: {$value}";
                }
            }
            $data[ $label ] = implode( ', ', $field_info );
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}
