<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * AWeber API Integration with OAuth2 and PKCE
 */
class TriggerNinja_Aweber_API
{
    const API_URL = 'https://api.aweber.com/1.0/';
    const AUTHORIZATION_URL = 'https://auth.aweber.com/oauth2/authorize';
    const TOKEN_URL = 'https://auth.aweber.com/oauth2/token';
    const CLIENT_ID = 'wG9E9E4PVpfA0ax93gvmlsUIhWrpH00U';
    const REDIRECT_URI = 'urn:ietf:wg:oauth:2.0:oob';
    const SCOPE = 'subscriber.write subscriber.read account.read list.read';
    
    /**
     * Form ID for logging
     * 
     * @var int
     */
    private $form_id;
    
    /**
     * Access token
     * 
     * @var string
     */
    private $access_token;
    
    /**
     * Refresh token
     * 
     * @var string
     */
    private $refresh_token;
    
    /**
     * Constructor
     * 
     * @param string $access_token Access token
     * @param string $refresh_token Refresh token
     * @param int $form_id Form ID for logging
     */
    public function __construct( $access_token = '', $refresh_token = '', $form_id = 0 )
    {
        $this->access_token = $access_token;
        $this->refresh_token = $refresh_token;
        $this->form_id = $form_id;
    }
    
    /**
     * Generate PKCE code verifier and challenge
     * 
     * @return array
     */
    public static function generate_pkce()
    {
        $verifier_bytes = random_bytes(64);
        $code_verifier = rtrim(strtr(base64_encode($verifier_bytes), "+/", "-_"), "=");
        $challenge_bytes = hash("sha256", $code_verifier, true);
        $code_challenge = rtrim(strtr(base64_encode($challenge_bytes), "+/", "-_"), "=");
        
        return array(
            'code_verifier' => $code_verifier,
            'code_challenge' => $code_challenge
        );
    }
    
    /**
     * Get authorization URL for OAuth2 flow
     * 
     * @param string $code_challenge PKCE code challenge
     * @param string $nonce Security nonce
     * @return string
     */
    public static function get_authorization_url( $code_challenge, $nonce )
    {
        $params = array(
            'response_type' => 'code',
            'client_id' => self::CLIENT_ID,
            'redirect_uri' => self::REDIRECT_URI,
            'scope' => self::SCOPE,
            'state' => $nonce,
            'code_challenge' => $code_challenge,
            'code_challenge_method' => 'S256'
        );
        
        return self::AUTHORIZATION_URL . '?' . http_build_query( $params );
    }
    
    /**
     * Exchange authorization code for tokens
     * 
     * @param string $auth_code Authorization code
     * @param string $code_verifier PKCE code verifier
     * @return TriggerNinja_Response
     */
    public static function exchange_code_for_tokens( $auth_code, $code_verifier )
    {
        $data = array(
            'code' => $auth_code,
            'client_id' => self::CLIENT_ID,
            'redirect_uri' => self::REDIRECT_URI,
            'grant_type' => 'authorization_code',
            'code_verifier' => $code_verifier
        );
        $headers = array(
            'Content-Type' => 'application/x-www-form-urlencoded'
        );
        return TriggerNinja_Http_Client::post( self::TOKEN_URL, $data, $headers, array(), null, 'aweber' );
    }
    
    /**
     * Refresh access token using refresh token
     * 
     * @param string $refresh_token Refresh token
     * @return TriggerNinja_Response
     */
    public static function refresh_access_token( $refresh_token )
    {
        $data = array(
            'refresh_token' => $refresh_token,
            'grant_type' => 'refresh_token',
            'client_id' => self::CLIENT_ID
        );
        
        $headers = array(
            'Content-Type' => 'application/x-www-form-urlencoded'
        );
        
        return TriggerNinja_Http_Client::post( self::TOKEN_URL, $data, $headers, array(), null, 'aweber' );
    }
    
    /**
     * Make authenticated API request to AWeber
     * 
     * @param string $endpoint API endpoint (without base URL)
     * @param string $method HTTP method
     * @param array $data Request data
     * @return TriggerNinja_Response
     */
    private function request( $endpoint, $method = 'GET', $data = array() )
    {
        if( empty( $this->access_token ) ) {
            return new TriggerNinja_Response( 401, array(), 'No access token available' );
        }
        
        $url = self::API_URL . $endpoint;
        
        $headers = array(
            'Accept' => 'application/json',
            'Content-Type' => 'application/json; charset=utf-8',
            'Authorization' => 'Bearer ' . $this->access_token
        );
        
        $options = array(
            'timeout' => 30
        );
        
        $response = TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $this->form_id, 'aweber' );
        
        // Handle AWeber specific error format
        if ( $response->status >= 400 && $response->details ) {
            $error_message = 'Unknown error occurred';
            
            // Try to extract AWeber error message
            if ( is_object( $response->details ) && isset( $response->details->error ) ) {
                if ( isset( $response->details->error->message ) ) {
                    $error_message = $response->details->error->message;
                } elseif ( is_string( $response->details->error ) ) {
                    $error_message = $response->details->error;
                }
            } elseif ( is_array( $response->details ) && isset( $response->details['error'] ) ) {
                if ( isset( $response->details['error']['message'] ) ) {
                    $error_message = $response->details['error']['message'];
                } elseif ( is_string( $response->details['error'] ) ) {
                    $error_message = $response->details['error'];
                }
            } elseif ( $response->error ) {
                $error_message = $response->error;
            }
            
            return new TriggerNinja_Response( $response->status, array(), $error_message );
        }
        
        return $response;
    }
    
    /**
     * Get AWeber accounts
     * 
     * @return TriggerNinja_Response
     */
    public function get_accounts()
    {
        return $this->request( 'accounts' );
    }
    
    /**
     * Get lists for a specific account
     * 
     * @param string $account_id Account ID
     * @return TriggerNinja_Response
     */
    public function get_lists( $account_id )
    {
        if( empty( $account_id ) ) {
            return new TriggerNinja_Response( 400, array(), 'Account ID is required' );
        }
        
        $endpoint = "accounts/{$account_id}/lists";
        $all_lists = array();
        
        do {
            $response = $this->request( $endpoint );
            
            if( $response->status >= 400 ) {
                return $response; // Return error response
            }
            
            if( $response->details && isset( $response->details->entries ) && is_array( $response->details->entries ) ) {
                $all_lists = array_merge( $all_lists, $response->details->entries );
            }
            
            // Check for pagination
            $endpoint = '';
            if( $response->details && isset( $response->details->next_collection_link ) ) {
                // Extract just the endpoint part from the full URL
                $next_url = $response->details->next_collection_link;
                if( strpos( $next_url, self::API_URL ) === 0 ) {
                    $endpoint = substr( $next_url, strlen( self::API_URL ) );
                }
            }
            
        } while( !empty( $endpoint ) );
        
        // Return all lists in the same format as a single response
        return new TriggerNinja_Response( 200, (object) array( 'entries' => $all_lists ) );
    }
    
    /**
     * Check if contact exists in a list
     * 
     * @param string $email Email address
     * @param string $account_id Account ID
     * @param string $list_id List ID
     * @return TriggerNinja_Response
     */
    public function check_contact_exists( $email, $account_id, $list_id )
    {
        if( empty( $email ) || empty( $account_id ) || empty( $list_id ) ) {
            return new TriggerNinja_Response( 400, array(), 'Email, account ID, and list ID are required' );
        }
        
        $endpoint = "accounts/{$account_id}/lists/{$list_id}/subscribers?" . http_build_query( array(
            'ws.op' => 'find',
            'email' => $email
        ) );
        
        return $this->request( $endpoint );
    }
    
    /**
     * Create/subscribe contact to list
     * 
     * @param string $email Email address
     * @param array $fields Additional fields (first_name, last_name, etc.)
     * @param string $account_id Account ID
     * @param string $list_id List ID
     * @return TriggerNinja_Response
     */
    public function create_contact( $email, $fields, $account_id, $list_id )
    {
        if( empty( $email ) || empty( $account_id ) || empty( $list_id ) ) {
            return new TriggerNinja_Response( 400, array(), 'Email, account ID, and list ID are required' );
        }
        
        $contact_data = array(
            'email' => $email
        );
        
        // Combine first and last name into a single name field
        $name_parts = array();
        if( !empty( $fields['first_name'] ) ) {
            $name_parts[] = $fields['first_name'];
        }
        if( !empty( $fields['last_name'] ) ) {
            $name_parts[] = $fields['last_name'];
        }
        if( !empty( $name_parts ) ) {
            $contact_data['name'] = implode( ' ', $name_parts );
        }
        
        $endpoint = "accounts/{$account_id}/lists/{$list_id}/subscribers";
        return $this->request( $endpoint, 'POST', $contact_data );
    }
    
    /**
     * Update existing contact in list
     * 
     * @param string $email Email address
     * @param array $fields Additional fields (first_name, last_name, etc.)
     * @param string $account_id Account ID
     * @param string $list_id List ID
     * @param string $subscriber_id Subscriber ID
     * @return TriggerNinja_Response
     */
    public function update_contact( $email, $fields, $account_id, $list_id, $subscriber_id )
    {
        if( empty( $account_id ) || empty( $list_id ) || empty( $subscriber_id ) ) {
            return new TriggerNinja_Response( 400, array(), 'Account ID, list ID, and subscriber ID are required' );
        }
        
        $contact_data = array();
        
        if( !empty( $email ) ) {
            $contact_data['email'] = $email;
        }
        
        // Combine first and last name into a single name field
        $name_parts = array();
        if( !empty( $fields['first_name'] ) ) {
            $name_parts[] = $fields['first_name'];
        }
        if( !empty( $fields['last_name'] ) ) {
            $name_parts[] = $fields['last_name'];
        }
        if( !empty( $name_parts ) ) {
            $contact_data['name'] = implode( ' ', $name_parts );
        }
        
        $endpoint = "accounts/{$account_id}/lists/{$list_id}/subscribers/{$subscriber_id}";
        return $this->request( $endpoint, 'PATCH', $contact_data );
    }
    
    /**
     * Test API connection
     * 
     * @return TriggerNinja_Response
     */
    public function test_connection()
    {
        return $this->get_accounts();
    }
}