<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Acelle Mail Platform Integration
 */
class TriggerNinja_Acelle extends TriggerNinja_Platform_Abstract
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->id = 'triggerninja_acelle';
        $this->name = __( 'TN Acelle Mail', 'triggerninja-for-ninja-forms' );
        
        parent::__construct();
    }

    /**
     * Initialize API connection
     */
    protected function init_api()
    {
        $api_endpoint = $this->get_setting( 'triggerninja_acelle_api_endpoint' );
        $api_token = $this->get_setting( 'triggerninja_acelle_api_token' );
        
        if( $api_endpoint && $api_token ) {
            $this->api = new TriggerNinja_Acelle_API( $api_endpoint, $api_token );
        }
    }

    /**
     * Get platform settings fields configuration
     * 
     * @return array
     */
    public function get_settings_fields()
    {
        $default_fields = array(
            'triggerninja_acelle_api_endpoint' => array(
                'id'    => 'triggerninja_acelle_api_endpoint',
                'type'  => 'textbox',
                'label' => __( 'Acelle Mail API Endpoint', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'acelle',
                'value' => $this->get_setting( 'triggerninja_acelle_api_endpoint' ),
                'placeholder' => __( 'https://your-acelle-domain.com/api/v1', 'triggerninja-for-ninja-forms' ),
                'desc' => __( 'Enter your Acelle Mail API endpoint URL. Usually in the format: https://your-domain.com/api/v1', 'triggerninja-for-ninja-forms' )
            ),
            'triggerninja_acelle_api_token' => array(
                'id'    => 'triggerninja_acelle_api_token',
                'type'  => 'textbox',
                'label' => __( 'Acelle Mail API Token', 'triggerninja-for-ninja-forms' ),
                'width' => 'full',
                'group' => 'acelle',
                'value' => $this->get_setting( 'triggerninja_acelle_api_token' ),
                'placeholder' => __( 'Enter your Acelle Mail API Token', 'triggerninja-for-ninja-forms' ),
                'desc' => sprintf( 
                    __( 'Go to My Profile > Account > API Token in your Acelle Mail dashboard to get your API token.', 'triggerninja-for-ninja-forms' )
                )
            )
        );
        
        /**
         * Filter to modify Acelle Mail settings fields
         */
        return apply_filters( 'triggerninja_acelle_settings_fields', $default_fields, $this );
    }

    /**
     * Get action fields configuration for forms
     * 
     * @return array
     */
    public function get_action_fields()
    {
        $default_fields = array(
            'list' => array(
                'name' => 'list',
                'type' => 'select',
                'label' => __( 'Acelle Mail List', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'options' => $this->get_lists(),
                'value' => '',
                'desc' => __( 'Select the Acelle Mail list to subscribe contacts to.', 'triggerninja-for-ninja-forms' )
            )
        );
        
        /**
         * Filter to modify Acelle Mail action fields
         */
        $fields = apply_filters( 'triggerninja_acelle_action_fields', $default_fields, $this );

        return array_merge( $fields, $this->get_common_fields() );
    }

    /**
     * Process form submission data
     * 
     * @param array $action_settings
     * @param int $form_id
     * @param array $data
     * @return array
     */
    public function process_submission( $action_settings, $form_id, $data )
    {
        if( empty( $action_settings[ 'list' ] ) ) {
            return $this->add_debug( $data, __( 'No Acelle Mail list specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        if( empty( $action_settings[ 'email_address' ] ) ) {
            return $this->add_debug( $data, __( 'No email address specified', 'triggerninja-for-ninja-forms' ), 'error' );
        }

        // Get API credentials (with extensibility hook for advanced users)
        $api_endpoint = $this->get_setting( 'triggerninja_acelle_api_endpoint' );
        $api_token = $this->get_setting( 'triggerninja_acelle_api_token' );
        
        /**
         * Filter to modify Acelle Mail API credentials for advanced setups
         */
        $api_endpoint = apply_filters( 'triggerninja_acelle_api_endpoint', $api_endpoint, $action_settings, $this );
        $api_token = apply_filters( 'triggerninja_acelle_api_token', $api_token, $action_settings, $this );
        
        if( empty( $api_endpoint ) || empty( $api_token ) ) {
            return $this->add_debug( $data, __( 'Acelle Mail API credentials not configured', 'triggerninja-for-ninja-forms' ), 'error' );
        }
        
        // Create API instance
        $api = new TriggerNinja_Acelle_API( $api_endpoint, $api_token, $form_id );

        // Prepare profile data
        $email = trim( $action_settings[ 'email_address' ] );
        $first_name = isset( $action_settings[ 'first_name' ] ) ? trim( $action_settings[ 'first_name' ] ) : '';
        $last_name = isset( $action_settings[ 'last_name' ] ) ? trim( $action_settings[ 'last_name' ] ) : '';
        
        // Prepare custom fields for Acelle Mail
        $custom_fields = array();
        
        /**
         * Filter to process additional fields
         */
        $custom_fields = apply_filters( 'triggerninja_process_fields', $custom_fields, $action_settings, $form_id, $data );
        $custom_fields = apply_filters( "triggerninja_{$this->id}_process_fields", $custom_fields, $action_settings, $form_id, $data );

        /**
         * Hook for pre-submission validation
         */
        do_action( 'triggerninja_before_api_request', compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
        do_action( "triggerninja_{$this->id}_before_api_request", compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );

        // Use the Acelle Mail API to subscribe
        $subscription_response = $api->subscribe_to_list( 
            $action_settings[ 'list' ], 
            $email, 
            $first_name,
            $last_name,
            $custom_fields
        );

        // Check for successful subscription
        if( $subscription_response->status >= 200 && $subscription_response->status < 300 ) {
            $response_details = $subscription_response->details;
            $subscriber_uid = is_object( $response_details ) && isset( $response_details->uid ) ? $response_details->uid : 'subscribed';
            
            $response_data = array(
                'subscriber_uid' => $subscriber_uid,
                'email' => $email,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'list_uid' => $action_settings[ 'list' ],
                'status' => 'subscribed',
                'custom_fields' => $custom_fields
            );

            /**
             * Filter response data before saving (for  extensions)
             */
            $response_data = apply_filters( 'triggerninja_api_response_data', $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings );
            $response_data = apply_filters( "triggerninja_{$this->id}_api_response_data", $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings );
            
            $data = $this->save_submission_data( $data, $response_data );
            $data = $this->add_debug( $data, __( 'Successfully subscribed to Acelle Mail list', 'triggerninja-for-ninja-forms' ), 'success' );
            
            /**
             * Hook after successful API request
             */
            do_action( 'triggerninja_after_api_success', $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_success", $response_data, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            
        } else {
            $error_message = $subscription_response->error ? $subscription_response->error : __( 'Failed to subscribe to Acelle Mail list', 'triggerninja-for-ninja-forms' );
            /* translators: %s is the error message returned by the API. */
            $data = $this->add_debug( $data, sprintf( __( 'Acelle Mail error: %s', 'triggerninja-for-ninja-forms' ), $error_message ), 'error' );
            
            /**
             * Hook after API error
             */
            do_action( 'triggerninja_after_api_error', $error_message, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
            do_action( "triggerninja_{$this->id}_after_api_error", $error_message, compact( 'email', 'first_name', 'last_name' ), $action_settings, $form_id, $data );
        }

        return $data;
    }

    /**
     * Get available lists from Acelle Mail
     * 
     * @return array
     */
    public function get_lists()
    {
        if( ! $this->is_connected() ) {
            return array( array( 'value' => '', 'label' => __( 'Please configure API credentials first', 'triggerninja-for-ninja-forms' ) ) );
        }

        $response = $this->api->get_lists();
        
        if( ! $response->details ) {
            return array( array( 'value' => '', 'label' => __( 'Error loading lists', 'triggerninja-for-ninja-forms' ) ) );
        }

        $lists = array();
        
        // Handle different response formats
        if( is_array( $response->details ) ) {
            foreach( $response->details as $list ) {
                if( is_object( $list ) && isset( $list->uid ) && isset( $list->name ) ) {
                    $lists[] = array(
                        'value' => $list->uid,
                        'label' => $list->name
                    );
                }
            }
        }
        
        // Add default option at the beginning
        array_unshift( $lists, array( 'value' => '', 'label' => '-' ) );
        
        /**
         * Filter to modify Acelle Mail lists
         */
        return apply_filters( 'triggerninja_acelle_lists', $lists, $this );
    }
    
    /**
     * Validate Acelle Mail connection
     * 
     * @return bool
     */
    public function is_connected()
    {
        return ! empty( $this->get_setting( 'triggerninja_acelle_api_endpoint' ) ) && 
               ! empty( $this->get_setting( 'triggerninja_acelle_api_token' ) ) && 
               $this->api !== null;
    }

    /**
     * Test connection to Acelle Mail
     * 
     * @return array
     */
    public function test_connection()
    {
        if( ! $this->is_connected() ) {
            return array(
                'success' => false,
                'message' => __( 'API credentials not configured', 'triggerninja-for-ninja-forms' )
            );
        }

        try {
            $response = $this->api->test_connection();
            
            if( $response->status >= 200 && $response->status < 300 && $response->details ) {
                $lists_count = is_array( $response->details ) ? count( $response->details ) : 0;
                return array(
                    'success' => true,
                    'message' => __( 'Successfully connected to Acelle Mail', 'triggerninja-for-ninja-forms' ),
                    'data' => array(
                        'lists_count' => $lists_count
                    )
                );
            }

            return array(
                'success' => false,
                'message' => __( 'Invalid response from Acelle Mail API', 'triggerninja-for-ninja-forms' )
            );

        } catch( Exception $e ) {
            return array(
                'success' => false,
                /* translators: %s is an error message. */
                'message' => sprintf( __( 'Connection failed: %s', 'triggerninja-for-ninja-forms' ), $e->getMessage() )
            );
        }
    }

    /**
     * Get platform-specific metabox data for submissions
     * 
     * @param array $submission_data
     * @return array
     */
    public function get_submission_metabox_data( $submission_data )
    {
        $data = array();

        if( isset( $submission_data[ 'subscriber_uid' ] ) ){
            $label = __( 'Subscriber UID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'subscriber_uid' ];
        }

        if( isset( $submission_data[ 'email' ] ) ){
            $label = __( 'Email Address', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'email' ];
        }

        if( isset( $submission_data[ 'first_name' ] ) && ! empty( $submission_data[ 'first_name' ] ) ){
            $label = __( 'First Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'first_name' ];
        }

        if( isset( $submission_data[ 'last_name' ] ) && ! empty( $submission_data[ 'last_name' ] ) ){
            $label = __( 'Last Name', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'last_name' ];
        }

        if( isset( $submission_data[ 'list_uid' ] ) ){
            $label = __( 'List UID', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'list_uid' ];
        }

        if( isset( $submission_data[ 'status' ] ) ){
            $label = __( 'Status', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = $submission_data[ 'status' ];
        }

        if( isset( $submission_data[ 'custom_fields' ] ) && ! empty( $submission_data[ 'custom_fields' ] ) ){
            $label = __( 'Custom Fields', 'triggerninja-for-ninja-forms' );
            $data[ $label ] = implode( ', ', array_map( function( $key, $value ) {
                return "{$key}: {$value}";
            }, array_keys( $submission_data[ 'custom_fields' ] ), $submission_data[ 'custom_fields' ] ) );
        }

        /**
         * Filter metabox data display
         */
        $data = apply_filters( 'triggerninja_metabox_data', $data, $submission_data, $this->id );
        $data = apply_filters( "triggerninja_{$this->id}_metabox_data", $data, $submission_data, $this->id );

        return $data;
    }
}