<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Acelle Mail API Integration
 */
class TriggerNinja_Acelle_API
{
    /**
     * API Endpoint URL
     * 
     * @var string
     */
    private $api_endpoint;
    
    /**
     * API Token
     * 
     * @var string
     */
    private $api_token;
    
    /**
     * Form ID for logging
     * 
     * @var int
     */
    private $form_id;
    
    /**
     * Constructor
     * 
     * @param string $api_endpoint Acelle Mail API endpoint URL
     * @param string $api_token API token
     * @param int $form_id Form ID for logging
     */
    public function __construct( $api_endpoint, $api_token, $form_id = 0 )
    {
        $this->api_endpoint = rtrim( $api_endpoint, '/' ) . '/';
        $this->api_token = $api_token;
        $this->form_id = $form_id;
    }
    
    /**
     * Make API request to Acelle Mail using centralized HTTP client
     * 
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array $data Request data
     * @return TriggerNinja_Response
     */
    private function request( $endpoint, $method = 'GET', $data = array() )
    {
        // Build the full URL with API token as query parameter
        $url = $this->api_endpoint . $endpoint;
        $url = add_query_arg( 'api_token', $this->api_token, $url );
        
        $headers = array();
        
        $options = array(
            'timeout' => 30
        );
        
        return TriggerNinja_Http_Client::request( $method, $url, $data, $headers, $options, $this->form_id, 'acelle' );
        
        // Handle Acelle Mail specific error format
        if ( $response->status >= 400 && $response->details ) {
            $error_message = 'Unknown error occurred';
            
            // Try to extract Acelle Mail error message
            if ( is_object( $response->details ) && isset( $response->details->message ) ) {
                $error_message = $response->details->message;
            } elseif ( is_array( $response->details ) && isset( $response->details['message'] ) ) {
                $error_message = $response->details['message'];
            } elseif ( $response->error ) {
                $error_message = $response->error;
            }
            
            return new TriggerNinja_Response( $response->status, array(), $error_message );
        }
        
        return $response;
    }
    
    /**
     * Get all subscriber lists
     * 
     * @return TriggerNinja_Response
     */
    public function get_lists()
    {
        return $this->request( 'lists' );
    }
    
    /**
     * Subscribe a contact to a list
     * 
     * @param string $list_uid List unique ID
     * @param string $email Email address
     * @param string $first_name First name (optional)
     * @param string $last_name Last name (optional)
     * @param array $custom_fields Additional custom fields (optional)
     * @return TriggerNinja_Response
     */
    public function subscribe_to_list( $list_uid, $email, $first_name = '', $last_name = '', $custom_fields = array() )
    {
        if ( empty( $list_uid ) || empty( $email ) ) {
            return new TriggerNinja_Response( 400, array(), 'List UID and email are required' );
        }
        
        $data = array(
            'EMAIL' => trim( $email ),
            'list_uid' => $list_uid
        );
        
        // Add first name if provided
        if ( ! empty( $first_name ) ) {
            $data['FIRST_NAME'] = $first_name;
        }
        
        // Add last name if provided
        if ( ! empty( $last_name ) ) {
            $data['LAST_NAME'] = $last_name;
        }
        
        // Add custom fields if provided
        if ( ! empty( $custom_fields ) ) {
            $data = array_merge( $data, $custom_fields );
        }
        
        // Remove empty values to avoid API issues
        $data = array_filter( $data, function( $value ) {
            return $value !== '' && $value !== null;
        });
        
        return $this->request( 'subscribers', 'POST', $data );
    }
    
    /**
     * Get subscriber details
     * 
     * @param string $subscriber_uid Subscriber unique ID
     * @return TriggerNinja_Response
     */
    public function get_subscriber( $subscriber_uid )
    {
        if ( empty( $subscriber_uid ) ) {
            return new TriggerNinja_Response( 400, array(), 'Subscriber UID is required' );
        }
        
        return $this->request( "subscribers/{$subscriber_uid}" );
    }
    
    /**
     * Update subscriber information
     * 
     * @param string $subscriber_uid Subscriber unique ID
     * @param array $data Subscriber data to update
     * @return TriggerNinja_Response
     */
    public function update_subscriber( $subscriber_uid, $data )
    {
        if ( empty( $subscriber_uid ) ) {
            return new TriggerNinja_Response( 400, array(), 'Subscriber UID is required' );
        }
        
        // Remove empty values to avoid API issues
        $data = array_filter( $data, function( $value ) {
            return $value !== '' && $value !== null;
        });
        
        return $this->request( "subscribers/{$subscriber_uid}", 'PATCH', $data );
    }
    
    /**
     * Unsubscribe a contact from a list
     * 
     * @param string $subscriber_uid Subscriber unique ID
     * @return TriggerNinja_Response
     */
    public function unsubscribe( $subscriber_uid )
    {
        if ( empty( $subscriber_uid ) ) {
            return new TriggerNinja_Response( 400, array(), 'Subscriber UID is required' );
        }
        
        return $this->request( "subscribers/{$subscriber_uid}/unsubscribe", 'PATCH' );
    }
    
    /**
     * Get list details
     * 
     * @param string $list_uid List unique ID
     * @return TriggerNinja_Response
     */
    public function get_list_details( $list_uid )
    {
        if ( empty( $list_uid ) ) {
            return new TriggerNinja_Response( 400, array(), 'List UID is required' );
        }
        
        return $this->request( "lists/{$list_uid}" );
    }
    
    /**
     * Test API connection by attempting to get lists
     * 
     * @return TriggerNinja_Response
     */
    public function test_connection()
    {
        return $this->get_lists();
    }
    
    /**
     * Get account information (if available in API)
     * 
     * @return TriggerNinja_Response
     */
    public function get_account_info()
    {
        // Acelle Mail might have an account endpoint - using lists as fallback
        return $this->get_lists();
    }
}