<?php if ( ! defined( 'ABSPATH' ) ) exit;

// @codingStandardsIgnoreFile

/**
 * TriggerNinja Logger
 * 
 * Handles logging of API requests and responses to database for debugging and tracking
 */
class TriggerNinja_Logger
{
    /**
     * Database table name
     * 
     * @var string
     */
    private static $table_name = 'triggerninja_logs';
    
    /**
     * Check if logging is enabled
     * 
     * @return bool
     */
    public static function is_logging_enabled()
    {
        return get_option( 'triggerninja_enable_logging', true );
    }
    
    /**
     * Log an API request and response
     * 
     * @param string $platform Platform name (e.g., 'mailchimp', 'acelle')
     * @param string $method HTTP method (GET, POST, etc.)
     * @param string $url Request URL
     * @param array $request_data Request data sent
     * @param array $response_data Response data received
     * @param int $status_code HTTP status code
     * @param string $status Status (success, error)
     * @param string $error_message Error message if any
     * @param int $form_id Form ID if available
     * @return int|false Log ID on success, false on failure
     */
    public static function log_request( $platform, $method, $url, $request_data, $response_data, $status_code, $status = 'success', $error_message = '', $form_id = 0 )
    {
        // Don't log if logging is disabled
        if ( ! self::is_logging_enabled() ) {
            return false;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;
        
        // Prepare data for insertion
        $request_json = is_array( $request_data ) || is_object( $request_data ) ? 
            wp_json_encode( $request_data, JSON_PRETTY_PRINT ) : 
            (string) $request_data;
            
        $response_json = is_array( $response_data ) || is_object( $response_data ) ? 
            wp_json_encode( $response_data, JSON_PRETTY_PRINT ) : 
            (string) $response_data;
        
        // Insert log entry
        $result = $wpdb->insert(
            $table_name,
            array(
                'platform' => sanitize_text_field( $platform ),
                'method' => sanitize_text_field( strtoupper( $method ) ),
                'url' => esc_url_raw( $url ),
                'request_data' => sanitize_textarea_field( $request_json ),
                'response_data' => sanitize_textarea_field( $response_json ),
                'status_code' => absint( $status_code ),
                'status' => sanitize_text_field( $status ),
                'error_message' => sanitize_textarea_field( $error_message ),
                'form_id' => absint( $form_id ),
                'created_at' => current_time( 'mysql' )
            ),
            array( '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%d', '%s' )
        );
        
        return $result !== false ? $wpdb->insert_id : false;
    }
    
    /**
     * Get logs with filtering and pagination
     * 
     * @param int $limit Number of logs to retrieve
     * @param int $offset Offset for pagination
     * @param string $platform Platform filter
     * @param string $status Status filter (success, error)
     * @param string $method Method filter (GET, POST, etc.)
     * @param int $form_id Form ID filter
     * @return array
     */
    public static function get_logs( $limit = 20, $offset = 0, $platform = '', $status = '', $method = '', $form_id = 0 )
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;

        $where_clauses = array( '1=1' );
        $params = array();

        if ( ! empty( $platform ) ) {
            $where_clauses[] = 'platform = %s';
            $params[] = $platform;
        }

        if ( ! empty( $status ) ) {
            $where_clauses[] = 'status = %s';
            $params[] = $status;
        }

        if ( ! empty( $method ) ) {
            $where_clauses[] = 'method = %s';
            $params[] = $method;
        }

        if ( $form_id > 0 ) {
            $where_clauses[] = 'form_id = %d';
            $params[] = $form_id;
        }

        $where_sql = implode( ' AND ', $where_clauses );
        $params[] = $limit;
        $params[] = $offset;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $query = $wpdb->prepare( "SELECT * FROM `{$table_name}` WHERE {$where_sql} ORDER BY created_at DESC LIMIT %d OFFSET %d", ...$params );
        
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $result = $wpdb->get_results( $query, ARRAY_A );

        return $result;
    }
    
    /**
     * Get total count of logs with filtering
     * 
     * @param string $platform Platform filter
     * @param string $status Status filter
     * @param string $method Method filter
     * @param int $form_id Form ID filter
     * @return int
     */
    public static function get_logs_count( $platform = '', $status = '', $method = '', $form_id = 0 )
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;

        $where_clauses = array( '1=1' );
        $params = array();

        if ( ! empty( $platform ) ) {
            $where_clauses[] = 'platform = %s';
            $params[] = $platform;
        }

        if ( ! empty( $status ) ) {
            $where_clauses[] = 'status = %s';
            $params[] = $status;
        }

        if ( ! empty( $method ) ) {
            $where_clauses[] = 'method = %s';
            $params[] = $method;
        }

        if ( $form_id > 0 ) {
            $where_clauses[] = 'form_id = %d';
            $params[] = $form_id;
        }

        $where_sql = implode( ' AND ', $where_clauses );

        if ( ! empty( $params ) ) {
            $query = $wpdb->prepare( "SELECT COUNT(*) FROM `{$table_name}` WHERE {$where_sql}", ...$params );
        } else {
            $query = "SELECT COUNT(*) FROM `{$table_name}` WHERE {$where_sql}";
        }

        return (int) $wpdb->get_var( $query );
    }
    
    /**
     * Get a single log entry by ID
     * 
     * @param int $log_id Log ID
     * @return array|null
     */
    public static function get_log_by_id( $log_id )
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;

        return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM `{$table_name}` WHERE id = %d", $log_id ), ARRAY_A );
    }
    
    /**
     * Clean up old logs
     * 
     * @param int $days Number of days to keep logs (older logs will be deleted)
     * @return int|false Number of deleted rows or false on error
     */
    public static function cleanup_old_logs( $days = 30 )
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;
        $query = $wpdb->prepare( "DELETE FROM `{$table_name}` WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)", $days );

        return $wpdb->query( $query );
    }
    
    /**
     * Delete all logs
     * 
     * @return int|false Number of deleted rows or false on error
     */
    public static function clear_all_logs()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;
        $query = "DELETE FROM `{$table_name}`";

        return $wpdb->query( $query );
    }
    
    /**
     * Get available platforms from logs
     * 
     * @return array
     */
    public static function get_available_platforms()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;
        $query = "SELECT DISTINCT platform FROM `{$table_name}` ORDER BY platform";
        $platforms = $wpdb->get_col( $query );

        return $platforms;
    }
    
    /**
     * Create the logs database table
     */
    public static function create_table()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE {$table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            platform varchar(100) NOT NULL,
            method varchar(10) NOT NULL,
            url text NOT NULL,
            request_data longtext,
            response_data longtext,
            status_code int(3) unsigned DEFAULT 0,
            status varchar(20) NOT NULL DEFAULT 'success',
            error_message text,
            form_id bigint(20) unsigned DEFAULT 0,
            created_at datetime NOT NULL,
            PRIMARY KEY (id),
            KEY platform (platform),
            KEY status (status),
            KEY form_id (form_id),
            KEY created_at (created_at)
        ) {$charset_collate};";
        
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );
    }
    
    /**
     * Drop the logs table (for uninstall)
     */
    public static function drop_table()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::$table_name;
        $query = "DROP TABLE IF EXISTS `{$table_name}`";
        
        $wpdb->query( $query );
    }
}