<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * TriggerNinja Admin Page
 * 
 * Handles the dedicated admin page for platform management
 */
class TriggerNinja_Admin_Page
{
    /**
     * Plugin instance
     * 
     * @var TriggerNinja_Plugin
     */
    private $plugin;
    
    /**
     * Platform manager instance
     * 
     * @var TriggerNinja_Platform_Manager
     */
    private $platform_manager;
    
    /**
     * Page hook suffix
     * 
     * @var string
     */
    private $page_hook;
    
    /**
     * Constructor
     * 
     * @param TriggerNinja_Plugin $plugin
     */
    public function __construct( $plugin )
    {
        $this->plugin = $plugin;
        $this->platform_manager = $plugin->get_platform_manager();
        
        // Hook into WordPress admin
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        
        // Handle form submissions
        add_action( 'admin_post_triggerninja_save_settings', array( $this, 'handle_settings_save' ) );
        
        // Handle AJAX requests for logs
        add_action( 'wp_ajax_triggerninja_view_log_details', array( $this, 'ajax_view_log_details' ) );
        add_action( 'wp_ajax_triggerninja_cleanup_logs', array( $this, 'ajax_cleanup_logs' ) );
    }
    
    /**
     * Add admin menu page
     */
    public function add_admin_menu()
    {
        // Check if Ninja Forms menu exists
        if ( ! menu_page_url( 'ninja-forms', false ) && ! class_exists( 'Ninja_Forms' ) ) {
            // If Ninja Forms isn't active, add as top-level menu
            $this->page_hook = add_menu_page(
                __( 'TriggerNinja', 'triggerninja-for-ninja-forms' ),
                __( 'TriggerNinja', 'triggerninja-for-ninja-forms' ),
                'manage_options',
                'triggerninja-platforms',
                array( $this, 'render_page' ),
                'dashicons-email-alt',
                30
            );
        } else {
            // Add as submenu under Ninja Forms
            $this->page_hook = add_submenu_page(
                'ninja-forms',
                __( 'TriggerNinja Platforms', 'triggerninja-for-ninja-forms' ),
                __( 'TriggerNinja', 'triggerninja-for-ninja-forms' ),
                'manage_options',
                'triggerninja-platforms',
                array( $this, 'render_page' )
            );
        }
        
        // Add help tab when page loads
        add_action( 'load-' . $this->page_hook, array( $this, 'add_help_tabs' ) );
    }
    
    /**
     * Enqueue scripts and styles for admin page
     */
    public function enqueue_scripts( $hook )
    {
        // Only load on our admin page
        if( $hook !== $this->page_hook ) return;
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab parameter is for UI display only, not form processing
        $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'platforms';
        if( $current_tab === 'logs' ) {
            wp_enqueue_script( 'thickbox' );
            wp_enqueue_style( 'thickbox' );
        }
        
        // Enqueue platform manager assets
        wp_enqueue_script( 
            'triggerninja-platform-manager', 
            $this->plugin->url( 'assets/js/platform-manager.js' ), 
            array( 'jquery' ), 
            $this->plugin->version(), 
            true 
        );
        
        wp_enqueue_script(
            'triggerninja-platform-connections',
            $this->plugin->url( 'assets/js/platform-connections.js' ),
            array( 'jquery', 'triggerninja-platform-manager' ),
            $this->plugin->version(),
            true
        );
        
        wp_enqueue_style( 
            'triggerninja-platform-manager', 
            $this->plugin->url( 'assets/css/platform-manager.css' ), 
            array(), 
            $this->plugin->version() 
        );
        
        // Add admin page specific styles
        wp_add_inline_style( 'triggerninja-platform-manager', wp_strip_all_tags( $this->get_admin_page_styles() ) );
        
        // Localize script for AJAX
        wp_localize_script( 'triggerninja-platform-manager', 'triggerNinjaAjax', array(
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'triggerninja_platform_manager' ),
            'strings' => array(
                'enabling'      => __( 'Enabling...', 'triggerninja-for-ninja-forms' ),
                'disabling'     => __( 'Disabling...', 'triggerninja-for-ninja-forms' ),
                'configuring'   => __( 'Loading configuration...', 'triggerninja-for-ninja-forms' ),
                'error'         => __( 'An error occurred. Please try again.', 'triggerninja-for-ninja-forms' ),
                'confirm_disable' => __( 'Are you sure you want to disable this platform? Any configured settings will be preserved.', 'triggerninja-for-ninja-forms' )
            ),
            'connections' => array(
                'aweber' => array(
                    'missingCode'     => __( 'Please enter the authorization code from AWeber.', 'triggerninja-for-ninja-forms' ),
                    'missingVerifier' => __( 'Missing PKCE code verifier. Please refresh the page and try again.', 'triggerninja-for-ninja-forms' ),
                    'missingNonce'    => __( 'Security data is missing. Please refresh the page and try again.', 'triggerninja-for-ninja-forms' ),
                    'connecting'      => __( 'Connecting...', 'triggerninja-for-ninja-forms' ),
                    'connectSuccess'  => __( 'AWeber connected successfully!', 'triggerninja-for-ninja-forms' ),
                    'connectError'    => __( 'Error connecting AWeber. Please try again.', 'triggerninja-for-ninja-forms' ),
                    'connectErrorPrefix' => __( 'Error connecting AWeber: ', 'triggerninja-for-ninja-forms' ),
                    'disconnectConfirm' => __( 'Are you sure you want to disconnect AWeber?', 'triggerninja-for-ninja-forms' ),
                    'disconnecting'   => __( 'Disconnecting...', 'triggerninja-for-ninja-forms' ),
                    'disconnectSuccess' => __( 'AWeber disconnected successfully.', 'triggerninja-for-ninja-forms' ),
                    'disconnectError' => __( 'Error disconnecting AWeber. Please try again.', 'triggerninja-for-ninja-forms' )
                ),
                'constantcontact' => array(
                    'disconnectConfirm' => __( 'Are you sure you want to disconnect Constant Contact?', 'triggerninja-for-ninja-forms' ),
                    'disconnectSuccess' => __( 'Constant Contact disconnected successfully.', 'triggerninja-for-ninja-forms' ),
                    'disconnectError'   => __( 'Error disconnecting Constant Contact.', 'triggerninja-for-ninja-forms' )
                )
            )
        ));
    }
    
    /**
     * Render the admin page
     */
    public function render_page()
    {
        // Check user permissions
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'triggerninja-for-ninja-forms' ) );
        }
        
        // Handle settings save
        $this->maybe_show_save_message();
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'platforms';
        ?>
        <div class="wrap triggerninja-admin-page">
            <h1 class="wp-heading-inline">
                <?php esc_html_e( 'TriggerNinja Platform Manager', 'triggerninja-for-ninja-forms' ); ?>
            </h1>
            
            <?php $this->render_page_tabs_nav(); ?>
            
            <div class="triggerninja-admin-content">
                <?php $this->render_current_tab_content( $current_tab ); ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render page tabs navigation only
     */
    private function render_page_tabs_nav()
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- this is a harmless UI tab value, sanitized below
        $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'platforms';
        $tabs = array(
            'platforms' => __( 'Platform Manager', 'triggerninja-for-ninja-forms' ),
            'settings'  => __( 'Global Settings', 'triggerninja-for-ninja-forms' ),
            'logs'      => __( 'Logs', 'triggerninja-for-ninja-forms' ),
            'help'      => __( 'Help & Documentation', 'triggerninja-for-ninja-forms' ),
        );
        
        echo '<nav class="nav-tab-wrapper wp-clearfix">';
        foreach( $tabs as $tab_key => $tab_label ) {
            $active_class = $current_tab === $tab_key ? 'nav-tab-active' : '';
            $url = add_query_arg( 'tab', $tab_key, admin_url( 'admin.php?page=triggerninja-platforms' ) );
            echo '<a href="' . esc_url( $url ) . '" class="nav-tab ' . esc_attr( $active_class ) . '">' . esc_html( $tab_label ) . '</a>';
        }
        echo '</nav>';
    }
    
    /**
     * Render current tab content
     */
    private function render_current_tab_content( $current_tab )
    {
        switch( $current_tab ) {
            case 'settings':
                $this->render_global_settings();
                break;
            case 'logs':
                $this->render_logs_page();
                break;
            case 'help':
                $this->render_help_content();
                break;
            default:
                $this->render_platform_manager();
                break;
        }
    }
    
    /**
     * Render the platform manager interface
     */
    private function render_platform_manager()
    {
        // Use the platform manager to render the interface
        $this->platform_manager->render_platform_manager_page();
    }
    
    /**
     * Render global settings tab
     */
    private function render_global_settings()
    {
        $logging_enabled = get_option( 'triggerninja_enable_logging', true );
        ?>
        <div class="triggerninja-settings-section">
            <h2><?php esc_html_e( 'Global Settings', 'triggerninja-for-ninja-forms' ); ?></h2>
            
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'triggerninja_save_settings', 'triggerninja_nonce' ); ?>
                <input type="hidden" name="action" value="triggerninja_save_settings" />
                
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'API Request Logging', 'triggerninja-for-ninja-forms' ); ?></th>
                        <td>
                            <label for="triggerninja_enable_logging">
                                <input type="checkbox" 
                                       id="triggerninja_enable_logging" 
                                       name="triggerninja_enable_logging" 
                                       value="1" 
                                       <?php checked( $logging_enabled ); ?> />
                                <?php esc_html_e( 'Enable logging of API requests and responses', 'triggerninja-for-ninja-forms' ); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e( 'When enabled, all API requests and responses will be logged to the database for debugging purposes. You can view these logs in the Logs tab.', 'triggerninja-for-ninja-forms' ); ?>
                                <br>
                                <strong><?php esc_html_e( 'Note:', 'triggerninja-for-ninja-forms' ); ?></strong> 
                                <?php esc_html_e( 'Sensitive information like API keys will be masked in the logs for security.', 'triggerninja-for-ninja-forms' ); ?>
                            </p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button( __( 'Save Settings', 'triggerninja-for-ninja-forms' ) ); ?>
            </form>
            
            <?php if( $logging_enabled ) : ?>
            <div class="triggerninja-logging-info">
                <h3><?php esc_html_e( 'Logging Statistics', 'triggerninja-for-ninja-forms' ); ?></h3>
                <?php 
                    $total_logs = TriggerNinja_Logger::get_logs_count();
                    $success_logs = TriggerNinja_Logger::get_logs_count( '', 'success' );
                    $error_logs = TriggerNinja_Logger::get_logs_count( '', 'error' );
                ?>
                <p>
                    <strong><?php esc_html_e( 'Total Logs:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php echo esc_html( number_format( $total_logs ) ); ?> 
                    | <strong><?php esc_html_e( 'Successful:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php echo esc_html( number_format( $success_logs ) ); ?>
                    | <strong><?php esc_html_e( 'Errors:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php echo esc_html( number_format( $error_logs ) ); ?>
                </p>
                <p>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=triggerninja-platforms&tab=logs' ) ); ?>" class="button">
                        <?php esc_html_e( 'View All Logs', 'triggerninja-for-ninja-forms' ); ?>
                    </a>
                </p>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render help content tab
     */
    private function render_help_content()
    {
        ?>
        <div class="triggerninja-help-section">
            <h2><?php esc_html_e( 'Help & Documentation', 'triggerninja-for-ninja-forms' ); ?></h2>

            <div class="help-grid">
                <div class="help-card full-width">
                    <h3><?php esc_html_e( 'Welcome to TriggerNinja!', 'triggerninja-for-ninja-forms' ); ?></h3>
                    <p><?php esc_html_e( 'TriggerNinja is a powerful tool that connects your Ninja Forms to your favorite marketing platforms. When a user submits a form, TriggerNinja automatically sends the data to your enabled platforms, such as email marketing services, CRMs, and more.', 'triggerninja-for-ninja-forms' ); ?></p>
                </div>

                <div class="help-card">
                    <h3><?php esc_html_e( 'Getting Started', 'triggerninja-for-ninja-forms' ); ?></h3>
                    <p><?php esc_html_e( 'Follow these steps to connect your first platform and start sending data from your forms:', 'triggerninja-for-ninja-forms' ); ?></p>
                    <ol>
                        <li><strong><?php esc_html_e( 'Enable a Platform:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php
                            /* translators: %s: URL to Platform Manager tab */
                            printf(
                                /* translators: %s: URL to Platform Manager tab */
                                wp_kses_post( __( 'Go to the <a href="%s">Platform Manager</a> tab.', 'triggerninja-for-ninja-forms' ) ),
                                esc_url( admin_url( 'admin.php?page=triggerninja-platforms&tab=platforms' ) )
                            );
                        ?></li>
                        <li><strong><?php esc_html_e( 'Configure the Platform:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php esc_html_e( 'Click the "Enable" button on the platform you want to use, then click "Configure" and enter your API credentials.', 'triggerninja-for-ninja-forms' ); ?></li>
                        <li><strong><?php esc_html_e( 'Add a Form Action:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php esc_html_e( 'Go to your Ninja Forms, edit a form, and add a new action. Select your enabled platform from the list of available actions.', 'triggerninja-for-ninja-forms' ); ?></li>
                        <li><strong><?php esc_html_e( 'Map Your Fields:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php esc_html_e( 'In the action settings, map your form fields to the corresponding platform fields (e.g., map the "Email" field to the platform\'s "email_address" field).', 'triggerninja-for-ninja-forms' ); ?></li>
                        <li><strong><?php esc_html_e( 'Test Your Form:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php esc_html_e( 'Submit a test entry to your form to ensure that the data is being sent to your platform correctly.', 'triggerninja-for-ninja-forms' ); ?></li>
                    </ol>
                </div>

                <div class="help-card">
                    <h3><?php esc_html_e( 'Troubleshooting', 'triggerninja-for-ninja-forms' ); ?></h3>
                    <p><?php esc_html_e( 'If you encounter any issues, here are some common solutions:', 'triggerninja-for-ninja-forms' ); ?></p>
                    <ul>
                        <li><strong><?php esc_html_e( 'API Connection Errors:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php esc_html_e( 'If you see an API connection error, double-check that your API credentials are correct and that your server is not blocking outbound requests to the platform\'s API.', 'triggerninja-for-ninja-forms' ); ?></li>
                        <li><strong><?php esc_html_e( 'Data Not Syncing:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php
                            
                            printf(
                                /* translators: %s: URL to Logs tab */
                                wp_kses_post( __( 'If data is not appearing in your platform, check the <a href="%s">Logs</a> tab for any error messages. The logs will provide detailed information about each API request and response.', 'triggerninja-for-ninja-forms' ) ),
                                esc_url( admin_url( 'admin.php?page=triggerninja-platforms&tab=logs' ) )
                            );
                        ?></li>
                        <li><strong><?php esc_html_e( 'Platform Not Appearing in Actions:', 'triggerninja-for-ninja-forms' ); ?></strong> <?php
                            printf(
                                /* translators: %s: URL to Platform Manager tab */
                                wp_kses_post( __( 'Make sure that the platform is enabled in the <a href="%s">Platform Manager</a>.', 'triggerninja-for-ninja-forms' ) ),
                                esc_url( admin_url( 'admin.php?page=triggerninja-platforms&tab=platforms' ) )
                            );
                        ?></li>
                    </ul>
                </div>

                <div class="help-card full-width">
                    <h3><?php esc_html_e( 'Support & Resources', 'triggerninja-for-ninja-forms' ); ?></h3>
                    <p><?php esc_html_e( 'If you need further assistance, please consult the following resources:', 'triggerninja-for-ninja-forms' ); ?></p>
                    <ul>
                        <li><a href="https://triggerninja.com/documentation/" target="_blank"><?php esc_html_e( 'Official Documentation', 'triggerninja-for-ninja-forms' ); ?></a></li>
                        <li><a href="https://wordpress.org/support/plugin/triggerninja/" target="_blank"><?php esc_html_e( 'WordPress.org Support Forum', 'triggerninja-for-ninja-forms' ); ?></a></li>
                        <li><a href="https://triggerninja.com/contact/" target="_blank"><?php esc_html_e( 'Contact Support', 'triggerninja-for-ninja-forms' ); ?></a></li>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render logs page
     */
    private function render_logs_page()
    {
        // Handle cleanup action
        if ( isset( $_POST['action'] ) && $_POST['action'] === 'cleanup_logs' && isset( $_POST['triggerninja_logs_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['triggerninja_logs_nonce'] ) ), 'triggerninja_cleanup_logs' ) ) {
            $days = isset( $_POST['cleanup_days'] ) ? intval( $_POST['cleanup_days'] ) : 30;
            $deleted = TriggerNinja_Logger::cleanup_old_logs( $days );
            if ( $deleted !== false ) {
                /* translators: %d is the number of logs deleted. */
                echo '<div class="notice notice-success"><p>' . esc_html( sprintf( __( '%d logs deleted successfully.', 'triggerninja-for-ninja-forms' ), $deleted ) ) . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html__( 'Failed to delete logs.', 'triggerninja-for-ninja-forms' ) . '</p></div>';
            }
        }
        
        // Create list table
        $logs_table = new TriggerNinja_Logs_List_Table();
        $logs_table->prepare_items();
        
        ?>
        <div class="triggerninja-logs-section">
            <h2><?php esc_html_e( 'API Request Logs', 'triggerninja-for-ninja-forms' ); ?></h2>
            
            <?php if( ! TriggerNinja_Logger::is_logging_enabled() ) : ?>
                <div class="notice notice-warning">
                    <p>
                        <?php esc_html_e( 'Logging is currently disabled.', 'triggerninja-for-ninja-forms' ); ?>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=triggerninja-platforms&tab=settings' ) ); ?>">
                            <?php esc_html_e( 'Enable it in Global Settings', 'triggerninja-for-ninja-forms' ); ?>
                        </a>
                    </p>
                </div>
            <?php endif; ?>
            
            <div class="tablenav top">
                <div class="alignleft actions">
                    <form method="post" style="display: inline;">
                        <?php wp_nonce_field( 'triggerninja_cleanup_logs', 'triggerninja_logs_nonce' ); ?>
                        <input type="hidden" name="action" value="cleanup_logs">
                        <label for="cleanup_days"><?php esc_html_e( 'Delete logs older than:', 'triggerninja-for-ninja-forms' ); ?></label>
                        <input type="number" name="cleanup_days" id="cleanup_days" value="30" min="1" max="365" style="width: 60px;">
                        <span><?php esc_html_e( 'days', 'triggerninja-for-ninja-forms' ); ?></span>
                        <input type="submit" class="button" value="<?php esc_attr_e( 'Clean Up', 'triggerninja-for-ninja-forms' ); ?>" 
                               onclick="return confirm('<?php echo esc_js( __( 'Are you sure you want to delete old logs?', 'triggerninja-for-ninja-forms' ) ); ?>');">
                    </form>
                </div>
                
                <div class="alignright actions">
                    <?php 
                        $total_logs = TriggerNinja_Logger::get_logs_count();
                        $success_logs = TriggerNinja_Logger::get_logs_count( '', 'success' );
                        $error_logs = TriggerNinja_Logger::get_logs_count( '', 'error' );
                    ?>
                    <span class="displaying-num">
                        <?php
                            printf(
                                /* translators: %1$s is the total number of logs, %2$s is the number of successful logs, %3$s is the number of error logs. */
                                esc_html__( '%1$s total logs (%2$s success, %3$s errors)', 'triggerninja-for-ninja-forms' ), 
                                esc_html( number_format_i18n( $total_logs ) ), 
                                esc_html( number_format_i18n( $success_logs ) ), 
                                esc_html( number_format_i18n( $error_logs ) ) 
                            );
                        ?>
                    </span>
                </div>
            </div>
            
            <form method="get">
                <input type="hidden" name="page" value="<?php echo isset( $_GET['page'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_GET['page'] ) ) ) : ''; ?>">
                <input type="hidden" name="tab" value="logs">
                <?php wp_nonce_field( 'triggerninja-logs-filter', 'triggerninja-logs-filter-nonce' ); ?>
                <?php $logs_table->display(); ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * AJAX handler for viewing log details
     */
    public function ajax_view_log_details()
    {
        check_ajax_referer( 'triggerninja_view_log_details', 'nonce' );
        if( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $log_id = intval( $_GET['log_id'] ?? 0 );
        if( ! $log_id ) {
            wp_die( esc_html__( 'Invalid log ID.', 'triggerninja-for-ninja-forms' ) );
        }
        
        $log = TriggerNinja_Logger::get_log_by_id( $log_id );
        if( ! $log ) {
            wp_die( esc_html__( 'Log not found.', 'triggerninja-for-ninja-forms' ) );
        }

        wp_enqueue_style(
            'triggerninja-actions-log-details',
            $this->plugin->url( 'assets/css/log-details.css' ),
            array(),
            $this->plugin->version()
        );
        
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="<?php echo esc_attr( get_bloginfo( 'charset' ) ); ?>">
            <title><?php esc_html_e( 'Log Details', 'triggerninja-for-ninja-forms' ); ?></title>
            <?php wp_print_styles( array( 'triggerninja-actions-log-details' ) ); ?>
        </head>
        <body>
            <h2><?php esc_html_e( 'API Request Log Details', 'triggerninja-for-ninja-forms' ); ?></h2>
            
            <div class="log-section">
                <h3><?php esc_html_e( 'Request Information', 'triggerninja-for-ninja-forms' ); ?></h3>
                <div class="log-meta">
                    <div class="meta-item">
                        <div class="meta-label"><?php esc_html_e( 'ID', 'triggerninja-for-ninja-forms' ); ?></div>
                        <div class="meta-value"><?php echo esc_html( $log['id'] ); ?></div>
                    </div>
                    <div class="meta-item">
                        <div class="meta-label"><?php esc_html_e( 'Platform', 'triggerninja-for-ninja-forms' ); ?></div>
                        <div class="meta-value"><?php echo esc_html( ucfirst( $log['platform'] ) ); ?></div>
                    </div>
                    <div class="meta-item">
                        <div class="meta-label"><?php esc_html_e( 'Method', 'triggerninja-for-ninja-forms' ); ?></div>
                        <div class="meta-value"><code><?php echo esc_html( $log['method'] ); ?></code></div>
                    </div>
                    <div class="meta-item">
                        <div class="meta-label"><?php esc_html_e( 'Status', 'triggerninja-for-ninja-forms' ); ?></div>
                        <div class="meta-value">
                            <span class="status-<?php echo esc_attr( $log['status'] ); ?>">
                                <?php echo esc_html( ucfirst( $log['status'] ) ); ?>
                            </span>
                            (<?php echo esc_html( $log['status_code'] ); ?>)
                        </div>
                    </div>
                    <div class="meta-item">
                        <div class="meta-label"><?php esc_html_e( 'Form ID', 'triggerninja-for-ninja-forms' ); ?></div>
                        <div class="meta-value"><?php echo $log['form_id'] ? esc_html( $log['form_id'] ) : '-'; ?></div>
                    </div>
                    <div class="meta-item">
                        <div class="meta-label"><?php esc_html_e( 'Date', 'triggerninja-for-ninja-forms' ); ?></div>
                        <div class="meta-value"><?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $log['created_at'] ) ) ); ?></div>
                    </div>
                </div>
            </div>
            
            <div class="log-section">
                <h3><?php esc_html_e( 'Request URL', 'triggerninja-for-ninja-forms' ); ?></h3>
                <div class="log-data">
                    <code><?php echo esc_html( $log['url'] ); ?></code>
                </div>
            </div>
            
            <?php if( ! empty( $log['error_message'] ) ) : ?>
            <div class="log-section">
                <h3><?php esc_html_e( 'Error Message', 'triggerninja-for-ninja-forms' ); ?></h3>
                <div class="log-data" style="background: #ffeaea; border-color: #dc3232;">
                    <pre><?php echo esc_html( $log['error_message'] ); ?></pre>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="log-section">
                <h3><?php esc_html_e( 'Request Data', 'triggerninja-for-ninja-forms' ); ?></h3>
                <div class="log-data">
                    <pre><?php echo esc_html( $log['request_data'] ); ?></pre>
                </div>
            </div>
            
            <div class="log-section">
                <h3><?php esc_html_e( 'Response Data', 'triggerninja-for-ninja-forms' ); ?></h3>
                <div class="log-data">
                    <pre><?php echo esc_html( $log['response_data'] ); ?></pre>
                </div>
            </div>
        </body>
        </html>
        <?php
        wp_die();
    }
    
    /**
     * Handle settings save
     */
    public function handle_settings_save()
    {
        // Verify nonce and permissions
        $nonce = isset( $_POST['triggerninja_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['triggerninja_nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, 'triggerninja_save_settings' ) ) {
            wp_die( esc_html__( 'Security check failed.', 'triggerninja-for-ninja-forms' ) );
        }
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Insufficient permissions.', 'triggerninja-for-ninja-forms' ) );
        }
        
        // Handle logging setting
        $logging_enabled = isset( $_POST['triggerninja_enable_logging'] ) ? 1 : 0;
        update_option( 'triggerninja_enable_logging', $logging_enabled );
        
        // Redirect back with success message and a nonce to validate the message display
        $redirect_url = add_query_arg( array(
            'page' => 'triggerninja-platforms',
            'tab'  => 'settings',
            'message' => 'settings_saved',
            'message_nonce' => wp_create_nonce( 'triggerninja_message' )
        ), admin_url( 'admin.php' ) );
        
        wp_safe_redirect( $redirect_url );
        exit;
    }
    /**
     * Maybe show save message
     */
    private function maybe_show_save_message()
    {
        // Require a valid message nonce to prevent processing arbitrary GET parameters
        if ( isset( $_GET['message'], $_GET['message_nonce'] ) ) {
            $message = sanitize_text_field( wp_unslash( $_GET['message'] ) );
            $message_nonce = sanitize_text_field( wp_unslash( $_GET['message_nonce'] ) );
            if ( ! wp_verify_nonce( $message_nonce, 'triggerninja_message' ) ) {
                return;
            }
            if ( $message === 'settings_saved' ) {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved successfully!', 'triggerninja-for-ninja-forms' ) . '</p></div>';
            } elseif ( $message === 'no_settings' ) {
                echo '<div class="notice notice-info is-dismissible"><p>' . esc_html__( 'No settings available to save yet.', 'triggerninja-for-ninja-forms' ) . '</p></div>';
            }
        }
    }
    
    /**
     * Add help tabs
     */
    public function add_help_tabs()
    {
        $screen = get_current_screen();
        
        $screen->add_help_tab( array(
            'id'      => 'triggerninja-overview',
            'title'   => __( 'Overview', 'triggerninja-for-ninja-forms' ),
            'content' => '<p>' . esc_html__( 'TriggerNinja connects your Ninja Forms to email marketing platforms. Use the Platform Manager to enable and configure the platforms you want to use.', 'triggerninja-for-ninja-forms' ) . '</p>',
        ) );
        
        $screen->add_help_tab( array(
            'id'      => 'triggerninja-getting-started',
            'title'   => __( 'Getting Started', 'triggerninja-for-ninja-forms' ),
            'content' => '<p>' . esc_html__( 'To get started, enable a platform, configure it with your API credentials, then add the platform action to your Ninja Forms.', 'triggerninja-for-ninja-forms' ) . '</p>',
        ) );
        
        $screen->set_help_sidebar(
            '<p><strong>' . esc_html__( 'For more information:', 'triggerninja-for-ninja-forms' ) . '</strong></p>' .
            '<p><a href="https://triggerninja.com/documentation/" target="_blank">' . esc_html__( 'Documentation', 'triggerninja-for-ninja-forms' ) . '</a></p>' .
            '<p><a href="https://wordpress.org/support/plugin/triggerninja/" target="_blank">' . esc_html__( 'Support', 'triggerninja-for-ninja-forms' ) . '</a></p>'
        );
    }
    
    /**
     * Get admin page specific styles
     */
    private function get_admin_page_styles()
    {
        return '
            /* Fix main layout structure */
            .triggerninja-admin-page .wp-heading-inline {
                display: block;
                width: 100%;
                margin-bottom: 20px;
            }
            .triggerninja-admin-page .nav-tab-wrapper {
                display: block;
                width: 100%;
                margin: 20px 0;
                clear: both;
            }
            .triggerninja-admin-page .nav-tab {
                float: left;
            }
            .triggerninja-admin-page .nav-tab-wrapper::after {
                content: "";
                display: table;
                clear: both;
            }
            /* Main content area - full width */
            .triggerninja-admin-content {
                width: 100%;
                box-sizing: border-box;
            }
            /* Ensure platform manager content does not overflow */
            .triggerninja-platform-manager {
                max-width: 100%;
                overflow-x: auto;
                box-sizing: border-box;
            }
            /* Ensure tabs display properly */
            .triggerninja-admin-page .nav-tab-wrapper {
                border-bottom: 1px solid #ccd0d4;
            }
            .triggerninja-admin-page .nav-tab {
                border-bottom: 1px solid transparent;
            }
            .triggerninja-admin-page .nav-tab-active {
                border-bottom: 1px solid #fff;
                margin-bottom: -1px;
            }
            .help-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
                gap: 20px;
            }
            .help-card {
                background: #fff;
                border: 1px solid #ccd0d4;
                padding: 20px;
                border-radius: 4px;
            }
            .help-card h3 {
                margin-top: 0;
                margin-bottom: 15px;
                color: #1d2327;
            }
            .triggerninja-settings-section {
                background: #fff;
                border: 1px solid #ccd0d4;
                padding: 20px;
                border-radius: 4px;
                margin-top: 20px;
            }
            /* Logs page specific styling */
            .triggerninja-logs-section {
                background: #fff;
                border: 1px solid #ccd0d4;
                padding: 0;
                border-radius: 4px;
                margin-top: 20px;
                overflow: hidden;
            }
            .triggerninja-logs-section h2 {
                margin: 0;
                padding: 20px;
                border-bottom: 1px solid #ccd0d4;
                background: #f8f9fa;
            }
            .triggerninja-logs-section .tablenav {
                padding: 10px 20px;
                background: #f8f9fa;
                border-bottom: 1px solid #ccd0d4;
            }
            .triggerninja-logs-section .wp-list-table {
                border: none;
                margin: 0;
            }
            .triggerninja-logs-section .wp-list-table th,
            .triggerninja-logs-section .wp-list-table td {
                padding: 12px 15px;
            }
            /* Log status styling */
            .status-success {
                color: #46b450;
                font-weight: 600;
            }
            .status-error {
                color: #dc3232;
                font-weight: 600;
            }
            .status-code {
                font-family: monospace;
                font-weight: bold;
                padding: 2px 6px;
                border-radius: 3px;
                font-size: 11px;
            }
            .status-code.status-success {
                background: #e8f5e9;
                color: #2e7d32;
            }
            .status-code.status-error {
                background: #ffebee;
                color: #c62828;
            }
            /* Platform badges */
            .platform-badge {
                display: inline-block;
                padding: 3px 8px;
                border-radius: 3px;
                font-size: 11px;
                font-weight: 600;
                text-transform: uppercase;
                color: #fff;
                background: #666;
            }
            .platform-badge.platform-mailchimp { background: #007c89; }
            .platform-badge.platform-emailoctopus { background: #e74c3c; }
            .platform-badge.platform-brevo { background: #0052cc; }
            .platform-badge.platform-klaviyo { background: #7b42bc; }
            .platform-badge.platform-kit { background: #f58220; }
            .platform-badge.platform-campaignmonitor { background: #509e2f; }
            .platform-badge.platform-acelle { background: #3498db; }
            .platform-badge.platform-lemlist { background: #ff6b35; }
            .platform-badge.platform-getresponse { background: #00baff; }
            .platform-badge.platform-mailerlite { background: #09c269; }
            .platform-badge.platform-aweber { background: #1f73b7; }
            /* Logging info section */
            .triggerninja-logging-info {
                margin-top: 20px;
                padding: 15px;
                background: #f0f8ff;
                border: 1px solid #b3d9ff;
                border-radius: 4px;
            }
            .triggerninja-logging-info h3 {
                margin-top: 0;
                color: #0073aa;
            }
        ';
    }
}

/**
 * Logs List Table for displaying API request logs
 */
if( ! class_exists( 'WP_List_Table' ) ) {
    require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

class TriggerNinja_Logs_List_Table extends WP_List_Table
{
    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct( array(
            'singular' => 'log',
            'plural'   => 'logs',
            'ajax'     => false
        ) );
    }
    
    /**
     * Define table columns
     */
    public function get_columns()
    {
        return array(
            'platform'      => esc_html__( 'Platform', 'triggerninja-for-ninja-forms' ),
            'method'        => esc_html__( 'Method', 'triggerninja-for-ninja-forms' ),
            'status_code'   => esc_html__( 'Code', 'triggerninja-for-ninja-forms' ),
            'form_id'       => esc_html__( 'Form ID', 'triggerninja-for-ninja-forms' ),
            'created_at'    => esc_html__( 'Date', 'triggerninja-for-ninja-forms' ),
            'details'       => esc_html__( 'Details', 'triggerninja-for-ninja-forms' )
        );
    }
    
    /**
     * Define sortable columns
     */
    public function get_sortable_columns()
    {
        return array(
            'platform'   => array( 'platform', false ),
            'method'     => array( 'method', false ),
            'created_at' => array( 'created_at', true )
        );
    }
    
    /**
     * Prepare table items
     */
    public function prepare_items()
    {
        // Only verify nonce if filter form has been submitted
        if ( isset( $_GET['filter_action'] ) && ( ! isset( $_GET['triggerninja-logs-filter-nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['triggerninja-logs-filter-nonce'] ) ), 'triggerninja-logs-filter' ) ) ) {
            return;
        }

        $per_page = 20;
        $current_page = $this->get_pagenum();
        $offset = ( $current_page - 1 ) * $per_page;
        
        // Get filter values
        $platform_filter = isset( $_GET['platform'] ) ? sanitize_text_field( wp_unslash( $_GET['platform'] ) ) : '';
        $status_filter = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '';
        $method_filter = isset( $_GET['method'] ) ? sanitize_text_field( wp_unslash( $_GET['method'] ) ) : '';
        $form_id_filter = isset( $_GET['form_id'] ) ? intval( $_GET['form_id'] ) : 0;
        
        // Get logs
        $logs = TriggerNinja_Logger::get_logs( $per_page, $offset, $platform_filter, $status_filter, $method_filter, $form_id_filter );
        $total_items = TriggerNinja_Logger::get_logs_count( $platform_filter, $status_filter, $method_filter, $form_id_filter );
        
        // Set pagination
        $this->set_pagination_args( array(
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil( $total_items / $per_page )
        ) );
        
        // Set column headers
        $this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );
        
        // Set items
        $this->items = $logs;
    }
    
    /**
     * Default column display
     */
    public function column_default( $item, $column_name )
    {
        switch( $column_name ) {
            case 'platform':
                return '<span class="platform-badge platform-' . esc_attr( $item['platform'] ) . '">' . esc_html( ucfirst( $item['platform'] ) ) . '</span>';
            case 'method':
                return '<code>' . esc_html( $item['method'] ) . '</code>';
            case 'status_code':
                $code_class = $item['status_code'] >= 200 && $item['status_code'] < 300 ? 'success' : 'error';
                return '<span class="status-code status-' . esc_attr( $code_class ) . '">' . esc_html( $item['status_code'] ) . '</span>';
            case 'form_id':
                return $item['form_id'] ? esc_html( $item['form_id'] ) : '-';
            case 'created_at':
                return esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $item['created_at'] ) ) );
            case 'details':
                $nonce = wp_create_nonce( 'triggerninja_view_log_details' );
                $details_url = add_query_arg( array(
                    'action'    => 'triggerninja_view_log_details',
                    'log_id'    => $item['id'],
                    'nonce'     => $nonce,
                    'TB_iframe' => 'true',
                    'width'     => '900',
                    'height'    => '700'
                ), admin_url( 'admin-ajax.php' ) );
                return '<a href="' . esc_url( $details_url ) . '" class="thickbox button button-small" title="' . esc_attr__( 'View Log Details', 'triggerninja-for-ninja-forms' ) . '">' . esc_html__( 'View', 'triggerninja-for-ninja-forms' ) . '</a>';
            default:
                return '';
        }
    }
    
    /**
     * Display when no items found
     */
    public function no_items()
    {
        if( ! TriggerNinja_Logger::is_logging_enabled() ) {
            esc_html_e( 'Logging is disabled. Enable it in Global Settings to start collecting logs.', 'triggerninja-for-ninja-forms' );
        } else {
            esc_html_e( 'No logs found. Logs will appear here after API requests are made.', 'triggerninja-for-ninja-forms' );
        }
    }
    
    /**
     * Display filters above the table
     */
    protected function extra_tablenav( $which )
    {
        if( $which == 'top' ) {
            echo '<div class="alignleft actions">';
            
            // Platform filter
            $platforms = TriggerNinja_Logger::get_available_platforms();
            if( ! empty( $platforms ) ) {
                echo '<select name="platform" id="platform-filter">';
                echo '<option value="">' . esc_html__( 'All Platforms', 'triggerninja-for-ninja-forms' ) . '</option>';
                                $current_platform = isset( $_GET['platform'] ) ? sanitize_text_field( wp_unslash( $_GET['platform'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                foreach( $platforms as $platform ) {
                    echo '<option value="' . esc_attr( $platform ) . '" ' . selected( $current_platform, $platform, false ) . '>' . esc_html( ucfirst( $platform ) ) . '</option>';
                }
                echo '</select>';
            }
            
            // Status filter
            echo '<select name="status" id="status-filter">';
            echo '<option value="">' . esc_html__( 'All Statuses', 'triggerninja-for-ninja-forms' ) . '</option>';
            $current_status = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            echo '<option value="success" ' . selected( $current_status, 'success', false ) . '>' . esc_html__( 'Success', 'triggerninja-for-ninja-forms' ) . '</option>';
            echo '<option value="error" ' . selected( $current_status, 'error', false ) . '>' . esc_html__( 'Error', 'triggerninja-for-ninja-forms' ) . '</option>';
            echo '</select>';
            
            // Method filter
            echo '<select name="method" id="method-filter">';
            echo '<option value="">' . esc_html__( 'All Methods', 'triggerninja-for-ninja-forms' ) . '</option>';
            $current_method = isset( $_GET['method'] ) ? sanitize_text_field( wp_unslash( $_GET['method'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $methods = array( 'GET', 'POST', 'PUT', 'DELETE', 'PATCH' );
            foreach( $methods as $method ) {
                echo '<option value="' . esc_attr( $method ) . '" ' . selected( $current_method, $method, false ) . '>' . esc_html( $method ) . '</option>';
            }
            echo '</select>';
            
            submit_button( __( 'Filter', 'triggerninja-for-ninja-forms' ), 'secondary', 'filter_action', false );
            
            echo '</div>';
        }
    }
}
