<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Abstract class TriggerNinja_Platform_Abstract
 * 
 * Provides base functionality for all marketing platform integrations
 */
abstract class TriggerNinja_Platform_Abstract implements TriggerNinja_Platform_Interface
{
    /**
     * Platform ID
     * 
     * @var string
     */
    protected $id;

    /**
     * Platform name
     * 
     * @var string
     */
    protected $name;

    /**
     * API instance
     * 
     * @var mixed
     */
    protected $api;

    /**
     * Platform settings
     * 
     * @var array
     */
    protected $settings = array();

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->load_settings();
        $this->init_api();
    }

    /**
     * Get the platform identifier
     * 
     * @return string
     */
    public function get_id()
    {
        return $this->id;
    }

    /**
     * Get the platform display name
     * 
     * @return string
     */
    public function get_name()
    {
        return $this->name;
    }

    /**
     * Load platform settings from WordPress options
     */
    protected function load_settings()
    {
        foreach( $this->get_settings_fields() as $field_id => $field_config ) {
            $this->settings[ $field_id ] = get_option( $field_id, '' );
        }
    }

    /**
     * Get setting value
     * 
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    protected function get_setting( $key, $default = '' )
    {
        return isset( $this->settings[ $key ] ) ? $this->settings[ $key ] : $default;
    }
    
    /**
     * Reload settings from database and reinitialize API
     */
    public function reload_settings()
    {
        $this->load_settings();
        $this->init_api();
    }

    /**
     * Initialize API connection (to be implemented by child classes)
     */
    abstract protected function init_api();

    /**
     * Get common field structure for forms
     * 
     * @return array
     */
    protected function get_common_fields()
    {
        $base_fields = array(
            'email_address' => array(
                'name' => 'email_address',
                'type' => 'textbox',
                'label' => __( 'Email Address', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            ),
            'first_name' => array(
                'name' => 'first_name',
                'type' => 'textbox',
                'label' => __( 'First Name', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            ),
            'last_name' => array(
                'name' => 'last_name',
                'type' => 'textbox',
                'label' => __( 'Last Name', 'triggerninja-for-ninja-forms' ),
                'group' => 'primary',
                'width' => 'full',
                'use_merge_tags' => true
            )
        );

        /**
         * Filter to add additional fields
         * 
         * @param array $base_fields The base fields (email, first_name, last_name)
         * @param string $platform_id The platform ID
         * @return array Extended fields array
         */
        $extended_fields = apply_filters( 'triggerninja_platform_fields', $base_fields, $this->id );
        $extended_fields = apply_filters( "triggerninja_{$this->id}_fields", $extended_fields, $this->id );

        return $extended_fields;
    }

    /**
     * Log debug information
     * 
     * @param array $data
     * @param string $message
     * @param string $type
     * @return array
     */
    protected function add_debug( $data, $message, $type = 'info' )
    {
        $data[ 'debug' ][ $this->get_id() ] = array(
            'message' => $message,
            'type' => $type,
            'timestamp' => current_time( 'timestamp' )
        );
        return $data;
    }

    /**
     * Save submission extra data
     * 
     * @param array $data
     * @param array $response_data
     * @return array
     */
    protected function save_submission_data( $data, $response_data )
    {
        if( isset( $data[ 'actions' ][ 'save' ][ 'sub_id' ] ) ){
            $sub = Ninja_Forms()->form()->sub( $data[ 'actions' ][ 'save' ][ 'sub_id' ] )->get();
            $sub->update_extra_value( $this->get_id(), $response_data );
            $sub->save();
        }
        return $data;
    }
}