/**
 * TriggerNinja Platform Manager JavaScript
 * 
 * Handles the interactive functionality of the platform management interface
 */

(function($) {
    'use strict';

    let PlatformManager = {

        init: function() {
            this.bindEvents();
            this.initSearchAndFilter();
        },

        bindEvents: function() {
            $(document).on('click', '.platform-enable', this.enablePlatform);
            $(document).on('click', '.platform-disable', this.disablePlatform);
            $(document).on('click', '.platform-configure', this.configurePlatform);
            $(document).on('click', '.modal-close, .modal-cancel', this.closeModal);
            $(document).on('click', '.modal-save', this.saveConfiguration);
            $(document).on('click', '.aweber-authorize-btn', this.authorizeAweber);
            $(document).on('click', '.platform-config-modal', function(e) {
                if (e.target === this) {
                    PlatformManager.closeModal();
                }
            });
        },

        initSearchAndFilter: function() {
            // Search functionality
            $('#platform-search').on('input', function() {
                const searchTerm = $(this).val().toLowerCase();
                PlatformManager.filterPlatforms(searchTerm, $('#platform-category-filter').val());
            });

            // Category filter
            $('#platform-category-filter').on('change', function() {
                const category = $(this).val();
                PlatformManager.filterPlatforms($('#platform-search').val().toLowerCase(), category);
            });
        },

        filterPlatforms: function(searchTerm, category) {
            $('.platform-category').each(function() {
                const $category = $(this);
                const categoryId = $category.data('category');
                let categoryHasVisiblePlatforms = false;

                // Check if category matches filter
                const categoryMatches = !category || category === categoryId;

                if (categoryMatches) {
                    // Filter platforms within this category
                    $category.find('.platform-card').each(function() {
                        const $card = $(this);
                        const platformName = $card.find('.platform-name').text().toLowerCase();
                        const platformDesc = $card.find('.platform-description').text().toLowerCase();
                        
                        const matchesSearch = !searchTerm || 
                            platformName.includes(searchTerm) || 
                            platformDesc.includes(searchTerm);

                        if (matchesSearch) {
                            $card.removeClass('hidden');
                            categoryHasVisiblePlatforms = true;
                        } else {
                            $card.addClass('hidden');
                        }
                    });
                }

                // Show/hide category based on whether it has visible platforms
                if (categoryMatches && categoryHasVisiblePlatforms) {
                    $category.removeClass('hidden');
                } else {
                    $category.addClass('hidden');
                }
            });
        },

        enablePlatform: function(e) {
            e.preventDefault();
            const $button = $(this);
            const platformId = $button.data('platform');
            
            PlatformManager.togglePlatform(platformId, 'enable', $button);
        },

        disablePlatform: function(e) {
            e.preventDefault();
            const $button = $(this);
            const platformId = $button.data('platform');
            
            if (confirm('Are you sure you want to disable this platform? Any configured settings will be preserved.')) {
                PlatformManager.togglePlatform(platformId, 'disable', $button);
            }
        },

        togglePlatform: function(platformId, action, $button) {
            const $card = $button.closest('.platform-card');
            const originalText = $button.text();
            
            // Show loading state
            $card.addClass('loading');
            $button.text(action === 'enable' ? triggerNinjaAjax.strings.enabling : triggerNinjaAjax.strings.disabling);
            $button.prop('disabled', true);
            
            // Safety timeout to remove loading state if AJAX doesn't complete
            const timeoutId = setTimeout(function() {
                $card.removeClass('loading');
                $button.prop('disabled', false);
                $button.text(originalText);
                PlatformManager.showNotice('Request timed out. Please try again.', 'error');
            }, 30000); // 30 second timeout

            $.ajax({
                url: triggerNinjaAjax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'triggerninja_toggle_platform',
                    platform_id: platformId,
                    action_type: action,
                    nonce: triggerNinjaAjax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        try {
                            PlatformManager.handleToggleSuccess(platformId, action, $card, response.data);
                        } catch (error) {
                            console.error('Error in handleToggleSuccess:', error);
                            PlatformManager.handleToggleError($button, originalText, 'Error processing success response');
                        }
                    } else {
                        PlatformManager.handleToggleError($button, originalText, response.data || triggerNinjaAjax.strings.error);
                    }
                },
                error: function(xhr, status, error) {
                    PlatformManager.handleToggleError($button, originalText, triggerNinjaAjax.strings.error);
                },
                complete: function() {
                    clearTimeout(timeoutId);
                    
                    // Aggressive cleanup - remove loading state from all cards
                    $('.platform-card.loading').removeClass('loading');
                    $card.removeClass('loading');
                    $button.prop('disabled', false);
                    $button.text(originalText);
                    
                    // Delayed cleanup as fallback
                    setTimeout(function() {
                        $('.platform-card.loading').removeClass('loading');
                        $card.removeClass('loading');
                    }, 100);
                }
            });
        },

        handleToggleSuccess: function(platformId, action, $card, data) {
            if (action === 'enable') {
                // Move card to enabled section
                const $enabledGrid = $('.enabled-grid');
                const $cardClone = $card.clone();
                
                // Update the cloned card for enabled state
                $cardClone.removeClass('available').addClass('enabled');
                $cardClone.find('.platform-enable').remove();
                $cardClone.find('.platform-actions').prepend(
                    '<button type="button" class="button button-primary platform-configure" data-platform="' + platformId + '">Configure</button>'
                );
                $cardClone.find('.platform-actions').append(
                    '<button type="button" class="button button-secondary platform-disable" data-platform="' + platformId + '">Disable</button>'
                );
                
                $enabledGrid.append($cardClone);
                $card.fadeOut(400, function() {
                    $(this).remove();
                });
                
                // Show enabled section if it was hidden
                $('.enabled-platforms-section').show();
                
            } else {
                // Move card back to available section
                const platformCategory = $card.data('category');
                const $availableGrid = $(`.platform-category[data-category="${platformCategory}"] .platform-grid`);
                const $cardClone = $card.clone();
                
                // Update the cloned card for available state
                $cardClone.removeClass('enabled').addClass('available');
                $cardClone.find('.platform-configure, .platform-disable').remove();
                $cardClone.find('.platform-actions').html(
                    '<button type="button" class="button button-primary platform-enable" data-platform="' + platformId + '">Enable</button>'
                );
                
                $availableGrid.append($cardClone);
                $card.fadeOut(400, function() {
                    $(this).remove();
                    
                    // Hide enabled section if no platforms remain
                    if ($('.enabled-grid .platform-card').length === 0) {
                        $('.enabled-platforms-section').hide();
                    }
                });
            }

            // Show success message
            PlatformManager.showNotice(data.message, 'success');
        },

        handleToggleError: function($button, originalText, errorMessage) {
            $button.text(originalText);
            PlatformManager.showNotice(errorMessage, 'error');
        },

        configurePlatform: function(e) {
            e.preventDefault();
            const $button = $(this);
            const platformId = $button.data('platform');
            
            $button.text(triggerNinjaAjax.strings.configuring);
            $button.prop('disabled', true);

            $.ajax({
                url: triggerNinjaAjax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'triggerninja_configure_platform',
                    platform_id: platformId,
                    nonce: triggerNinjaAjax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        PlatformManager.openConfigModal(response.data.title, response.data.form_html, platformId);
                    } else {
                        PlatformManager.showNotice(response.data || triggerNinjaAjax.strings.error, 'error');
                    }
                },
                error: function() {
                    PlatformManager.showNotice(triggerNinjaAjax.strings.error, 'error');
                },
                complete: function() {
                    $button.text('Configure');
                    $button.prop('disabled', false);
                }
            });
        },

        openConfigModal: function(title, formHtml, platformId) {
            const $modal = $('#platform-config-modal');
            $modal.find('.modal-title').text(title);
            $modal.find('.modal-body').html(formHtml);
            $modal.data('platform-id', platformId);
            $modal.show();
            
            // Focus first input
            setTimeout(function() {
                $modal.find('input, select, textarea').first().focus();
            }, 100);
        },

        closeModal: function() {
            $('#platform-config-modal').hide();
        },

        saveConfiguration: function() {
            const $modal = $('#platform-config-modal');
            const platformId = $modal.data('platform-id');
            const $saveButton = $('.modal-save');
            const originalText = $saveButton.text();
            
            // Collect form data
            const formData = {
                action: 'triggerninja_save_platform_config',
                platform_id: platformId,
                nonce: triggerNinjaAjax.nonce
            };
            
            $modal.find('input, select, textarea').each(function() {
                const $field = $(this);
                const name = $field.attr('name');
                if (name) {
                    if ($field.is(':checkbox')) {
                        formData[name] = $field.is(':checked') ? '1' : '0';
                    } else {
                        formData[name] = $field.val();
                    }
                }
            });

            // Show loading state
            $saveButton.text('Saving...').prop('disabled', true);

            // Save via AJAX
            $.ajax({
                url: triggerNinjaAjax.ajaxurl,
                type: 'POST',
                data: formData,
                success: function(response) {
                    if (response.success) {
                        PlatformManager.closeModal();
                        PlatformManager.showNotice(response.data.message, 'success');
                    } else {
                        PlatformManager.showNotice(response.data || triggerNinjaAjax.strings.error, 'error');
                    }
                },
                error: function() {
                    PlatformManager.showNotice(triggerNinjaAjax.strings.error, 'error');
                },
                complete: function() {
                    $saveButton.text(originalText).prop('disabled', false);
                }
            });
        },

        authorizeAweber: function(e) {
            e.preventDefault();
            const $button = $(this);
            const $modal = $('#platform-config-modal');
            const originalText = $button.text();
            
            // Get form values
            const authCode = $modal.find('#triggerninja_aweber_auth_code').val();
            const codeVerifier = $modal.find('#triggerninja_aweber_code_verifier').val();
            
            // Validate input
            if (!authCode || !authCode.trim()) {
                PlatformManager.showNotice('Please enter your AWeber authorization code first.', 'error');
                return;
            }
            
            // Show loading state
            $button.text('Authorizing...').prop('disabled', true);
            
            const nonce = $modal.find('#triggerninja_aweber_nonce').val();

            // Make AJAX request
            $.ajax({
                url: triggerNinjaAjax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'triggerninja_authorize_aweber',
                    auth_code: authCode.trim(),
                    code_verifier: codeVerifier,
                    nonce: nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Update connection status in the modal - find the field wrapper that contains the status
                        $modal.find('.field-wrapper').each(function() {
                            const $wrapper = $(this);
                            const $label = $wrapper.find('label');
                            if ($label.length && $label.text().trim() === 'Connection Status') {
                                // Update the content after the label
                                const $content = $wrapper.find('label').nextAll();
                                $content.remove();
                                $wrapper.append(response.data.status_html);
                            }
                        });
                        
                        // Don't clear the auth code field - keep it for reference
                        // $modal.find('#triggerninja_aweber_auth_code').val('');
                        
                        // Show success message
                        PlatformManager.showNotice(response.data.message, 'success');
                        
                        // Don't hide the authorize button - just disable it and change text
                        $button.text('Authorized').prop('disabled', true).removeClass('button-primary').addClass('button-secondary');
                        
                    } else {
                        PlatformManager.showNotice(response.data.message || 'Authorization failed. Please try again.', 'error');
                    }
                },
                error: function() {
                    PlatformManager.showNotice('Authorization failed. Please check your connection and try again.', 'error');
                },
                complete: function() {
                    $button.text(originalText).prop('disabled', false);
                }
            });
        },

        showNotice: function(message, type) {
            // Create or update notice
            let $notice = $('.triggerninja-notice');
            if ($notice.length === 0) {
                $notice = $('<div class="triggerninja-notice"></div>').insertAfter('.platform-manager-header');
            }
            
            $notice
                .removeClass('notice-success notice-error notice-warning')
                .addClass('notice notice-' + type)
                .html('<p>' + message + '</p>')
                .show();
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                $notice.fadeOut();
            }, 5000);
        }
    };

    // Initialize when DOM is ready
    $(document).ready(function() {
        PlatformManager.init();
    });

})(jQuery);