/**
 * Handles OAuth connection flows for TriggerNinja admin interfaces.
 */
(function($) {
    'use strict';

    var config = window.triggerNinjaAjax || {};
    var ajaxUrl = config.ajaxurl || window.ajaxurl || '';
    var strings = config.strings || {};
    var connections = config.connections || {};
    var fallbackError = strings.error || 'An error occurred. Please try again.';

    function getStrings(key) {
        return connections[key] || {};
    }

    function storeButtonState($button, loadingText) {
        var originalText = $button.data('triggerninja-original-text');
        if (!originalText) {
            originalText = $.trim($button.text());
            $button.data('triggerninja-original-text', originalText);
        }
        if (loadingText) {
            $button.text(loadingText);
        }
        $button.prop('disabled', true);
        return originalText;
    }

    function restoreButtonState($button, originalText) {
        if (typeof originalText !== 'undefined') {
            $button.text(originalText);
        }
        $button.prop('disabled', false);
    }

    function postRequest(payload) {
        if (!ajaxUrl) {
            return $.Deferred(function(defer) {
                defer.reject();
            }).promise();
        }
        return $.post(ajaxUrl, payload);
    }

    function handleAweberConnect(e) {
        e.preventDefault();

        var alerts = getStrings('aweber');
        var $button = $('#aweber-save-code');
        if (!$button.length) {
            return;
        }

        var authCode = $.trim($('#aweber-auth-code').val() || '');
        if (!authCode) {
            window.alert(alerts.missingCode || fallbackError);
            return;
        }

        var codeVerifier = $('#aweber-code-verifier').val();
        if (!codeVerifier) {
            window.alert(alerts.missingVerifier || fallbackError);
            return;
        }

        var codeChallenge = $('#aweber-code-challenge').val();
        var nonce = $('#aweber-auth-nonce').val();
        if (!nonce) {
            window.alert(alerts.missingNonce || fallbackError);
            return;
        }

        var originalText = storeButtonState($button, alerts.connecting || null);
        var success = false;

        postRequest({
            action: 'triggerninja_aweber_exchange_token',
            nonce: nonce,
            auth_code: authCode,
            pkce_data: {
                code_verifier: codeVerifier,
                code_challenge: codeChallenge
            }
        })
        .done(function(response) {
            if (response && response.success) {
                success = true;
                window.alert(alerts.connectSuccess || fallbackError);
                window.location.reload();
                return;
            }

            var message = response && response.data && response.data.message ? response.data.message : (alerts.connectError || fallbackError);
            window.alert((alerts.connectErrorPrefix || '') + message);
        })
        .fail(function() {
            window.alert(alerts.connectError || fallbackError);
        })
        .always(function() {
            if (!success) {
                restoreButtonState($button, originalText);
            }
        });
    }

    function handleAweberDisconnect(e) {
        e.preventDefault();

        var alerts = getStrings('aweber');
        if (alerts.disconnectConfirm && !window.confirm(alerts.disconnectConfirm)) {
            return;
        }

        var nonce = $('#aweber-auth-nonce').val();
        if (!nonce) {
            window.alert(alerts.missingNonce || fallbackError);
            return;
        }

        var $button = $('#aweber-disconnect');
        if (!$button.length) {
            return;
        }
        var originalText = storeButtonState($button, alerts.disconnecting || null);
        var success = false;

        postRequest({
            action: 'triggerninja_aweber_disconnect',
            nonce: nonce
        })
        .done(function(response) {
            if (response && response.success) {
                success = true;
                window.alert(alerts.disconnectSuccess || fallbackError);
                window.location.reload();
                return;
            }
            var message = response && response.data && response.data.message ? response.data.message : (alerts.disconnectError || fallbackError);
            window.alert(message);
        })
        .fail(function() {
            window.alert(alerts.disconnectError || fallbackError);
        })
        .always(function() {
            if (!success) {
                restoreButtonState($button, originalText);
            }
        });
    }

    function handleConstantContactDisconnect(e) {
        e.preventDefault();

        var alerts = getStrings('constantcontact');
        if (alerts.disconnectConfirm && !window.confirm(alerts.disconnectConfirm)) {
            return;
        }

        var nonce = $('#constantcontact-disconnect-nonce').val();
        if (!nonce) {
            window.alert(alerts.disconnectError || fallbackError);
            return;
        }

        var $button = $('#constantcontact-disconnect');
        if (!$button.length) {
            return;
        }
        var originalText = storeButtonState($button, alerts.disconnecting || null);
        var success = false;

        postRequest({
            action: 'triggerninja_constantcontact_disconnect',
            nonce: nonce
        })
        .done(function(response) {
            if (response && response.success) {
                success = true;
                window.alert(alerts.disconnectSuccess || fallbackError);
                window.location.reload();
                return;
            }

            var message = response && response.data && response.data.message ? response.data.message : (alerts.disconnectError || fallbackError);
            window.alert(message);
        })
        .fail(function() {
            window.alert(alerts.disconnectError || fallbackError);
        })
        .always(function() {
            if (!success) {
                restoreButtonState($button, originalText);
            }
        });
    }

    $(document).on('click', '#aweber-save-code', handleAweberConnect);
    $(document).on('keydown', '#aweber-auth-code', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            $('#aweber-save-code').trigger('click');
        }
    });
    $(document).on('click', '#aweber-disconnect', handleAweberDisconnect);
    $(document).on('click', '#constantcontact-disconnect', handleConstantContactDisconnect);

})(jQuery);
