<?php
/**
 * Plugin Name: TrafficWatchdog Sales Bot
 * Plugin URI: https://trafficwatchdog.pl/pl/lp/sales-bot
 * Description: Wtyczka TrafficWatchdog Sales Bot
 * Version: 1.01
 * Author: TrafficWatchdog Team
 * Author URI: https://trafficwatchdog.pl
 * License: GPL2
 * Text Domain: trafficwatchdog-sales-bot
 * Domain Path: /languages
 * Requires at least: 6.2
 * Requires PHP: 7.4
 * 
 */

namespace trafficwatchdogpl;

if (!defined('ABSPATH')) {
    exit;
}

require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

define("TRAFFICWATCHDOGPL_SALES_BOT_PLUGIN_FILE_URL", __FILE__);

class TwdSalesBot {
    private static $id = 'twd_sb';

    private static $instance = null;

    const TWD_DATA_TABLE_NAME = 'trafficwatchdog_instances_data';

    private function __construct()
    {
        
    }

    /**
     * Return plugin instance
     * @return TwdSalesBot
     */
    public static function twd_sb_get_instance()
    {
        if (empty(self::$instance))
        {
            self::$instance = new self();
            self::$instance->twd_sb_init();
        }

        return self::$instance;
    }

    public function twd_sb_init()
    {
        add_action('admin_menu', [$this, 'add_plugin_menu']);
        add_action('wp_footer', [$this, 'add_footer_script']);

        register_deactivation_hook(TRAFFICWATCHDOGPL_SALES_BOT_PLUGIN_FILE_URL, [$this, 'uninstall']);
    }

    /**
     * TrafficWatchdog code linking
     * @return void
     */
    public function add_footer_script()
    {
        wp_enqueue_script('twd', 'https://trafficscanner.pl/ws2/script/twd.js', [], null, ['in_footer' => true]);
    }

    /**
     * The plugin menu
     * @return void
     */
    public function add_plugin_menu() {
        add_menu_page(
            __('TrafficWatchdog Sales Bot', 'trafficwatchdog-sales-bot'),
            __('TrafficWatchdog Sales Bot', 'trafficwatchdog-sales-bot'),
            'manage_options',
            'trafficwatchdog-sales-bot',
            [$this, 'twd_sb_menu_main_page'],
            plugin_dir_url( __FILE__ ) . 'includes/images/twd-icon.png',
            20
        );
    
        add_submenu_page(
            'trafficwatchdog-sales-bot',
            __('Panel TrafficWatchdog', 'trafficwatchdog-sales-bot'),
            __('Idź do panelu TrafficWatchdog', 'trafficwatchdog-sales-bot'),
            'manage_options',       // Uprawnienia
            $this->twd_sb_panel_url(),
            null
        );
    }

    public function twd_sb_panel_url()
    {
        $instance_id = $this->twd_sb_db_get_instance_id();
        if (empty($instance_id)) {
            $instance_id = 'empty';
        }

        return sprintf('https://panel.trafficwatchdog.pl/woocommerceapp/?woocommerce_instance_id=%s&apptype=salesbot&utm_medium=salesbot', $instance_id);
    }

    public function twd_sb_menu_main_page() {
        ?>
        <div class="wrap">
            <h2 style="text-align: center;"><?php echo esc_html_e('Twój TrafficWatchdog Sales Bot został zainstalowany!', 'trafficwatchdog-sales-bot') ?></h2>
            <p>
                <?php echo esc_html_e('Cieszymy się, że korzystasz z naszych usług! Jeżeli jeszcze tego nie zrobiłeś zarejestruj się w naszym panelu aby skonfigurować swojego TrafficWatchdog Sales Bota!', 'trafficwatchdog-sales-bot') ?>
            </p>
            <p>
                <a href="<?php echo esc_url($this->twd_sb_panel_url()) ?>" target="blank"><?php echo esc_html_e('Idź do panelu TrafficWatchdog', 'trafficwatchdog-sales-bot') ?></a>
            </p>
        </div>
        <?php
    }

    /**
     * Callback after plugin activation
     * @return boolean
     */
    public function twd_sb_callback($action = 'install')
    {
        $url = 'https://panel.trafficwatchdog.pl/woocommerceapp-init';
        $post_params = [
            'timestamp' => time(),
            'url' => get_site_url(),
            'action' => $action,
            'apptype' => 'salesbot',
            'version' => 'wordpress',
            'instance_id' => $this->twd_sb_db_get_instance_id()
        ];

        $args = [
            'headers'   => [
                'Content-Type' => 'application/x-www-form-urlencoded; charset=UTF-8'
            ],
            'body'      => $post_params,
        ];

        $http_client = new \WP_Http();

        return is_array($http_client->post($url, $args));
    }

    /**
     * Creating a table with data used by the plugin and initializing it.
     * @return void
     */
    public function twd_sb_db_table_create()
    {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE %s (
          id mediumint(9) NOT NULL AUTO_INCREMENT,
          `key` varchar(128) NOT NULL,
          `value` varchar(128) NOT NULL,
          PRIMARY KEY  (id),
          UNIQUE (`key`)
        )%s;";

        if ($this->twd_sb_db_get_instance_id() === null) {
            $this->twd_sb_db_table_drop();
        
            dbDelta( sprintf($sql, $wpdb->prefix . static::TWD_DATA_TABLE_NAME, $charset_collate) );

            $this->twd_sb_db_table_init();
        }

        $this->twd_sb_instance_count_manager('install');
    }

    /**
     * Removing table with data used by the plugin.
     * @return void
     */
    public function twd_sb_db_table_drop()
    {
        global $wpdb;

        if (!empty($this->twd_sb_db_get_instance_count()) && $this->twd_sb_db_get_instance_count() > 0) {
            return;
        }

        $preparedQuery = $wpdb->prepare("DROP TABLE IF EXISTS %i;", $wpdb->prefix . static::TWD_DATA_TABLE_NAME);

        $wpdb->query($preparedQuery);
    }

    /**
     * Getting instance count
     * @return ?int
     */
    public function twd_sb_db_get_instance_count()
    {
        global $wpdb;
        
        $preparedQuery = $wpdb->prepare("SELECT `value` FROM %i WHERE `key` = 'instance_count' ORDER BY `id` ASC LIMIT 1;", $wpdb->prefix . static::TWD_DATA_TABLE_NAME);
        
        $response = $wpdb->get_results($preparedQuery, ARRAY_A);

        return (!empty($response) && isset($response[0]['value'])) ? $response[0]['value'] : null;
    }

    /**
     * Increment or decrement instance count depending of the type action
     * @param mixed $action string
     * @return void
     */
    public function twd_sb_instance_count_manager($action = 'install')
    {
        global $wpdb;

        if ($this->twd_sb_db_get_instance_count() === null) {
            $wpdb->insert( 
                $wpdb->prefix . static::TWD_DATA_TABLE_NAME, 
                array( 
                    'key' => 'instance_count', 
                    'value' => 0, 
                ) 
            );
        }

        $instanceCount = (int)$this->twd_sb_db_get_instance_count();
        if (!is_numeric($instanceCount)) {
            $instanceCount = 0;
        }

        if ($action == 'install') {
            $instanceCount += 1;

            $wpdb->update(
                $wpdb->prefix . static::TWD_DATA_TABLE_NAME,
                array (
                    'value' => $instanceCount
                ),
                array (
                    'key' => 'instance_count'
                )
            );
        } else if ($action == 'uninstall') {
            $instanceCount -= 1;

            if ($instanceCount < 0) {
                $instanceCount = 0;
            }

            $wpdb->update(
                $wpdb->prefix . static::TWD_DATA_TABLE_NAME,
                array (
                    'value' => $instanceCount
                ),
                array (
                    'key' => 'instance_count'
                )
            );
        }
    }

    public function uninstall()
    {
        $this->twd_sb_callback('uninstall');
        $this->twd_sb_instance_count_manager('uninstall');
        $this->twd_sb_db_table_drop();
    }

    /**
     * Getting instance ID
     * @return ?string
     */
    public function twd_sb_db_get_instance_id()
    {
        global $wpdb;

        $preparedQuery = $wpdb->prepare("SELECT `value` FROM %i WHERE `key` = 'instance_id' ORDER BY `id` ASC LIMIT 1;", $wpdb->prefix . static::TWD_DATA_TABLE_NAME);
        
        $response = $wpdb->get_results($preparedQuery, ARRAY_A);

        return (!empty($response) && isset($response[0]['value'])) ? $response[0]['value'] : null;
    }

    /**
     * Generating instance ID and insert it to table.
     * @return void
     */
    public function twd_sb_db_table_init()
    {
        global $wpdb;

        $wpdb->insert( 
            $wpdb->prefix . static::TWD_DATA_TABLE_NAME, 
            array( 
                'key' => 'instance_id', 
                'value' => self::generate_uuid(TRAFFICWATCHDOGPL_SALES_BOT_PLUGIN_FILE_URL.get_site_url()), 
            ) 
        );
    }

    /**
     * Creating simple UUID.
     * @param ?string $data
     * 
     * @return string
     */
    private static function generate_uuid($data = null) {
        return md5($data);
    }
}



use trafficwatchdogpl;

/**
 * Instantiate plugin.
 */

 if (!function_exists(sprintf('%s\twd_sb_plugin_instantiate', __NAMESPACE__))) {
    function twd_sb_plugin_instantiate()
    {

        $twd_sb_plugin = TwdSalesBot::twd_sb_get_instance();
    }
}
add_action('plugins_loaded', sprintf('%s\twd_sb_plugin_instantiate', __NAMESPACE__));

/**
 * Activation hook.
 */

 if (!function_exists(sprintf('%s\twd_sb_plugin_activation', __NAMESPACE__))) {
    function twd_sb_plugin_activation()
    {

        $twd_sb_plugin = TwdSalesBot::twd_sb_get_instance();
        $twd_sb_plugin->twd_sb_db_table_create();
        $twd_sb_plugin->twd_sb_callback();
    }
}
register_activation_hook(TRAFFICWATCHDOGPL_SALES_BOT_PLUGIN_FILE_URL, sprintf('%s\twd_sb_plugin_activation', __NAMESPACE__));
?>
