<?php
namespace WpIntegrity\TradeMate\Product;

use WP_Customize_Color_Control;
use WpIntegrity\TradeMate\Customizer\RadioIconControl;
use WpIntegrity\TradeMate\Helper;

/**
 * Stock Functions Manager Class.
 *
 * Manages default product stock and sold individually settings.
 * 
 * @since 1.0.1
 */
class StockBadge {

    /**
     * Constructor function.
     *
     */
    public function __construct() {
        if( Helper::get_option( 'stock_badge', 'trademate_product' ) === 'on' ) {
            add_action( 'trademate_customize_register', [ $this, 'stock_badge_customizer_options' ] );
            $this->stock_badge_placement_hooks();
            add_action( 'wp_enqueue_scripts', [ $this, 'badge_dynamic_styles' ] );
        }
    }

    /**
     * Out of Stock section customizer options 
     *
     * @param object $wp_customize
     * @return void
     * @since 1.0.1
     */
    public function stock_badge_customizer_options( $wp_customize ) {
        $wp_customize->add_section(
            'trademate_stock_badge',
            [
                'title'    => __( 'Stock Badge', 'trademate' ),
                'priority' => 10,
                'panel'    => 'trademate',
            ]
        );

        // Add Setting for Out of Stock Badge Text
        $wp_customize->add_setting(
            'out_of_stock_badge_text', 
            [
                'default'   => __( 'Out of Stock', 'trademate' ),
                'sanitize_callback' => 'sanitize_text_field',
            ]
        );

        // Add Control for Out of Stock Badge Text
        $wp_customize->add_control(
            'out_of_stock_badge_text_control', 
            [
                'label'    => __( '"Out of Stock" Badge Text', 'trademate' ),
                'section'  => 'trademate_stock_badge',
                'settings' => 'out_of_stock_badge_text',
                'type'     => 'text',
            ]
        );

        // Add Setting for Badge Background Color
        $wp_customize->add_setting( 
            'out_of_stock_badge_bg_color', 
            [
                'default'           => '#ff0000',
                'sanitize_callback' => 'sanitize_hex_color',
            ]
        );

        // Add Control for Badge Background Color
        $wp_customize->add_control( 
            new WP_Customize_Color_Control( 
                $wp_customize, 
                'out_of_stock_badge_bg_color_control', 
                [
                    'label'    => __( 'Out of Stock Badge Background Color', 'trademate' ),
                    'section'  => 'trademate_stock_badge',
                    'settings' => 'out_of_stock_badge_bg_color',
                ]
            )
        );

        // Add Setting for Badge Text Color
        $wp_customize->add_setting(
            'out_of_stock_badge_text_color', 
            [
                'default'   => '#ffffff',
                'sanitize_callback' => 'sanitize_hex_color',
            ]
        );

        // Add Control for Badge Text Color
        $wp_customize->add_control(
            new WP_Customize_Color_Control(
                $wp_customize, 
                'out_of_stock_badge_text_color_control', 
                [
                    'label'    => __('Out of Stock Badge Text Color', 'trademate'),
                    'section'  => 'trademate_stock_badge',
                    'settings' => 'out_of_stock_badge_text_color',
                ]
            )
        );

         // Add Setting for Backorder Badge Text
         $wp_customize->add_setting(
            'backorder_badge_text', 
            [
                'default'   => __( 'Available on Backorder', 'trademate' ),
                'sanitize_callback' => 'sanitize_text_field',
            ]
        );

        // Add Control for Backorder Badge Text
        $wp_customize->add_control(
            'backorder_badge_text_control', 
            [
                'label'    => __( '"Backorder" Badge Text', 'trademate' ),
                'section'  => 'trademate_stock_badge',
                'settings' => 'backorder_badge_text',
                'type'     => 'text',
            ]
        );

        // Backorder Badge Background Color Setting
        $wp_customize->add_setting( 
            'backorder_badge_bg_color', 
            [
                'default'           => '#dd9933',
                'sanitize_callback' => 'sanitize_hex_color',
            ]
        );

        // Backorder Badge Background Color Controller
        $wp_customize->add_control( 
            new WP_Customize_Color_Control( 
                $wp_customize, 
                'backorder_badge_bg_color_control', 
                [
                    'label'    => __( 'Backorder Badge Background Color', 'trademate' ),
                    'section'  => 'trademate_stock_badge',
                    'settings' => 'backorder_badge_bg_color',
                ]
            )
        );

        // Add Setting for Badge Text Color
        $wp_customize->add_setting(
            'backorder_badge_text_color', 
            [
                'default'   => '#ffffff',
                'sanitize_callback' => 'sanitize_hex_color',
            ]
        );

        // Add Control for Badge Text Color
        $wp_customize->add_control(
            new WP_Customize_Color_Control(
                $wp_customize, 
                'backorder_badge_text_color_control', 
                [
                    'label'    => __('Backorder Badge Text Color', 'trademate'),
                    'section'  => 'trademate_stock_badge',
                    'settings' => 'backorder_badge_text_color',
                ]
            )
        );

        // Badge Width
        $wp_customize->add_setting(
            'stock_badge_width',
            [
                'default'           => '50',
                'sanitize_callback' => 'sanitize_text_field',
            ]
        );
        $wp_customize->add_control(
            'stock_badge_width_control',
            [
                'label'    => __('Badge Width', 'trademate'),
                'section'  => 'trademate_stock_badge',
                'settings' => 'stock_badge_width',
                'type'     => 'select',
                'choices'  => [
                    '30'  => __('30%', 'trademate'),
                    '50'  => __('50%', 'trademate'),
                    '80'  => __('80%', 'trademate'),
                    '100' => __('100%', 'trademate'),
                ],
            ]
        );

        // Stock Badge Postion X (Horizontal)
        $wp_customize->add_setting(
            'stock_badge_position_x', 
            [
                'default'           => 'flex-start',
                'sanitize_callback' => 'sanitize_text_field',
            ]
        );

        $wp_customize->add_control(
            new RadioIconControl(
                $wp_customize,
                'stock_badge_position_control_x',
                [
                    'label'       => __('Horizontal Position', 'trademate'),
                    'description' => __('Set Stock Badge Horizontal Postion for the Shop/Product Archive', 'trademate'),
                    'section'     => 'trademate_stock_badge',
                    'settings'    => 'stock_badge_position_x',
                    'choices'     => [
                        'flex-start' => 'dashicons-align-left',
                        'center'     => 'dashicons-align-center',
                        'flex-end'   => 'dashicons-align-right',
                    ]
                ]
            )
        );

        // Stock Badge Postion Y (Vertical)
        $wp_customize->add_setting(
            'stock_badge_position_y', 
            [
                'default'           => 'flex-start',
                'sanitize_callback' => 'sanitize_text_field',
            ]
        );

        $wp_customize->add_control(
            new RadioIconControl(
                $wp_customize,
                'stock_badge_position_control_y',
                [
                    'label'       => __('Vertical Position', 'trademate'),
                    'description' => __('Set Stock Badge Veritcal Postion for the Shop/Product Archive', 'trademate'),
                    'section'     => 'trademate_stock_badge',
                    'settings'    => 'stock_badge_position_y',
                    'choices'     => [
                        'flex-start' => 'dashicons-arrow-up-alt2',
                        'center'     => 'dashicons-editor-code',
                        'flex-end'   => 'dashicons-arrow-down-alt2',
                    ]
                ]
            )
        );
    }

    /**
     * Show Stock Badge on the Shop Page
     *
     * @return void
     * @since 1.0.1
     */
    private function stock_badge_placement_hooks() {
        if( Helper::is_theme_active( 'blocksy' ) ) {
            add_action( 'blocksy:woocommerce:product-card:thumbnail:end', [ $this, 'add_stock_badge' ] );
        } elseif( Helper::is_theme_active( 'oceanwp' ) ) {
            add_action( 'ocean_after_archive_product_image', [ $this, 'add_stock_badge' ] );
        } else {
            add_action( 'woocommerce_before_shop_loop_item', [ $this, 'image_parent_wrapper_open' ] );
            add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'add_stock_badge' ] );
            add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'image_parent_wrapper_close' ] );
        }
    }

    /**
     * Wrapped Image with a parent div (Opener)
     *
     * @return void
     * @since 1.0.1
     */
    public function image_parent_wrapper_open() {
        echo '<div class="tm-woocommere-image-parent-wrapper" style="position: relative;">';
    }

    /**
     * Wrapped Image with a parent div (Closer)
     *
     * @return void
     * @since 1.0.1
     */
    public function image_parent_wrapper_close() {
        echo '</div>';
    }

    /**
     * Out of Stock Badge Frontend
     *
     * @return void
     * @since 1.0.1
     */
    public function add_stock_badge() {
        global $product;

        if ( ! $product ) {
            return;
        }

        // Check the stock status
        $stock_status = $product->get_stock_status(); // Returns 'instock', 'outofstock', or 'onbackorder'

        // Determine badge text
        if ( $stock_status === 'outofstock' ) {
            $badge_text = get_theme_mod( 'out_of_stock_badge_text', __('Out of Stock', 'trademate') );
            $badge_class = 'out-of-stock';
        } elseif ( $stock_status === 'onbackorder' ) {
            $badge_text = get_theme_mod( 'backorder_badge_text', __('Available on Backorder', 'trademate') );
            $badge_class = 'backorder';
        } else {
            return;
        }

        // Output the badge
        echo '
            <div class="tm-badge-wrapper">
                <div class="tm-badge-container">
                    <div class="tm-'.esc_attr($badge_class).'-badge">' . esc_html($badge_text) . '</div>
                </div>
            </div>
        ';
    }

    /**
     * Dynamic Styles for the Stock Badge
     *
     * @return void
     * @since 1.0.1
     */
    public function badge_dynamic_styles() {
        wp_register_style( 'trademate-stock-badge', false, [], TRADEMATE_VERSION );
        wp_enqueue_style( 'trademate-stock-badge' );

        $oos_bg_color   = get_theme_mod( 'out_of_stock_badge_bg_color', '#ff0000' );
        $oos_text_color = get_theme_mod( 'out_of_stock_badge_text_color', '#ffffff' );
        $obo_bg_color   = get_theme_mod( 'backorder_badge_bg_color', '#dd9933' );
        $obo_text_color = get_theme_mod( 'backorder_badge_text_color', '#ffffff' );
        $badge_width    = get_theme_mod( 'stock_badge_width', '50' );
        $position_x     = get_theme_mod( 'stock_badge_position_x', 'flex-start' );
        $position_y     = get_theme_mod( 'stock_badge_position_y', 'flex-start' );

        $inline_style = "
            .tm-badge-wrapper {
                width: 100%;
                height: 100%;
                position: absolute;
                top: 0;
                display: flex;
                justify-content: {$position_x};
                align-items: {$position_y};
            }
            .tm-badge-container {
                width: {$badge_width}%;
            }
            .tm-out-of-stock-badge {
                padding: 10px;
                color: {$oos_text_color};
                background-color: {$oos_bg_color};
                text-align: center;
            }
            .tm-backorder-badge {
                padding: 10px;
                color: {$obo_text_color};
                background-color: {$obo_bg_color};
                text-align: center;
            }
        ";

        // Inject the inline CSS
        wp_add_inline_style('trademate-stock-badge', $inline_style);
    }
}
