<?php
/**
 * Totals Report for WooCommerce Plugin
 *
 * @package           TotalsReportForWooCommerce
 * @author            BigDrop.gr
 * @copyright         2024 BigDrop.gr
 * @license           GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name:       Totals Report for WooCommerce
 * Plugin URI:        https://bigdrop.gr/totals-report-for-woocommerce
 * Description:       Adds a comprehensive Totals page under the WooCommerce admin menu to display product counts, stock information, and export capabilities.
 * Version:           1.0.2
 * Requires at least: 5.0
 * Requires PHP:      7.2
 * Requires Plugins:  woocommerce
 * Author:            BigDrop.gr
 * Author URI:        https://bigdrop.gr
 * Text Domain:       totals-report-for-woocommerce
 * Domain Path:       /languages
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * WC requires at least: 5.0.0
 * WC tested up to:   9.7.1
 * WooCommerce HPOS Compatible: true
 */

// Prevent direct file access.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define plugin constants.
define( 'TRWC_VERSION', '1.0.2' );
define( 'TRWC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'TRWC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'TRWC_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Check if WooCommerce is active
 */
function trwc_is_woocommerce_active() {
    $active_plugins = apply_filters( 'active_plugins', get_option( 'active_plugins' ) );
    
    // Check for single site
    if ( in_array( 'woocommerce/woocommerce.php', $active_plugins, true ) ) {
        return true;
    }
    
    // Check for multisite
    if ( is_multisite() ) {
        $network_active_plugins = array_keys( get_site_option( 'active_sitewide_plugins', [] ) );
        if ( in_array( 'woocommerce/woocommerce.php', $network_active_plugins, true ) ) {
            return true;
        }
    }
    
    return false;
}

/**
 * Add global admin notice at the top of the page
 */
function trwc_admin_notice() {
    global $pagenow;
    
    // Only show on the plugins page
    if ( $pagenow !== 'plugins.php' ) {
        return;
    }
    
    // Only show if WooCommerce is not active
    if ( trwc_is_woocommerce_active() ) {
        return;
    }
    
    // Use the same format as in the screenshot with red left border
    echo '<div class="notice notice-error" style="border-left-color: #dc3232;">';
    echo '<p>' . esc_html__( 'Totals Report for WooCommerce cannot be activated because it requires the WooCommerce plugin to be installed and active.', 'totals-report-for-woocommerce' ) . '</p>';
    echo '</div>';
}
add_action( 'admin_notices', 'trwc_admin_notice' );

/**
 * Add notice below plugin in the list
 */
function trwc_after_plugin_row( $plugin_file, $plugin_data, $status ) {
    if ( trwc_is_woocommerce_active() ) {
        return;
    }
    
    $wp_list_table = _get_list_table( 'WP_Plugins_List_Table' );
    $colspan = $wp_list_table->get_column_count();
    
    echo '<tr class="plugin-update-tr active">';
    echo '<td colspan="' . esc_attr( $colspan ) . '" class="plugin-update colspanchange">';
    echo '<div class="update-message notice inline notice-error notice-alt">';
    echo '<p>' . esc_html__( 'This plugin cannot be activated because required plugins are missing or inactive.', 'totals-report-for-woocommerce' ) . '</p>';
    echo '</div>';
    echo '</td>';
    echo '</tr>';
}
add_action( 'after_plugin_row_' . plugin_basename( __FILE__ ), 'trwc_after_plugin_row', 10, 3 );

/**
 * Modify plugin description to add "Requires: WooCommerce"
 */
function trwc_modify_plugin_description( $plugin_meta, $plugin_file, $plugin_data, $status ) {
    if ( $plugin_file === plugin_basename( __FILE__ ) ) {
        // Add "Requires: WooCommerce" text
        echo '<br>';
        echo esc_html__( 'Requires:', 'totals-report-for-woocommerce' ) . ' <a href="' . esc_url( admin_url( 'plugin-install.php?s=woocommerce&tab=search&type=term' ) ) . '">WooCommerce</a>';
    }
    
    return $plugin_meta;
}
add_action( 'plugin_row_meta', 'trwc_modify_plugin_description', 10, 4 );

/**
 * Prevent plugin activation when dependencies are not met
 */
function trwc_check_dependencies() {
    if ( ! trwc_is_woocommerce_active() ) {
        // Make sure Totals Report for WooCommerce is not active
        if ( is_plugin_active( plugin_basename( __FILE__ ) ) ) {
            deactivate_plugins( plugin_basename( __FILE__ ) );
            
            // Prevent WordPress from redirecting to the activation page
            if ( isset( $_GET['activate'] ) && ! isset( $_REQUEST['_wpnonce'] ) ) {
                // Check if this is an admin request
                if ( is_admin() ) {
                    wp_verify_nonce( null );
                }
                unset( $_GET['activate'] );
            }
        }
    }
}
add_action( 'admin_init', 'trwc_check_dependencies' );

/**
 * Activation hook
 */
register_activation_hook( __FILE__, 'trwc_activate' );

function trwc_activate() {
    // Check if WooCommerce is active
    if ( ! trwc_is_woocommerce_active() ) {
        // Deactivate our plugin
        deactivate_plugins( plugin_basename( __FILE__ ) );
        
        // Stop activation and show error
        wp_die( esc_html__( 'Totals Report for WooCommerce requires WooCommerce to be installed and active. Please install and activate WooCommerce first, then activate this plugin.', 'totals-report-for-woocommerce' ) );
    }
    
    // Normal activation tasks
    $current_version = get_option( 'trwc_version' );
    
    // First-time installation
    if ( ! $current_version ) {
        update_option( 'trwc_version', TRWC_VERSION );
    }
    
    // Version update logic
    if ( version_compare( $current_version, TRWC_VERSION, '<' ) ) {
        update_option( 'trwc_version', TRWC_VERSION );
    }
}

// Only proceed if WooCommerce is active
if ( ! trwc_is_woocommerce_active() ) {
    return;
}

// Include necessary files.
require_once TRWC_PLUGIN_DIR . 'includes/class-trwc-report.php';
require_once TRWC_PLUGIN_DIR . 'includes/class-trwc-report-export.php';
require_once TRWC_PLUGIN_DIR . 'includes/class-trwc-report-admin.php';

// Initialize the plugin.
function trwc_init() {
    // No need for load_plugin_textdomain since WordPress 4.6+ for plugins hosted on WordPress.org

    // Initialize classes
    TRWC_Report::get_instance();
    TRWC_Report_Export::get_instance();
    TRWC_Report_Admin::get_instance();
}
add_action( 'plugins_loaded', 'trwc_init' );

// Add HPOS compatibility.
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );

// Deactivation hook.
register_deactivation_hook( __FILE__, 'trwc_deactivate' );

/**
 * Plugin deactivation routine.
 */
function trwc_deactivate() {
    // Cleanup tasks if needed
}