<?php
/**
 * Core reporting functionality for Totals Report for WooCommerce
 *
 * @package TotalsReportForWooCommerce
 */

// Prevent direct file access.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * TRWC_Report Class
 * 
 * Handles core reporting logic and data collection
 */
class TRWC_Report {
    /**
     * Singleton instance
     *
     * @var TRWC_Report
     */
    private static $instance = null;

    /**
     * Cached report data
     *
     * @var array
     */
    private $cached_report = null;

    /**
     * Singleton pattern
     *
     * @return TRWC_Report
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'woocommerce_flush_cache', array( $this, 'clear_cached_report' ) );
    }

    /**
     * Generate comprehensive product report
     *
     * @param array $args Optional arguments for filtering.
     * @return array Detailed product report
     */
    public function generate_report( $args = array() ) {
        // Check if cached report exists and is still valid
        if ( null !== $this->cached_report ) {
            return $this->cached_report;
        }

        // Default query arguments
        $default_args = array(
            'post_type'      => 'product',
            'posts_per_page' => -1,
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'cache_results'  => true,
        );
        $query_args = wp_parse_args( $args, $default_args );

        // Get product IDs
        $product_ids = get_posts( $query_args );

        // Initialize report data
        $report_data = $this->initialize_report_structure();

        // Process each product
        foreach ( $product_ids as $product_id ) {
            $product = wc_get_product( $product_id );
            if ( ! $product ) {
                continue;
            }

            $this->process_product( $product, $report_data );
        }

        // Calculate additional metrics
        $this->calculate_derived_metrics( $report_data );

        // Cache the report
        $this->cached_report = $report_data;

        return $report_data;
    }

    /**
     * Initialize basic report structure
     *
     * @return array Initial report structure
     */
    private function initialize_report_structure() {
        return array(
            'totals' => array(
                'total_products'             => 0,
                'available_products'         => 0,
                'out_of_stock_products'      => 0,
                'low_stock_products'         => 0,
                'backorder_products'         => 0,
                'total_variations'           => 0,
                'total_stock_quantity'       => 0,
            ),
            'product_types' => array(),
            'categories'    => array(),
            'detailed_data' => array(),
        );
    }

    /**
     * Process individual product
     *
     * @param WC_Product $product Product object.
     * @param array      &$report_data Reference to report data array.
     */
    private function process_product( $product, &$report_data ) {
        // Increment total products
        $report_data['totals']['total_products']++;

        // Track product type
        $product_type = $product->get_type();
        $report_data['product_types'][ $product_type ] = 
            isset( $report_data['product_types'][ $product_type ] ) 
            ? $report_data['product_types'][ $product_type ] + 1 
            : 1;

        // Track product stock status for later use with categories
        $is_in_stock = $product->is_in_stock();
        $is_variable = $product->is_type('variable');
        $is_on_backorder = $product->is_on_backorder();
        $is_low_stock = $this->is_product_low_stock($product);
        
        // Process product categories
        $this->process_product_categories( $product, $report_data, $is_in_stock, $is_variable, $is_low_stock, $is_on_backorder );

        // Handle different product types
        if ( $product->is_type( 'simple' ) ) {
            $this->process_simple_product( $product, $report_data );
        } elseif ( $product->is_type( 'variable' ) ) {
            $this->process_variable_product( $product, $report_data );
        }
    }

    /**
     * Check if a product is low in stock
     *
     * @param WC_Product $product Product object.
     * @return bool Whether the product is low in stock.
     */
    private function is_product_low_stock( $product ) {
        if ( ! $product->managing_stock() ) {
            return false;
        }
        
        $stock_quantity = $product->get_stock_quantity();
        if ( $stock_quantity === null ) {
            return false;
        }
        
        // Get WooCommerce low stock threshold
        $low_stock_amount = get_option( 'woocommerce_notify_low_stock_amount', 2 );
        
        // Product-specific low stock threshold
        $product_low_stock_threshold = $product->get_low_stock_amount();
        if ( '' !== $product_low_stock_threshold ) {
            $low_stock_amount = $product_low_stock_threshold;
        }
        
        return $stock_quantity <= $low_stock_amount && $stock_quantity > 0;
    }

    /**
     * Process simple product
     *
     * @param WC_Product $product Product object.
     * @param array      &$report_data Reference to report data array.
     */
    private function process_simple_product( $product, &$report_data ) {
        if ( $product->is_in_stock() ) {
            $report_data['totals']['available_products']++;
            
            // Check for low stock
            if ( $this->is_product_low_stock($product) ) {
                $report_data['totals']['low_stock_products']++;
            }
            
            // Check for backorder
            if ( $product->is_on_backorder() ) {
                $report_data['totals']['backorder_products']++;
            }
        } else {
            $report_data['totals']['out_of_stock_products']++;
        }

        $stock_qty = $product->get_stock_quantity();
        if ( $stock_qty !== null ) {
            $report_data['totals']['total_stock_quantity'] += intval( $stock_qty );
        }
    }

    /**
     * Process variable product
     *
     * @param WC_Product $product Product object.
     * @param array      &$report_data Reference to report data array.
     */
    private function process_variable_product( $product, &$report_data ) {
        $variation_ids = $product->get_children();
        $has_in_stock_variation = false;
        $has_low_stock_variation = false;
        $has_backorder_variation = false;

        foreach ( $variation_ids as $variation_id ) {
            $variation = wc_get_product( $variation_id );
            if ( ! $variation ) {
                continue;
            }

            // Count variations
            $report_data['totals']['total_variations']++;

            // Check variation stock
            if ( $variation->is_in_stock() ) {
                $has_in_stock_variation = true;
                
                // Check for low stock
                if ( $this->is_product_low_stock($variation) ) {
                    $has_low_stock_variation = true;
                }
                
                // Check for backorder
                if ( $variation->is_on_backorder() ) {
                    $has_backorder_variation = true;
                }
            }

            // Add variation stock
            $variation_stock_qty = $variation->get_stock_quantity();
            if ( $variation_stock_qty !== null ) {
                $report_data['totals']['total_stock_quantity'] += intval( $variation_stock_qty );
            }
        }

        // Mark product availability
        if ( $has_in_stock_variation ) {
            $report_data['totals']['available_products']++;
            
            if ( $has_low_stock_variation ) {
                $report_data['totals']['low_stock_products']++;
            }
            
            if ( $has_backorder_variation ) {
                $report_data['totals']['backorder_products']++;
            }
        } else {
            $report_data['totals']['out_of_stock_products']++;
        }
    }

    /**
     * Process product categories
     *
     * @param WC_Product $product     Product object.
     * @param array      &$report_data Reference to report data array.
     * @param bool       $is_in_stock  Whether the product is in stock.
     * @param bool       $is_variable  Whether the product is variable.
     * @param bool       $is_low_stock Whether the product is low in stock.
     * @param bool       $is_on_backorder Whether the product is on backorder.
     */
    private function process_product_categories( $product, &$report_data, $is_in_stock, $is_variable, $is_low_stock, $is_on_backorder ) {
        $category_ids = $product->get_category_ids();
        foreach ( $category_ids as $category_id ) {
            $category = get_term( $category_id, 'product_cat' );
            if ( $category && ! is_wp_error( $category ) ) {
                // Use category name as the key
                $category_key = $category->name;
                
                // Initialize category data if not exists
                if (!isset($report_data['categories'][$category_key])) {
                    $report_data['categories'][$category_key] = array(
                        'total' => 0,
                        'available' => 0,
                        'out_of_stock' => 0,
                        'low_stock' => 0,
                        'backorder' => 0,
                        'variable' => 0
                    );
                }
                
                // Update category counts
                $report_data['categories'][$category_key]['total']++;
                
                if ($is_in_stock) {
                    $report_data['categories'][$category_key]['available']++;
                    
                    if ($is_low_stock) {
                        $report_data['categories'][$category_key]['low_stock']++;
                    }
                    
                    if ($is_on_backorder) {
                        $report_data['categories'][$category_key]['backorder']++;
                    }
                } else {
                    $report_data['categories'][$category_key]['out_of_stock']++;
                }
                
                if ($is_variable) {
                    $report_data['categories'][$category_key]['variable']++;
                }
            }
        }
    }

    /**
     * Calculate derived metrics
     *
     * @param array &$report_data Reference to report data array.
     */
    private function calculate_derived_metrics( &$report_data ) {
        // Additional calculations can be added here
        $report_data['metrics'] = array(
            'available_percentage' => 
                $report_data['totals']['total_products'] > 0 
                ? round( 
                    ($report_data['totals']['available_products'] / $report_data['totals']['total_products']) * 100, 
                    2 
                ) 
                : 0,
            'average_stock_per_product' => 
                $report_data['totals']['total_products'] > 0 
                ? round( 
                    $report_data['totals']['total_stock_quantity'] / $report_data['totals']['total_products'], 
                    2 
                ) 
                : 0,
            'low_stock_percentage' =>
                $report_data['totals']['available_products'] > 0
                ? round(
                    ($report_data['totals']['low_stock_products'] / $report_data['totals']['available_products']) * 100,
                    2
                )
                : 0,
            'backorder_percentage' =>
                $report_data['totals']['available_products'] > 0
                ? round(
                    ($report_data['totals']['backorder_products'] / $report_data['totals']['available_products']) * 100,
                    2
                )
                : 0,
        );
    }

    /**
     * Clear cached report
     */
    public function clear_cached_report() {
        $this->cached_report = null;
    }
}