<?php
/**
 * Export functionality for Totals Report for WooCommerce
 *
 * @package TotalsReportForWooCommerce
 */

// Prevent direct file access.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * TRWC_Report_Export Class
 * 
 * Handles exporting of report data in various formats
 */
class TRWC_Report_Export {
    /**
     * Singleton instance
     *
     * @var TRWC_Report_Export
     */
    private static $instance = null;

    /**
     * Allowed export formats
     *
     * @var array
     */
    private $allowed_formats = array( 'csv', 'json', 'xml' );

    /**
     * Singleton pattern
     *
     * @return TRWC_Report_Export
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'admin_init', array( $this, 'maybe_export_report' ) );
    }

    /**
     * Check if an export is requested and handle it
     */
    public function maybe_export_report() {
        // Check if export is requested
        if ( 
            ! isset( $_GET['page'] ) || 
            $_GET['page'] !== 'wc-totals-report' || 
            ! isset( $_GET['export'] ) 
        ) {
            return;
        }

        // Verify nonce and user capabilities
        check_admin_referer( 'trwc_report_export' );
        
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( 
                esc_html__( 'You do not have sufficient permissions to export reports.', 'totals-report-for-woocommerce' ), 
                esc_html__( 'Export Error', 'totals-report-for-woocommerce' ), 
                403 
            );
        }

        // Sanitize export format
        $format = sanitize_key( $_GET['export'] );
        
        // Validate format
        if ( ! in_array( $format, $this->allowed_formats, true ) ) {
            wp_die( 
                esc_html__( 'Invalid export format.', 'totals-report-for-woocommerce' ), 
                esc_html__( 'Export Error', 'totals-report-for-woocommerce' ), 
                400 
            );
        }

        // Get report data
        $report = TRWC_Report::get_instance()->generate_report();

        // Export based on format
        switch ( $format ) {
            case 'csv':
                $this->export_csv( $report );
                break;
            case 'json':
                $this->export_json( $report );
                break;
            case 'xml':
                $this->export_xml( $report );
                break;
        }
        exit;
    }

    /**
     * Export report as CSV
     *
     * @param array $report Report data.
     */
    private function export_csv( $report ) {
        // Set headers for CSV download
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=totals-report-' . gmdate( 'Y-m-d' ) . '.csv' );
        
        // Create a file pointer
        $output = fopen( 'php://output', 'w' );
        
        if ( ! $output ) {
            return;
        }

        // Output CSV headers
        $headers = array(
            'Section',
            'Metric',
            'Value',
        );
        fputcsv( $output, $headers );

        // Output totals
        fputcsv( $output, array( 'Totals', 'Total Products', $report['totals']['total_products'] ) );
        fputcsv( $output, array( 'Totals', 'In Stock Products', $report['totals']['available_products'] ) );
        fputcsv( $output, array( 'Totals', 'Out of Stock Products', $report['totals']['out_of_stock_products'] ) );
        fputcsv( $output, array( 'Totals', 'Low Stock Products', $report['totals']['low_stock_products'] ) );
        fputcsv( $output, array( 'Totals', 'On Backorder Products', $report['totals']['backorder_products'] ) );
        fputcsv( $output, array( 'Totals', 'Total Variations', $report['totals']['total_variations'] ) );
        fputcsv( $output, array( 'Totals', 'Total Stock Quantity', $report['totals']['total_stock_quantity'] ) );

        // Output product types
        foreach ( $report['product_types'] as $type => $count ) {
            fputcsv( $output, array( 'Product Types', ucfirst( $type ), $count ) );
        }

        // Output categories with detailed data
        foreach ( $report['categories'] as $category => $data ) {
            fputcsv( $output, array( 'Product Categories', $category . ' - Total', $data['total'] ) );
            fputcsv( $output, array( 'Product Categories', $category . ' - In Stock', $data['available'] ) );
            fputcsv( $output, array( 'Product Categories', $category . ' - Out of Stock', $data['out_of_stock'] ) );
            fputcsv( $output, array( 'Product Categories', $category . ' - Low Stock', $data['low_stock'] ) );
            fputcsv( $output, array( 'Product Categories', $category . ' - On Backorder', $data['backorder'] ) );
            fputcsv( $output, array( 'Product Categories', $category . ' - Variable', $data['variable'] ) );
        }

        // No need to close the file pointer here as PHP will automatically close it at the end of the script
        flush();
    }

    /**
     * Export report as JSON
     *
     * @param array $report Report data.
     */
    private function export_json( $report ) {
        // Set headers for JSON download
        header( 'Content-Type: application/json; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=totals-report-' . gmdate( 'Y-m-d' ) . '.json' );
        
        // Output formatted JSON
        echo wp_json_encode( $report, JSON_PRETTY_PRINT );
    }

    /**
     * Export report as XML
     *
     * @param array $report Report data.
     */
    private function export_xml( $report ) {
        // Set headers for XML download
        header( 'Content-Type: application/xml; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=totals-report-' . gmdate( 'Y-m-d' ) . '.xml' );
        
        // Create XML document
        $xml = new SimpleXMLElement( '<?xml version="1.0" encoding="UTF-8"?><totals-report></totals-report>' );
        
        // Convert report to XML
        $this->array_to_xml( $report, $xml );
        
        // Output XML with proper escaping
        echo wp_kses_post( $xml->asXML() );
    }

    /**
     * Recursively convert array to XML
     *
     * @param array            $data Data to convert.
     * @param SimpleXMLElement $xml  XML element to add to.
     */
    private function array_to_xml( $data, &$xml ) {
        foreach ( $data as $key => $value ) {
            if ( is_array( $value ) ) {
                $new_xml = $xml->addChild( str_replace( '_', '-', $key ) );
                $this->array_to_xml( $value, $new_xml );
            } else {
                $xml->addChild( str_replace( '_', '-', $key ), htmlspecialchars( (string) $value, ENT_QUOTES, 'UTF-8' ) );
            }
        }
    }
}