<?php
/**
 * Admin functionality for Totals Report for WooCommerce
 *
 * @package TotalsReportForWooCommerce
 */

// Prevent direct file access.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * TRWC_Report_Admin Class
 * 
 * Handles admin reporting UI and functionality
 */
class TRWC_Report_Admin {
    /**
     * Singleton instance
     *
     * @var TRWC_Report_Admin
     */
    private static $instance = null;

    /**
     * Reporting instance
     *
     * @var TRWC_Report
     */
    private $report;

    /**
     * Export instance
     *
     * @var TRWC_Report_Export
     */
    private $export;

    /**
     * Default items per page
     *
     * @var int
     */
    private $default_per_page = 10;

    /**
     * Singleton pattern
     *
     * @return TRWC_Report_Admin
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Ensure required classes are loaded
        if ( ! class_exists( 'TRWC_Report' ) || ! class_exists( 'TRWC_Report_Export' ) ) {
            add_action( 'admin_notices', array( $this, 'show_class_load_error' ) );
            return;
        }

        // Initialize instances
        $this->report = TRWC_Report::get_instance();
        $this->export = TRWC_Report_Export::get_instance();

        // Admin hooks
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
    }

    /**
     * Add submenu page under WooCommerce
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'Totals Report', 'totals-report-for-woocommerce' ),
            __( 'Totals Report', 'totals-report-for-woocommerce' ),
            'manage_woocommerce',
            'wc-totals-report',
            array( $this, 'render_report_page' )
        );
    }

    /**
     * Enqueue admin assets
     *
     * @param string $hook Current page hook.
     */
    public function enqueue_admin_assets( $hook ) {
        // Only enqueue on our plugin page
        if ( 'woocommerce_page_wc-totals-report' !== $hook ) {
            return;
        }

        wp_enqueue_style(
            'trwc-report-admin',
            TRWC_PLUGIN_URL . 'assets/css/trwctotals-admin.css',
            array(),
            TRWC_VERSION
        );

        wp_enqueue_script(
            'trwc-report-admin-js',
            TRWC_PLUGIN_URL . 'assets/js/trwctotals-admin.js',
            array( 'jquery' ),
            TRWC_VERSION,
            true
        );

        wp_localize_script(
            'trwc-report-admin-js',
            'trwcAdminData',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'trwc_admin_nonce' ),
            )
        );
    }

    /**
     * Render the report page
     */
    public function render_report_page() {
        // Verify nonce for pagination request
        $nonce_verified = true;
        if (isset($_GET['trwc_pagination_nonce'])) {
            $nonce_verified = wp_verify_nonce(sanitize_key($_GET['trwc_pagination_nonce']), 'trwc_pagination_nonce');
        }

        if (!$nonce_verified) {
            wp_die(esc_html__('Security check failed. Please try again.', 'totals-report-for-woocommerce'));
        }

        // Fetch report data
        $report_data = $this->report->generate_report();

        // Handle pagination for categories safely
        $items_per_page = isset($_GET['per_page']) && $nonce_verified ? absint($_GET['per_page']) : $this->default_per_page;
        if ($items_per_page < 1) {
            $items_per_page = $this->default_per_page;
        }
        
        $current_page = isset($_GET['paged']) && $nonce_verified ? absint($_GET['paged']) : 1;
        if ($current_page < 1) {
            $current_page = 1;
        }
        
        $categories = $report_data['categories'];
        $total_items = count($categories);
        $total_pages = ceil($total_items / $items_per_page);
        
        if ($current_page > $total_pages && $total_pages > 0) {
            $current_page = $total_pages;
        }
        
        $offset = ($current_page - 1) * $items_per_page;
        $categories_paged = array_slice($categories, $offset, $items_per_page, true);

        // Create nonced pagination URL base
        $pagination_base_url = add_query_arg([
            'page' => 'wc-totals-report',
            'per_page' => $items_per_page,
            'trwc_pagination_nonce' => wp_create_nonce('trwc_pagination_nonce')
        ]);

        // Start output
        ?>
        <div class="wrap woocommerce-totals-report">
            <h1><?php echo esc_html__('Totals Report for WooCommerce', 'totals-report-for-woocommerce'); ?></h1>

            <div class="report-sections">
                <div class="report-section totals-section">
                    <h2><?php echo esc_html__('Inventory Totals', 'totals-report-for-woocommerce'); ?></h2>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php echo esc_html__('Metric', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('Value', 'totals-report-for-woocommerce'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($report_data['totals'] as $key => $value) : ?>
                                <tr>
                                    <td><?php echo esc_html($this->format_metric_label($key)); ?></td>
                                    <td><?php echo esc_html($value); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="report-section product-types-section">
                    <h2><?php echo esc_html__('Product Types', 'totals-report-for-woocommerce'); ?></h2>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php echo esc_html__('Product Type', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('Count', 'totals-report-for-woocommerce'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($report_data['product_types'] as $type => $count) : ?>
                                <tr>
                                    <td><?php echo esc_html(ucfirst($type)); ?></td>
                                    <td><?php echo esc_html($count); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="report-section categories-section">
                    <h2><?php echo esc_html__('Product Categories', 'totals-report-for-woocommerce'); ?></h2>
                    
                    <div class="tablenav top">
                        <div class="alignleft actions">
                            <form method="get" action="">
                                <input type="hidden" name="page" value="wc-totals-report">
                                <?php wp_nonce_field('trwc_pagination_nonce', 'trwc_pagination_nonce'); ?>
                                <label for="per-page" class="screen-reader-text"><?php echo esc_html__('Number of items per page:', 'totals-report-for-woocommerce'); ?></label>
                                <select name="per_page" id="per-page">
                                    <?php
                                    $pagination_options = array(10, 25, 50, 100);
                                    foreach ($pagination_options as $option) {
                                        echo '<option value="' . esc_attr($option) . '" ' . selected($items_per_page, $option, false) . '>' . esc_html($option) . '</option>';
                                    }
                                    ?>
                                </select>
                                <input type="submit" class="button" value="<?php echo esc_attr__('Apply', 'totals-report-for-woocommerce'); ?>">
                            </form>
                        </div>
                        
                        <div class="tablenav-pages">
                            <span class="displaying-num">
                                <?php
                                echo sprintf(
                                    /* translators: %s: number of items */
                                    esc_html__('%s items', 'totals-report-for-woocommerce'),
                                    esc_html($total_items)
                                );
                                ?>
                            </span>
                            <?php
                            if ($total_pages > 1) {
                                $page_links = paginate_links(array(
                                    'base'      => add_query_arg('paged', '%#%', $pagination_base_url),
                                    'format'    => '',
                                    'prev_text' => __('&laquo;', 'totals-report-for-woocommerce'),
                                    'next_text' => __('&raquo;', 'totals-report-for-woocommerce'),
                                    'total'     => $total_pages,
                                    'current'   => $current_page
                                ));
                                
                                echo '<span class="pagination-links">' . wp_kses_post($page_links) . '</span>';
                            }
                            ?>
                        </div>
                    </div>
                    
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php echo esc_html__('Category', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('Product Count', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('In Stock', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('Out of Stock', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('Low Stock', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('On Backorder', 'totals-report-for-woocommerce'); ?></th>
                                <th><?php echo esc_html__('Variable Products', 'totals-report-for-woocommerce'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($categories_paged as $category => $data) : ?>
                                <tr>
                                    <td><?php echo esc_html($category); ?></td>
                                    <td><?php echo esc_html($data['total']); ?></td>
                                    <td><?php echo esc_html($data['available']); ?></td>
                                    <td><?php echo esc_html($data['out_of_stock']); ?></td>
                                    <td><?php echo esc_html($data['low_stock']); ?></td>
                                    <td><?php echo esc_html($data['backorder']); ?></td>
                                    <td><?php echo esc_html($data['variable']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <?php
                            if ($total_pages > 1) {
                                echo '<span class="pagination-links">' . wp_kses_post($page_links) . '</span>';
                            }
                            ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="export-buttons">
                <h2><?php echo esc_html__('Export Report', 'totals-report-for-woocommerce'); ?></h2>
                <?php $this->render_export_buttons(); ?>
            </div>
        </div>
        <?php
    }

    /**
     * Format metric label for display
     *
     * @param string $key Metric key.
     * @return string Formatted label
     */
    private function format_metric_label($key) {
        $label_map = array(
            'total_products' => __('Total Products', 'totals-report-for-woocommerce'),
            'available_products' => __('In Stock Products', 'totals-report-for-woocommerce'),
            'out_of_stock_products' => __('Out of Stock Products', 'totals-report-for-woocommerce'),
            'low_stock_products' => __('Low Stock Products', 'totals-report-for-woocommerce'),
            'backorder_products' => __('On Backorder Products', 'totals-report-for-woocommerce'),
            'total_variations' => __('Total Variations', 'totals-report-for-woocommerce'),
            'total_stock_quantity' => __('Total Stock Quantity', 'totals-report-for-woocommerce'),
        );
        
        if (isset($label_map[$key])) {
            return $label_map[$key];
        }
        
        return ucwords(str_replace('_', ' ', $key));
    }

    /**
     * Render export buttons
     */
    private function render_export_buttons() {
        // Export formats
        $export_formats = array(
            'csv'  => __('Export as CSV', 'totals-report-for-woocommerce'),
            'json' => __('Export as JSON', 'totals-report-for-woocommerce'),
            'xml'  => __('Export as XML', 'totals-report-for-woocommerce'),
        );

        // Create export links
        foreach ($export_formats as $format => $label) {
            $export_url = add_query_arg(array(
                'page'   => 'wc-totals-report',
                'export' => $format,
            ), admin_url('admin.php'));
            $export_url = wp_nonce_url($export_url, 'trwc_report_export');
            ?>
            <a href="<?php echo esc_url($export_url); ?>" class="button button-primary">
                <?php echo esc_html($label); ?>
            </a>
            <?php
        }
    }

    /**
     * Show error notice if class is not loaded
     */
    public function show_class_load_error() {
        ?>
        <div class="notice notice-error">
            <p><?php 
                echo esc_html__( 
                    'Totals Report for WooCommerce: Unable to load required classes. Please check your plugin installation.', 
                    'totals-report-for-woocommerce' 
                ); 
            ?></p>
        </div>
        <?php
    }
}