<?php
/**
 * Plugin Name: Total Views
 * Plugin URI:  https://dizetech.in/shop/total-views-plugin/
 * Description: Counts total page views and displays via shortcode with live preview and customizable styles. Use the [total_views] shortcode to display total page views with customizable label and styles anywhere on your site.
 * Version:     1.0
 * Author:      DIZE TECH IT SOLUTIONS
 * Author URI:  https://dizetech.in
 * License:     GPL2
 * Requires at least: 5.0
 * Requires PHP: 7.2
 * Text Domain: total-views
 */

if (!defined('ABSPATH')) exit;

// Increment total views
function total_views_increment_counter() {
    $total_views = get_option('total_views_total', 0);
    $total_views++;
    update_option('total_views_total', $total_views);
}
add_action('wp_head', 'total_views_increment_counter');

// Shortcode
function total_views_display_shortcode() {
    $total_views = get_option('total_views_total', 0);
    $label = get_option('total_views_label', 'Total Page Views');
    $bold = get_option('total_views_bold', '') ? 'font-weight:bold;' : '';
    $italic = get_option('total_views_italic', '') ? 'font-style:italic;' : '';
    $underline = get_option('total_views_underline', '') ? 'text-decoration:underline;' : '';
    $color = get_option('total_views_color', '#000000');

    $label_style = "color:$color;$bold$italic$underline";

    return "<div class='total-views'>
        <span class='total-views-label' style='$label_style'>{$label}:</span>
        <span class='total-views-count' style='$label_style'>" . number_format($total_views) . "</span>
    </div>";
}
add_shortcode('total_views', 'total_views_display_shortcode');

// Admin menu
function total_views_settings_menu() {
    add_options_page(
        'Total Views Settings',
        'Total Views',
        'manage_options',
        'total-views-settings',
        'total_views_settings_page'
    );
}
add_action('admin_menu', 'total_views_settings_menu');

// Register settings
function total_views_register_settings() {
    register_setting('total_views_settings_group', 'total_views_total', array(
        'type' => 'integer',
        'sanitize_callback' => 'absint',
        'default' => 0,
    ));
        register_setting('total_views_settings_group','total_views_label',array('type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'Total Page Views'));
        register_setting('total_views_settings_group','total_views_bold',array('type'=>'boolean','sanitize_callback'=>'absint','default'=>0));
        register_setting('total_views_settings_group','total_views_italic',array('type'=>'boolean','sanitize_callback'=>'absint','default'=>0));
        register_setting('total_views_settings_group','total_views_underline',array('type'=>'boolean','sanitize_callback'=>'absint','default'=>0));
        register_setting('total_views_settings_group','total_views_color',array('type'=>'string','sanitize_callback'=>'sanitize_hex_color','default'=>'#000000'));
}
add_action('admin_init', 'total_views_register_settings');

// Enqueue admin CSS/JS
function total_views_admin_assets($hook) {
    if ($hook !== 'settings_page_total-views-settings') return;
    wp_enqueue_style('total-views-admin', plugin_dir_url(__FILE__).'assets/css/admin-style.css', array(), '1.2'); 
    wp_enqueue_script('total-views-admin', plugin_dir_url(__FILE__).'assets/js/admin-script.js', array('jquery'), '1.0', true);
}
add_action('admin_enqueue_scripts', 'total_views_admin_assets');

// Settings page
function total_views_settings_page() {
    ?>
    <div class="wrap">
        <h1>Total Views Settings</h1>
        <div class="total-views-settings-wrapper">

            <!-- Left column: settings -->
            <div class="total-views-settings-left">
                <form method="post" action="options.php">
                    <?php settings_fields('total_views_settings_group'); ?>
                    <?php do_settings_sections('total_views_settings_group'); ?>

                    <table class="form-table">
                        <tr valign="top">
                            <th>Total Page Views</th>
                            <td>
                                <input type="number" name="total_views_total" value="<?php echo esc_attr(get_option('total_views_total', 0)); ?>" min="0" />
                                <p class="description">You can manually update the total page views here.</p>
                            </td>
                        </tr>
                        <tr>
                            <th>Label Name</th>
                            <td><input type="text" name="total_views_label" value="<?php echo esc_attr(get_option('total_views_label', 'Total Page Views')); ?>"></td>
                        </tr>
                        <tr>
                            <th>Bold</th>
                            <td><input type="checkbox" name="total_views_bold" value="1" <?php checked(1, get_option('total_views_bold', 0)); ?>></td>
                        </tr>
                        <tr>
                            <th>Italic</th>
                            <td><input type="checkbox" name="total_views_italic" value="1" <?php checked(1, get_option('total_views_italic', 0)); ?>></td>
                        </tr>
                        <tr>
                            <th>Underline</th>
                            <td><input type="checkbox" name="total_views_underline" value="1" <?php checked(1, get_option('total_views_underline', 0)); ?>></td>
                        </tr>
                        <tr>
                            <th>Text Color</th>
                            <td><input type="color" name="total_views_color" value="<?php echo esc_attr(get_option('total_views_color', '#000000')); ?>"></td>
                        </tr>
                    </table>

                    <?php submit_button(); ?>
                </form>
            </div>

            <!-- Right column: preview + note + reset + tutorial -->
            <div class="total-views-settings-right">
                    <p><strong>Preview (Shortcode Output):</strong></p>
                    <p>
                        <?php echo do_shortcode('[total_views]'); ?>
                    </p>
                <hr>
                <p>If you want to customize your own label and count text styles, use the <code>total-views-label</code> and <code>total-views-count</code> classes in your CSS and add <code>!important</code> if needed.</p>
                <p>
                    <a href="https://www.youtube.com/watch?v=VIDEO_ID" target="_blank" class="button button-primary">
                        &#128279; Check tutorial
                    </a>
                </p>
                <hr>
                <p>
                    <strong>Note:</strong> To reset the counter, set the "Total Page Views" value to <code>0</code> in the left settings panel and click "Save Changes".
                </p>
            </div>
            <div style="background: #ffffff; padding: 20px; border: 1px solid #ddd; margin-top: 40px;">
                    <h3>🚀 Thank you for using our plugins.</h3>
                    <p>
                        This plugin was created by 
                        <a href="https://dizetech.in" target="_blank"><strong>DIZETECH IT SOLUTIONS</strong></a>. 
                        Any help or want to create a custom plugin for your website? Contact us at 
                        <a href="mailto:info@dizetech.in" target="_blank"><strong>info@dizetech.in</strong></a>. 
                        Looking for more helpful tools? 
                        <a href="<?php echo esc_url(admin_url('plugin-install.php?s=dizetech&tab=search&type=author')); ?>" target="_blank">
                            <strong>Click here to view all our plugins</strong>
                        </a> directly from your WordPress dashboard.
                    </p>
            </div>
        </div>
    </div>
    <?php
}
