<?php
/**
 * Plugin Name: Tohidul Certificate Verification System
 * Plugin URI: https://tohidul.com/en/certificate-verification-system
 * Description: A complete certificate verification system with import/export and backup features
 * Version: 2.1.0
 * Author: Md Tohidul Islam Bhuiyan
 * Author URI: https://www.tohidul.com/en/
 * License: GPL v2 or later
 * Text Domain: tohidul-certificate-verification-system
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * Domain Path: /languages
 */

declare(strict_types=1);

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! defined( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL' ) ) {
    define( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}

if ( ! defined( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_PATH' ) ) {
    define( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );
}

if ( ! defined( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION' ) ) {
    define( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION', '2.1.0' );
}

if ( ! defined( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_TABLE_NAME' ) ) {
    define( 'TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_TABLE_NAME', 'tcvs_certificates' );
}

class Tohidul_Certificate_Verification_System {
    
    private $database;
    private $security;
    private $admin;
    private $export_import;
    private $backup;
    private $shortcodes;
    private $settings;
    
    public function __construct() {
        $this->load_dependencies();
        $this->init_components();
        $this->init_hooks();
    }
    
    private function load_dependencies(): void {
        $files = array(
            'class-database.php',
            'class-security.php', 
            'class-admin.php',
            'class-export-import.php',
            'class-backup.php',
            'class-shortcodes.php',
            'class-settings.php'
        );
        
        foreach ( $files as $file ) {
            $file_path = TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_PATH . 'includes/' . $file;
            if ( file_exists( $file_path ) ) {
                require_once $file_path;
            }
        }
    }
    
    private function init_components(): void {
        $this->database = new Tohidul_Certificate_Verification_System_Database();
        $this->security = new Tohidul_Certificate_Verification_System_Security();
        $this->admin = new Tohidul_Certificate_Verification_System_Admin( $this->database );
        $this->export_import = new Tohidul_Certificate_Verification_System_Export_Import( $this->database );
        $this->backup = new Tohidul_Certificate_Verification_System_Backup( $this->database );
        $this->shortcodes = new Tohidul_Certificate_Verification_System_Shortcodes( $this->database );
        $this->settings = new Tohidul_Certificate_Verification_System_Settings( $this->database );
    }
    
    private function init_hooks(): void {
        register_activation_hook( __FILE__, array( $this, 'activate' ) );
        register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
        
        add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
        
        // AJAX Hooks - Renamed to tcvs_ prefix
        add_action( 'wp_ajax_tcvs_search', array( $this, 'handle_certificate_search' ) );
        add_action( 'wp_ajax_nopriv_tcvs_search', array( $this, 'handle_certificate_search' ) );
        
        add_action( 'admin_post_tohidul_certificate_verification_system_form_action', array( $this, 'handle_form_action' ) );
    }
    
    public function enqueue_scripts(): void {
        if ( is_admin() ) {
            return;
        }
        
        wp_enqueue_style( 
            'tcvs-frontend-style', 
            TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL . 'assets/css/style.css', 
            array(), 
            TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION 
        );
        
        wp_enqueue_script( 
            'tcvs-frontend-script', 
            TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL . 'assets/js/script.js', 
            array( 'jquery' ), 
            TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION, 
            array( 'in_footer' => true ) 
        );
        
        // Renamed localized object to tcvs_ajax
        wp_localize_script( 'tcvs-frontend-script', 'tcvs_ajax', array(
            'ajax_url'      => admin_url( 'admin-ajax.php' ),
            'nonce'         => wp_create_nonce( 'tcvs_ajax_nonce' ),
            'error_message' => __( 'An error occurred. Please try again.', 'tohidul-certificate-verification-system' ),
            'loading_text'  => __( 'Loading...', 'tohidul-certificate-verification-system' ),
            'verify_text'   => __( 'Verify Certificate', 'tohidul-certificate-verification-system' ),
        ) );
    }
    
    public function enqueue_admin_scripts( string $hook ): void {
        $plugin_pages = array(
            'toplevel_page_certificate-verification',
            'certificate-verification_page_certificate-add',
            'certificate-verification_page_certificate-import-export',
            'certificate-verification_page_certificate-backup',
            'certificate-verification_page_certificate-settings',
        );
        
        if ( in_array( $hook, $plugin_pages, true ) || false !== strpos( $hook, 'certificate-verification' ) ) {
            wp_enqueue_style( 
                'tcvs-admin-style', 
                TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL . 'assets/css/admin-style.css', 
                array( 'wp-jquery-ui-dialog' ), 
                TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION 
            );
            
            wp_enqueue_script( 
                'tcvs-admin-script', 
                TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_URL . 'assets/js/admin-script.js', 
                array( 'jquery', 'jquery-ui-dialog', 'wp-util' ), 
                TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION, 
                array( 'in_footer' => true ) 
            );
            
            wp_enqueue_media();
            
            $admin_nonce = wp_create_nonce( 'tcvs_admin_nonce' );
            
            // Renamed localized object to tcvs_admin
            wp_localize_script( 'tcvs-admin-script', 'tcvs_admin', array(
                'ajax_url'              => admin_url( 'admin-ajax.php' ),
                'nonce'                 => $admin_nonce,
                'delete_nonce'          => wp_create_nonce( 'tcvs_delete_nonce' ),
                'bulk_delete_nonce'     => wp_create_nonce( 'tcvs_bulk_delete_nonce' ),
                'confirm_delete'        => __( 'Are you sure you want to delete this certificate?', 'tohidul-certificate-verification-system' ),
                'confirm_bulk_delete'   => __( 'Are you sure you want to delete selected certificates?', 'tohidul-certificate-verification-system' ),
                'upload_title'          => __( 'Select or Upload Image', 'tohidul-certificate-verification-system' ),
                'upload_button'         => __( 'Use this image', 'tohidul-certificate-verification-system' ),
                'admin_url'             => admin_url(),
                'export_nonce'          => wp_create_nonce( 'tcvs_export_nonce' ),
                'import_nonce'          => wp_create_nonce( 'tcvs_import_nonce' ),
                'backup_nonce'          => wp_create_nonce( 'tcvs_backup_nonce' ),
                'media_uploader'        => array(
                    'title'  => __( 'Select or Upload Image', 'tohidul-certificate-verification-system' ),
                    'button' => __( 'Use this image', 'tohidul-certificate-verification-system' ),
                )
            ) );
            
            $this->add_admin_inline_styles();
        }
    }
    
    private function add_admin_inline_styles(): void {
        $css = '
            .tcvs-admin-card { margin-top: 20px; }
            .tcvs-image-preview img { max-width: 150px; height: auto; border: 1px solid #ddd; padding: 3px; background: #fff; }
            .tcvs-admin-table { margin-top: 20px; }
        ';
        wp_add_inline_style( 'tcvs-admin-style', $css );
    }
    
    public function activate(): void {
        if ( version_compare( get_bloginfo( 'version' ), '5.0', '<' ) ) {
            wp_die( esc_html__( 'Requires WordPress 5.0 or higher.', 'tohidul-certificate-verification-system' ) );
        }
        
        if ( class_exists( 'Tohidul_Certificate_Verification_System_Database' ) ) {
            $database = new Tohidul_Certificate_Verification_System_Database();
            $database->create_tables();
        }
        
        if ( ! get_option( 'tohidul_certificate_verification_system_db_version' ) ) {
            update_option( 'tohidul_certificate_verification_system_db_version', TOHIDUL_CERTIFICATE_VERIFICATION_SYSTEM_PLUGIN_VERSION );
        }
        
        flush_rewrite_rules();
    }
    
    public function deactivate(): void {
        flush_rewrite_rules();
    }
    
    public function get_database() {
        return $this->database;
    }
    
    public function load_textdomain(): void {
        load_textdomain( 'tohidul-certificate-verification-system', WP_LANG_DIR . '/plugins/tohidul-certificate-verification-system-' . get_locale() . '.mo' );
    }
    
    public function handle_form_action(): void {
        if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'tcvs_action' ) ) {
            wp_safe_redirect( wp_get_referer() );
            exit;
        }
    }
    
    public function handle_certificate_search(): void {
        check_ajax_referer( 'tcvs_ajax_nonce', 'nonce' );
        $certificate_id = isset( $_POST['certificate_id'] ) ? intval( $_POST['certificate_id'] ) : 0;
        $result = $this->database->get_certificate( $certificate_id );
        
        if ( $result ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( array( 'message' => __( 'Certificate not found.', 'tohidul-certificate-verification-system' ) ) );
        }
    }
}

function tohidul_certificate_verification_system_initialize() {
    static $instance = null;
    if ( null === $instance ) {
        $instance = new Tohidul_Certificate_Verification_System();
    }
    return $instance;
}

add_action( 'plugins_loaded', 'tohidul_certificate_verification_system_initialize', 5 );

register_activation_hook( __FILE__, 'tohidul_certificate_verification_system_plugin_activation' );

function tohidul_certificate_verification_system_plugin_activation() {
    require_once plugin_dir_path( __FILE__ ) . 'includes/class-database.php';
    $database = new Tohidul_Certificate_Verification_System_Database();
    $database->create_tables();
}

add_action('admin_init', function() {
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (isset($_GET['page']) && $_GET['page'] == 'certificate-verification') {
        $database = new Tohidul_Certificate_Verification_System_Database();
        $database->create_tables();
    }
});