<?php
/**
 * Handles the frontend shortcodes for certificate verification.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Tohidul_Certificate_Verification_System_Shortcodes {
    
    private $database;
    
    public function __construct( Tohidul_Certificate_Verification_System_Database $database ) {
        $this->database = $database;
        
        // Changed Shortcode Name to be unique (tcvs_)
        add_shortcode( 'tcvs_verification', array( $this, 'verification_form_shortcode' ) );
        
        // Changed AJAX Handlers to tcvs_ prefix
        add_action( 'wp_ajax_tcvs_verify_certificate', array( $this, 'ajax_verify_certificate' ) );
        add_action( 'wp_ajax_nopriv_tcvs_verify_certificate', array( $this, 'ajax_verify_certificate' ) );
    }

    public function verification_form_shortcode( $atts ) { 
        wp_enqueue_style( 'tcvs-frontend-style' );
        wp_enqueue_script( 'tcvs-frontend-script' );
        
        ob_start(); 
        ?>
        <div class="tcvs-verification-form" style="max-width:600px; margin:40px auto; padding:30px; background:#fff; border-radius:10px; border:1px solid #eee; box-shadow: 0 5px 15px rgba(0,0,0,0.05);">
            <h3 style="text-align:center; color:#1a3c5e; margin-bottom:20px;">Verify Your Certificate</h3>
            <form id="tcvs-verification-form">
                <input type="text" id="tcvs_certificate_number" placeholder="Enter Registration / Certificate No." required 
                       style="width:100%; padding:15px; border:2px solid #e1e5e9; border-radius:6px; font-size:16px; margin-bottom:15px; box-sizing: border-box;">
                <button type="submit" class="tcvs-submit-button" 
                        style="width:100%; padding:15px; background:#1a3c5e; color:#fff; border:none; border-radius:6px; font-weight:600; font-size:16px; cursor:pointer; transition: background 0.3s;">
                    Verify Now
                </button>
            </form>
            <div id="tcvs-result" style="display:none; margin-top:30px;"></div>
            <div class="tcvs-messages" style="margin-top:10px;"></div>
        </div>

        <?php 
        return ob_get_clean();
    }

    public function ajax_verify_certificate(): void {
        check_ajax_referer( 'tcvs_ajax_nonce', 'nonce' );

        $certificate_number = sanitize_text_field( wp_unslash( $_POST['certificate_number'] ?? '' ) );
        
        if ( empty( $certificate_number ) ) {
            wp_send_json_error( 'Please enter a certificate number.' );
        }

        $certificate = $this->database->get_certificate_by_number( $certificate_number );
        
        if ( ! $certificate ) {
            wp_send_json_error( 'Certificate not found. Please check the number and try again.' );
        }
        
        $html = $this->display_certificate( $certificate );
        
        wp_send_json_success( array( 'html' => $html ) );
    }

    public function display_certificate( $certificate ): string {
        $settings = $this->database->get_institute_settings();
        
        $inst_name    = !empty($settings->institute_name) ? $settings->institute_name : 'Your Institute Name';
        $inst_logo    = $settings->institute_logo ?? '';
        $inst_addr    = $settings->institute_address ?? '';
        $inst_phone   = $settings->institute_phone ?? ''; 
        $inst_email   = $settings->institute_email ?? '';
        $inst_sig     = $settings->signature_image ?? '';
        $verified_img = $settings->verified_by_image ?? ''; 
        
        $date_source = '';
        if ( ! empty( $certificate->issue_date ) && $certificate->issue_date !== '0000-00-00' ) {
            $date_source = $certificate->issue_date;
        } elseif ( ! empty( $certificate->created_at ) ) {
            $date_source = $certificate->created_at;
        } else {
            $date_source = current_time( 'Y-m-d' );
        }
        
        $timestamp = strtotime( $date_source );
        if ( $timestamp ) {
            $formatted_date = date_i18n( get_option( 'date_format' ), $timestamp );
        } else {
            $formatted_date = date_i18n( get_option( 'date_format' ) );
        }

        ob_start();
        ?>
        <div class="tcvs-result-wrapper">
            <div class="tcvs-screen-view">
                <div class="tcvs-verified-header">
                    <span class="dashicons dashicons-yes-alt" style="vertical-align: text-bottom;"></span> Verified Certificate
                </div>
                
                <div class="tcvs-student-profile">
                    <div class="tcvs-profile-image">
                        <?php if(!empty($certificate->student_image)): ?>
                            <img src="<?php echo esc_url($certificate->student_image); ?>" alt="Student">
                        <?php else: ?>
                            <div style="line-height:120px; color:#999; background:#f0f0f0;">No Photo</div>
                        <?php endif; ?>
                    </div>
                    <h2 class="tcvs-student-name"><?php echo esc_html($certificate->student_name); ?></h2>
                    <span class="tcvs-reg-badge">Reg: <?php echo esc_html($certificate->certificate_number); ?></span>
                </div>
                
                <div class="tcvs-details-list">
                    <div class="tcvs-detail-row">
                        <span class="tcvs-label">Father's Name</span>
                        <span class="tcvs-value"><?php echo esc_html($certificate->father_name); ?></span>
                    </div>
                    <div class="tcvs-detail-row">
                        <span class="tcvs-label">Course Name</span>
                        <span class="tcvs-value"><?php echo esc_html($certificate->course_name); ?></span>
                    </div>
                    <div class="tcvs-detail-row">
                        <span class="tcvs-label">Batch No</span>
                        <span class="tcvs-value"><?php echo esc_html($certificate->batch_number); ?></span>
                    </div>
                    <div class="tcvs-detail-row">
                        <span class="tcvs-label">Issue Date</span>
                        <span class="tcvs-value"><?php echo esc_html($formatted_date); ?></span>
                    </div>
                    <div class="tcvs-detail-row">
                        <span class="tcvs-label">Institute</span>
                        <span class="tcvs-value"><?php echo esc_html($inst_name); ?></span>
                    </div>
                </div>
                
                <div class="tcvs-actions">
                    <button type="button" class="tcvs-btn-print">
                        <span class="dashicons dashicons-printer"></span> Print Original Certificate
                    </button>
                    <button type="button" class="tcvs-btn-back" onclick="location.reload()">Check Another</button>
                </div>
            </div>

            <div class="tcvs-print-view" style="display:none;">
                <div class="cert-page" style="margin: 0 auto;">
                    <div class="cert-border">
                        <div class="cert-content">
                            <div class="cert-header">
                                <?php if($inst_logo): ?>
                                    <img src="<?php echo esc_url($inst_logo); ?>" class="cert-logo" alt="Logo">
                                <?php endif; ?>
                                <h1 class="cert-inst-name"><?php echo esc_html($inst_name); ?></h1>
                                <p class="cert-inst-addr"><?php echo esc_html($inst_addr); ?></p>
                                
                                <p class="cert-inst-contact" style="font-size: 10pt; color: #555; margin-top: 5px; font-family: 'Roboto', sans-serif;">
                                    <?php if($inst_phone): ?>
                                        <span>Phone: <?php echo esc_html($inst_phone); ?></span>
                                    <?php endif; ?>
                                    <?php if($inst_phone && $inst_email): ?> | <?php endif; ?>
                                    <?php if($inst_email): ?>
                                        <span>Email: <?php echo esc_html($inst_email); ?></span>
                                    <?php endif; ?>
                                </p>
                            </div>
                            
                            <div class="cert-title-block">
                                <h2 class="cert-main-title">Certificate of Completion</h2>
                                <p class="cert-subtitle">This is to certify that</p>
                            </div>
                            
                            <?php if(!empty($certificate->student_image)): ?>
                            <div class="cert-print-photo">
                                <img src="<?php echo esc_url($certificate->student_image); ?>" alt="Student">
                            </div>
                            <?php endif; ?>
                            
                            <div class="cert-student-info">
                                <h2 class="cert-student-name-print"><?php echo esc_html($certificate->student_name); ?></h2>
                                <p class="cert-info-text">
                                    Son/Daughter of <strong><?php echo esc_html($certificate->father_name); ?></strong><br>
                                    has successfully completed the course
                                </p>
                                <h3 class="cert-course-name"><?php echo esc_html($certificate->course_name); ?></h3>
                            </div>
                            
                            <div class="cert-details-box">
                                <div class="cert-detail-item">
                                    <span>Registration No.</span>
                                    <strong><?php echo esc_html($certificate->certificate_number); ?></strong>
                                </div>
                                <div class="cert-detail-item">
                                    <span>Batch No.</span>
                                    <strong><?php echo esc_html($certificate->batch_number); ?></strong>
                                </div>
                                <div class="cert-detail-item">
                                    <span>Issue Date</span>
                                    <strong><?php echo esc_html($formatted_date); ?></strong>
                                </div>
                            </div>
                            
                            <div class="cert-footer">
                                <div class="cert-sig-area">
                                    <?php if($verified_img): ?>
                                        <img src="<?php echo esc_url($verified_img); ?>" class="cert-sig-img" alt="Verified By">
                                    <?php else: ?>
                                        <div style="height:50px;"></div>
                                    <?php endif; ?>
                                    <div class="cert-sig-line">Verified By</div>
                                </div>
                                
                                <div class="cert-sig-area">
                                    <?php if($inst_sig): ?>
                                        <img src="<?php echo esc_url($inst_sig); ?>" class="cert-sig-img" alt="Signature">
                                    <?php else: ?>
                                        <div style="height:50px;"></div>
                                    <?php endif; ?>
                                    <div class="cert-sig-line">Principal Signature</div> 
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
}